﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
########################################################################################################################
# Description:
# This script outputs the majority of useful information to be found in the Zerto REST APIs into CSVs and PowerShell tables for your use
################################################ 
# Requirements:
# - PowerShell 6.x
# - ZVM 6.0+ 
# - Zerto ServerName, Username and password with permission to access the API of the ZVM
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted"
# - A valid export directory which the script has permission to write to
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below
################################################
$ZertoServer = "192.168.3.9"
$ExportDir = "C:\ZertoInfov6\"
# Prompting for username and password to authenicate with ZVM
$Credentials = Get-Credential -Credential $null
$ZertoUser = $Credentials.UserName
$ZertoPassword = [System.Net.NetworkCredential]::new('',$Credentials.Password).Password
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Setting export file names
################################################
$Month = Get-Date -Format dd-MM-yy
$Time = Get-Date -Format hh-mm-ss
# Exports direct from API
$VPGExport = $ExportDir + $Month + " " + $Time + " ZertoVPGs" + ".csv"
$ProtectedVMExport = $ExportDir + $Month + " " + $Time + " ZertoVPGVMs" + ".csv"
$VRAVMExport = $ExportDir + $Month + " " + $Time  + " ZertoVRAs" + ".csv"
$TargetSiteExport = $ExportDir + $Month + " " + $Time + " ZertoTargetSites" + ".csv"
$UnprotectedVMExport = $ExportDir + $Month + " " + $Time + " vCenterUnProtectedVMs" + ".csv"
# Exports from target site table building
$ClustersExport = $ExportDir + $Month + " " + $Time + " vCenterClusters" + ".csv"
$HostsExport = $ExportDir + $Month + " " + $Time  + " vCenterHosts" + ".csv"
$DatastoresExport = $ExportDir + $Month + " " + $Time + " vCenterDatastores" + ".csv"
$DatastoreClustersExport = $ExportDir + $Month + " " + $Time + " vCenterDatastoreClusters" + ".csv"
$PortGroupsExport = $ExportDir + $Month + " " + $Time + " vCenterPortGroups" + ".csv"
$FoldersExport = $ExportDir + $Month + " " + $Time + " vCenterFolders" + ".csv"
$ResourcePoolsExport = $ExportDir + $Month + " " + $Time + " vCenterResourcePools" + ".csv"
# Exports from VPG settings table building
$VPGSettingsExport = $ExportDir + $Month + " " + $Time + " Settings-VPGs" + ".csv"
$VMSettingsExport = $ExportDir + $Month + " " + $Time + " Settings-VMs" + ".csv"
$VolumeSettingsExport = $ExportDir + $Month + " " + $Time + " Settings-VMVolumes" + ".csv"
$NICSettingsExport = $ExportDir + $Month + " " + $Time + " Settings-VMNICs" + ".csv"
# Exports from summary of data
$DatastoreVPGUsageExport = $ExportDir + $Month + " " + $Time + " AllDatastoresVPGUsage" + ".csv"
$HostVPGUsageExport = $ExportDir + $Month + " " + $Time + " AllHostsVMUsage" + ".csv"
$SiteTotalsExport = $ExportDir + $Month + " " + $Time + " AllSiteTotals" + ".csv"
################################################
# Creating tables required
################################################
$VPGTable = @()
$VPGVMTable = @()
$VRATable = @()
$UnprotectedVMTable =@()
$ClusterTable = @()
$ESXiHostTable = @()
$DatastoreTable = @()
$DatastoreClusterTable = @()
$PortGroupTable = @()
$FolderTable = @()
$ResourcePoolTable = @()
$VPGSettingsTable = @()
$VMSettingsTable = @()
$VMVolumeSettingsTable = @()
$VMNICSettingsTable = @()
$DatastoreVPGUsageTable = @()
$HostVMUsageTable = @()
$SiteTotalsTable = @()
################################################
# Building ZVR API string and invoking REST API
################################################
$BaseURL = "https://" + $ZertoServer + ":9669/v1/"
$ZertoSessionURL = $BaseURL + "session/add"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($ZertoUser+":"+$ZertoPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$ZertoSessionResponse = Invoke-WebRequest -Method POST -Uri $ZertoSessionURL -Headers $Header -ContentType $Type -SkipCertificateCheck
$ZertoSessionSuccess = $TRUE
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
$ZertoSessionSuccess = $FALSE
}
# Extracting the token from the JSON response
$ZertoSession = $ZertoSessionResponse.headers.get_item("x-zerto-session")
$ZertoSessionHeader = @{"Accept"="application/json"
"x-zerto-session"="$ZertoSession"}
################################################
# IF not authenticated with Zerto no point continuing
################################################
IF ($ZertoSessionSuccess -eq $TRUE)
{
################################################
# Getting VPGs
################################################
$VPGListUrl = $BaseURL+"vpgs"
Try 
{
$VPGList = Invoke-RestMethod -Method GET -Uri $VPGListUrl -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
################################################
# Tidying Up VPG Information
################################################
ForEach ($VPG in $VPGList)
{
# Setting values
$VPGVpgName = $VPG.VpgName
$VPGVpgIdentifier = $VPG.VpgIdentifier
$VPGStatus = $VPG.Status
$VPGSubStatus = $VPG.SubStatus
$VPGActualRPO = $VPG.ActualRPO
$VPGConfiguredRpoSeconds = $VPG.ConfiguredRpoSeconds
$VPGVmsCount = $VPG.VmsCount
$VPGPriority = $VPG.Priority
$VPGProvisionedStorageInMB = $VPG.ProvisionedStorageInMB
$VPGUsedStorageInMB = $VPG.UsedStorageInMB
$VPGThroughputInMB = $VPG.ThroughputInMB
$VPGProgressPercentage = $VPG.ProgressPercentage
$VPGOrganizationName = $VPG.OrganizationName
$VPGSourceSite = $VPG.SourceSite
$VPGProtectedSite = $VPG.ProtectedSite.identifier
$VPGRecoverySite = $VPG.RecoverySite.identifier
$VPGTargetSite = $VPG.TargetSite
$VPGLastTest = $VPG.LastTest
$VPGServiceProfile = $VPG.ServiceProfile
$VPGServiceProfileName = $VPG.ServiceProfileName
$VPGServiceProfileIdentifier = $VPG.ServiceProfileIdentifier
$VPGIOPs = $VPG.IOPs
$VPGBackupEnabled = $VPG.BackupEnabled
$VPGActualHistoryInMinutes = $VPG.HistoryStatusApi.ActualHistoryInMinutes
$VPGConfiguredHistoryInMinutes = $VPG.HistoryStatusApi.ConfiguredHistoryInMinutes
$VPGEarliestCheckpoint = $VPG.HistoryStatusApi.EarliestCheckpoint
$VPGActualFailSafeHistory = $VPG.FailSafeHistory.ActualFailSafeHistory
$VPGConfiguredFailSafeHistory = $VPG.FailSafeHistory.ConfiguredFailSafeHistory
$VPGFailSafeDescription = $VPG.FailSafeHistory.FailSafeDescription
$VPGZorg = $VPG.Zorg.identifier
# Adding row to table
$VPGTableRow = New-Object PSObject
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "VpgName" -Value $VPGVpgName
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "VpgIdentifier" -Value $VPGVpgIdentifier
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "Status" -Value $VPGStatus
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "SubStatus" -Value $VPGSubStatus
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ActualRPO" -Value $VPGActualRPO
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ConfiguredRpoSeconds" -Value $VPGConfiguredRpoSeconds
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "VmsCount" -Value $VPGVmsCount
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "Priority" -Value $VPGPriority
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ProvisionedStorageInMB" -Value $VPGProvisionedStorageInMB
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "UsedStorageInMB" -Value $VPGUsedStorageInMB
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ThroughputInMB" -Value $VPGThroughputInMB
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ProgressPercentage" -Value $VPGProgressPercentage
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "OrganizationName" -Value $VPGOrganizationName
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "SourceSite" -Value $VPGSourceSite
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedSite" -Value $VPGProtectedSite
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "RecoverySite" -Value $VPGRecoverySite
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "TargetSite" -Value $VPGTargetSite
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "LastTest" -Value $VPGLastTest
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ServiceProfile" -Value $VPGServiceProfile
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ServiceProfileName" -Value $VPGServiceProfileName
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ServiceProfileIdentifier" -Value $VPGServiceProfileIdentifier
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "IOPs" -Value $VPGIOPs
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "BackupEnabled" -Value $VPGBackupEnabled
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ActualHistoryInMinutes" -Value $VPGActualHistoryInMinutes
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ConfiguredHistoryInMinutes" -Value $VPGConfiguredHistoryInMinutes
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "EarliestCheckpoint" -Value $VPGEarliestCheckpoint
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ActualFailSafeHistory" -Value $VPGActualFailSafeHistory
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "ConfiguredFailSafeHistory" -Value $VPGConfiguredFailSafeHistory
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "FailSafeDescription" -Value $VPGFailSafeDescription
$VPGTableRow | Add-Member -MemberType NoteProperty -Name "Zorg" -Value $VPGZorg
$VPGTable += $VPGTableRow
}																
################################################
# Getting Protected VMs
################################################
$VMListUrl = $BaseURL+"vms"
Try 
{
$VMList = Invoke-RestMethod -Method GET -Uri $VMListUrl -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
################################################
# Tidying Up VM Information
################################################
ForEach ($VM in $VMList)
{
# Setting values
$VMVmName = $VM.VmName
$VMVmIdentifier = $VM.VmIdentifier
$VMVpgName = $VM.VpgName
$VMVpgIdentifier = $VM.VpgIdentifier
$VMActualRPO = $VM.ActualRPO
$VMIsVmExists = $VM.IsVmExists
$VMStatus = $VM.Status
$VMSubStatus = $VM.SubStatus
$VMPriority = $VM.Priority
$VMProvisionedStorageInMB = $VM.ProvisionedStorageInMB
$VMUsedStorageInMB = $VM.UsedStorageInMB
$VMJournalUsedStorageMb = $VM.JournalUsedStorageMb
$VMJournalHardLimit = $VM.JournalHardLimit.LimitValue
$VMJournalHardLimitType = $VM.JournalHardLimit.LimitType
$VMJournalWarningThreshold = $VM.JournalWarningThreshold.LimitValue
$VMJournalWarningThresholdType = $VM.JournalWarningThreshold.LimitType
$VMOutgoingBandWidthInMbps = $VM.OutgoingBandWidthInMbps
$VMThroughputInMB = $VM.ThroughputInMB
$VMSourceSite = $VM.SourceSite
$VMProtectedSite = $VM.ProtectedSite.identifier
$VMTargetSite = $VM.TargetSite
$VMRecoverySite = $VM.RecoverySite.identifier
$VMOrganizationName = $VM.OrganizationName
$VMVolumes = $VM.Volumes.vmVolumeIdentifier
$VMIOPs = $VM.IOPs
$VMLastTest = $VM.LastTest
$VMIsFlrEnabled = $VM.EnabledActions.IsFlrEnabled
# Changing volumes to comma seperated list
IF ($VMVolumes -ne $null){$VMVolumes = $VMVolumes -Join " "}ELSE{$VMVolumes = $null}
# Adding row to table
$VPGVMTableRow = New-Object PSObject
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "VmName" -Value $VMVmName
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "VmIdentifier" -Value $VMVmIdentifier
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "VpgName" -Value $VMVpgName
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "VpgIdentifier" -Value $VMVpgIdentifier
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "ActualRPO" -Value $VMActualRPO
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "IsVmExists" -Value $VMIsVmExists
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "Status" -Value $VMStatus
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "SubStatus" -Value $VMSubStatus
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "Priority" -Value $VMPriority
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "ProvisionedStorageInMB" -Value $VMProvisionedStorageInMB
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "UsedStorageInMB" -Value $VMUsedStorageInMB
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "JournalUsedStorageMb" -Value $VMJournalUsedStorageMb
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "JournalHardLimit" -Value $VMJournalHardLimit
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "JournalHardLimitType" -Value $VMJournalHardLimitType
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "JournalWarningThreshold" -Value $VMJournalWarningThreshold
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "JournalWarningThresholdType" -Value $VMJournalWarningThresholdType
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "OutgoingBandWidthInMbps" -Value $VMOutgoingBandWidthInMbps
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "ThroughputInMB" -Value $VMThroughputInMB
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "SourceSite" -Value $VMSourceSite
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedSite" -Value $VMProtectedSite
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "TargetSite" -Value $VMTargetSite
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "RecoverySite" -Value $VMRecoverySite
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "OrganizationName" -Value $VMOrganizationName
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "Volumes" -Value $VMVolumes
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "IOPs" -Value $VMIOPs
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "LastTest" -Value $VMLastTest
$VPGVMTableRow | Add-Member -MemberType NoteProperty -Name "IsFlrEnabled" -Value $VMIsFlrEnabled
$VPGVMTable += $VPGVMTableRow
}																
################################################
# Getting Local Site ID
################################################
$LocalSiteURL = $BaseURL+"localsite"
Try 
{
$LocalSite = Invoke-RestMethod -Method GET -Uri $LocalSiteURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
$LocalSiteID = $LocalSite | Select -ExpandProperty SiteIdentifier
################################################
# Getting VRAs
################################################
$VRAListURL = $BaseURL+"vras"
Try 
{
$VRAList = Invoke-RestMethod -Method GET -Uri $VRAListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
################################################
# Tidying Up VRA Information
################################################
ForEach ($VRA in $VRAList)
{
# Setting values
$VRAVraName = $VRA.VraName
$VRAVraIdentifier = $VRA.VraIdentifier
$VRAStatus = $VRA.Status
$VRAProgress = $VRA.Progress
$VRAVraVersion = $VRA.VraVersion
$VRAHostVersion = $VRA.HostVersion
$VRAVraGroup = $VRA.VraGroup
$VRAMemoryInGB = $VRA.MemoryInGB
$VRAIPConfig = $VRA.VraNetworkDataApi.VraIPConfigurationTypeApi
$VRAIpAddress = $VRA.IpAddress
$VRASubnet = $VRA.VraNetworkDataApi.SubnetMask
$VRAGateway = $VRA.VraNetworkDataApi.DefaultGateway
$VRANetworkName = $VRA.NetworkName
$VRANetworkIdentifier = $VRA.NetworkIdentifier
$VRAHostIdentifier = $VRA.HostIdentifier
$VRADatastoreName = $VRA.DatastoreName
$VRADatastoreIdentifier = $VRA.DatastoreIdentifier
$VRADatastoreClusterName = $VRA.DatastoreClusterName
$VRADatastoreClusterIdentifier = $VRA.DatastoreClusterIdentifier
$VRAProtectedVpgs = $VRA.ProtectedCounters.Vpgs
$VRAProtectedVMs = $VRA.ProtectedCounters.Vms
$VRAProtectedVolumes = $VRA.ProtectedCounters.Volumes
$VRARecoveryVpgs = $VRA.RecoveryCounters.Vpgs
$VRARecoveryVMs = $VRA.RecoveryCounters.Vms
$VRARecoveryVolumes = $VRA.RecoveryCounters.Volumes
$VRASelfProtectedVpgs = $VRA.SelfProtectedVpgs
# Adding row to table
$VRATableRow = New-Object PSObject
$VRATableRow | Add-Member -MemberType NoteProperty -Name "VraName" -Value $VRAVraName
$VRATableRow | Add-Member -MemberType NoteProperty -Name "VraIdentifier" -Value $VRAVraIdentifier
$VRATableRow | Add-Member -MemberType NoteProperty -Name "Status" -Value $VRAStatus
$VRATableRow | Add-Member -MemberType NoteProperty -Name "Progress" -Value $VRAProgress
$VRATableRow | Add-Member -MemberType NoteProperty -Name "VraVersion" -Value $VRAVraVersion
$VRATableRow | Add-Member -MemberType NoteProperty -Name "HostVersion" -Value $VRAHostVersion
$VRATableRow | Add-Member -MemberType NoteProperty -Name "VraGroup" -Value $VRAVraGroup
$VRATableRow | Add-Member -MemberType NoteProperty -Name "MemoryInGB" -Value $VRAMemoryInGB
$VRATableRow | Add-Member -MemberType NoteProperty -Name "IpConfig" -Value $VRAIPConfig
$VRATableRow | Add-Member -MemberType NoteProperty -Name "IpAddress" -Value $VRAIpAddress
$VRATableRow | Add-Member -MemberType NoteProperty -Name "Subnet" -Value $VRASubnet
$VRATableRow | Add-Member -MemberType NoteProperty -Name "Gateway" -Value $VRAGateway
$VRATableRow | Add-Member -MemberType NoteProperty -Name "NetworkName" -Value $VRANetworkName
$VRATableRow | Add-Member -MemberType NoteProperty -Name "NetworkIdentifier" -Value $VRANetworkIdentifier
$VRATableRow | Add-Member -MemberType NoteProperty -Name "HostIdentifier" -Value $VRAHostIdentifier
$VRATableRow | Add-Member -MemberType NoteProperty -Name "DatastoreName" -Value $VRADatastoreName
$VRATableRow | Add-Member -MemberType NoteProperty -Name "DatastoreIdentifier" -Value $VRADatastoreIdentifier
$VRATableRow | Add-Member -MemberType NoteProperty -Name "DatastoreClusterName" -Value $VRADatastoreClusterName
$VRATableRow | Add-Member -MemberType NoteProperty -Name "DatastoreClusterIdentifier" -Value $VRADatastoreClusterIdentifier
$VRATableRow | Add-Member -MemberType NoteProperty -Name "ProtectedVPGs" -Value $VRAProtectedVpgs
$VRATableRow | Add-Member -MemberType NoteProperty -Name "ProtectedVMs" -Value $VRAProtectedVMs
$VRATableRow | Add-Member -MemberType NoteProperty -Name "ProtectedVolumes" -Value $VRAProtectedVolumes
$VRATableRow | Add-Member -MemberType NoteProperty -Name "RecoveryVPGs" -Value $VRARecoveryVpgs
$VRATableRow | Add-Member -MemberType NoteProperty -Name "RecoveryVMs" -Value $VRARecoveryVpgs
$VRATableRow | Add-Member -MemberType NoteProperty -Name "RecoveryVolumes" -Value $VRARecoveryVolumes
$VRATableRow | Add-Member -MemberType NoteProperty -Name "SelfProtectedVPGs" -Value $VRASelfProtectedVpgs
$VRATable += $VRATableRow
}										
################################################
# Get Target site IDs
################################################
$TargetSiteListURL = $BaseURL+"virtualizationsites"
Try 
{
$TargetSiteList = Invoke-RestMethod -Method GET -Uri $TargetSiteListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################################################################
# For Each Remote Site Getting Data
################################################################################################
ForEach ($TargetSite in $TargetSiteList)
{
$TargetSiteName = $TargetSite.VirtualizationSiteName
$TargetSiteID = $TargetSite.SiteIdentifier
################################################
# Getting Remote Site Unprotected VMs
################################################
$UnprotectedVMListURL = $BaseURL+"virtualizationsites/"+$TargetSiteID+"/vms"
Try 
{
$UnprotectedVMList = Invoke-RestMethod -Method GET -Uri $UnprotectedVMListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($UnprotectedVM in $UnprotectedVMList)
{
$UnprotectedVMID = $UnprotectedVM.VmIdentifier
$UnprotectedVMName = $UnprotectedVM.VmName
# Adding row to table
$UnprotectedVMTableRow = New-Object PSObject
$UnprotectedVMTableRow | Add-Member -MemberType NoteProperty -Name "VM" -Value $UnprotectedVMName
$UnprotectedVMTableRow | Add-Member -MemberType NoteProperty -Name "VMID" -Value $UnprotectedVMID
$UnprotectedVMTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$UnprotectedVMTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$UnprotectedVMTable += $UnprotectedVMTableRow
}
################################################
# Get Remote Site VMware Clusters
################################################
$ClusterListURL = $BaseURL+"virtualizationsites/$TargetSiteID/hostclusters"
Try 
{
$ClusterList = Invoke-RestMethod -Method GET -Uri $ClusterListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($Cluster in $ClusterList)
{
$ClusterID = $Cluster.ClusterIdentifier
$ClusterName = $Cluster.VirtualizationClusterName
# Adding row to table
$ClusterTableRow = New-Object PSObject
$ClusterTableRow | Add-Member -MemberType NoteProperty -Name "Cluster" -Value $ClusterName
$ClusterTableRow | Add-Member -MemberType NoteProperty -Name "ClusterID" -Value $ClusterID
$ClusterTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$ClusterTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$ClusterTable += $ClusterTableRow
}
################################################
# Get Remote Site VMware Hosts
################################################
$ESXiHostListURL = $BaseURL+"virtualizationsites/$TargetSiteID/hosts"
Try 
{
$ESXiHostList = Invoke-RestMethod -Method GET -Uri $ESXiHostListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($ESXiHost in $ESXiHostList)
{
$ESXiHostID = $ESXiHost.HostIdentifier
$ESXiHostName = $ESXiHost.VirtualizationHostName
# Adding row to table
$ESXiHostTableRow = New-Object PSObject
$ESXiHostTableRow | Add-Member -MemberType NoteProperty -Name "ESXiHost" -Value $ESXiHostName
$ESXiHostTableRow | Add-Member -MemberType NoteProperty -Name "ESXiHostID" -Value $ESXiHostID
$ESXiHostTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$ESXiHostTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$ESXiHostTable += $ESXiHostTableRow
}
################################################
# Get Remote Site Datastores
################################################
$DatastoreListURL = $BaseURL+"virtualizationsites/$TargetSiteID/datastores"
Try 
{
$DatastoreList = Invoke-RestMethod -Method GET -Uri $DatastoreListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($Datastore in $DatastoreList)
{
$DatastoreID = $Datastore.DatastoreIdentifier
$DatastoreName = $Datastore.DatastoreName
# Adding row to table
$DatastoreTableRow = New-Object PSObject
$DatastoreTableRow | Add-Member -MemberType NoteProperty -Name "Datastore" -Value $DatastoreName
$DatastoreTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreID" -Value $DatastoreID
$DatastoreTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$DatastoreTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$DatastoreTable += $DatastoreTableRow
}
################################################
# Get Remote Site Datastores Clusters
################################################
$DatastoreClusterListURL = $BaseURL+"virtualizationsites/$TargetSiteID/datastoreclusters"
Try 
{
$DatastoreClusterList = Invoke-RestMethod -Method GET -Uri $DatastoreClusterListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($DatastoreCluster in $DatastoreClusterList)
{
$DatastoreClusterID = $DatastoreCluster.DatastoreClusterIdentifier
$DatastoreClusterName = $DatastoreCluster.DatastoreClusterName
# Adding row to table
$DatastoreClusterTableRow = New-Object PSObject
$DatastoreClusterTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreCluster" -Value $DatastoreClusterName
$DatastoreClusterTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreClusterID" -Value $DatastoreClusterID
$DatastoreClusterTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$DatastoreClusterTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$DatastoreClusterTable += $DatastoreClusterTableRow
}
################################################
# Get Remote Site Port Group ID
################################################
$PortGroupListURL = $BaseURL+"virtualizationsites/$TargetSiteID/networks"
Try 
{
$PortGroupList = Invoke-RestMethod -Method GET -Uri $PortGroupListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($PortGroup in $PortGroupList)
{
$PortGroupID = $PortGroup.NetworkIdentifier
$PortGroupName = $PortGroup.VirtualizationNetworkName
# Adding row to table
$PortGroupTableRow = New-Object PSObject
$PortGroupTableRow | Add-Member -MemberType NoteProperty -Name "PortGroup" -Value $PortGroupName
$PortGroupTableRow | Add-Member -MemberType NoteProperty -Name "PortGroupID" -Value $PortGroupID
$PortGroupTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$PortGroupTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$PortGroupTable += $PortGroupTableRow
}
################################################
# Get Remote Site Folders ID
################################################
$FolderListURL = $BaseURL+"virtualizationsites/$TargetSiteID/folders"
Try 
{
$FolderList = Invoke-RestMethod -Method GET -Uri $FolderListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($Folder in $FolderList)
{
$FolderID = $Folder.FolderIdentifier
$FolderName = $Folder.FolderName
# Adding row to table
$FolderTableRow = New-Object PSObject
$FolderTableRow | Add-Member -MemberType NoteProperty -Name "Folder" -Value $FolderName
$FolderTableRow | Add-Member -MemberType NoteProperty -Name "FolderID" -Value $FolderID
$FolderTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$FolderTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$FolderTable += $FolderTableRow
################################################
# Get Remote Site Resource Pools
################################################
$ResourcePoolListURL = $BaseURL+"virtualizationsites/$TargetSiteID/resourcepools"
Try 
{
$ResourcePoolList = Invoke-RestMethod -Method GET -Uri $ResourcePoolListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Cycling through each object
ForEach ($ResourcePool in $ResourcePoolList)
{
$ResourcePoolID = $ResourcePool.ResourcePoolIdentifier
$ResourcePoolName = $ResourcePool.ResourcepoolName
# Adding row to table
$ResourcePoolTableRow = New-Object PSObject
$ResourcePoolTableRow | Add-Member -MemberType NoteProperty -Name "ResourcePool" -Value $ResourcePoolName
$ResourcePoolTableRow | Add-Member -MemberType NoteProperty -Name "ResourcePoolID" -Value $ResourcePoolID
$ResourcePoolTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$ResourcePoolTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$ResourcePoolTable += $ResourcePoolTableRow
}
}
################################################
# End of For Each Remote Site Getting Data below
################################################
}
# End of For Each Remote Site Getting Data above
################################################################################################
# Starting for each VPG action of collecting ZVM VPG data
################################################################################################
ForEach ($VPG in $VPGList)
{
$VPGName = $VPG.VpgName
$VPGID = $VPG.vpgidentifier
$VPGOrg = $VPG.OrganizationName
$VPGVMCount = $VPG.VmsCount
$VPGProvisionedStorageInMB = $VPG.ProvisionedStorageInMB
$VPGUsedStorageInMB = $VPG.UsedStorageInMB
$VPGJSON = 
"{
""VpgIdentifier"":""$VPGID""
}"
################################################
# Posting the VPG JSON Request to the API
################################################
$VPGEditSettingsURL = $BaseURL+"vpgSettings"
Try 
{
$VPGSettingsID = Invoke-RestMethod -Method POST -Uri $VPGEditSettingsURL -Body $VPGJSON -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
$ValidVPGSettingsID = $true
}
Catch 
{
$ValidVPGSettingsID = $false
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Getting VPG settings from API
################################################
# Skipping if unable to obtain valid VPG setting identifier
IF ($ValidVPGSettingsID -eq $true)
{
$VPGSettingsURL = $BaseURL+"vpgSettings/"+$VPGSettingsID
Try 
{
$VPGSettings = Invoke-RestMethod -Method GET -Uri $VPGSettingsURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Setting VPG settings from API
################################################
# Setting recovery site ID (needed anyway for network settings)
$VPGRecoverySiteID = $VPGSettings.Basic.RecoverySiteIdentifier
# Setting all VPG Settings
$VPGJournalHistoryInHours = $VPGSettings.Basic.JournalHistoryInHours
$VPGName = $VPGSettings.Basic.Name
$VPGPriortiy = $VPGSettings.Basic.Priority
$VPGProtectedSiteID = $VPGSettings.Basic.ProtectedSiteIdentifier
$VPGRpoInSeconds = $VPGSettings.Basic.RpoInSeconds
$VPGServiceProfileID = $VPGSettings.Basic.ServiceProfileIdentifier
$VPGTestIntervalInMinutes = $VPGSettings.Basic.TestIntervalInMinutes
$VPGUseWanCompression = $VPGSettings.Basic.UseWanCompression
$VPGZorgID = $VPGSettings.Basic.ZorgIdentifier
# Setting Boot Group IDs
$VPGBootGroups = $VPGSettings.BootGroups.BootGroups
$VPGBootGroupCount = $VPGSettings.BootGroups.BootGroups.Count
$VPGBootGroupNames = $VPGSettings.BootGroups.BootGroups.Name
$VPGBootGroupDelays = $VPGSettings.BootGroups.BootGroups.BootDelayInSeconds
$VPGBootGroupIDs = $VPGSettings.BootGroups.BootGroups.BootGroupIdentifier
# Converting multiple boot groups into list
IF ($VPGBootGroupNames -ne "Default"){$VPGBootGroupNames = $VPGBootGroupNames -Join " "}
IF ($VPGBootGroupDelays -ne "0"){$VPGBootGroupDelays = $VPGBootGroupDelays -Join " "}
IF ($VPGBootGroupIDs -ne "00000000-0000-0000-0000-000000000000"){$VPGBootGroupIDs = $VPGBootGroupIDs -Join " "}
# Setting Journal info
$VPGJournalDatastoreClusterID = $VPGSettings.Journal.DatastoreClusterIdentifier
$VPGJournalDatastoreID = $VPGSettings.Journal.DatastoreIdentifier
$VPGJournalHardLimitInMB = $VPGSettings.Journal.Limitation.HardLimitInMB
$VPGJournalHardLimitInPercent = $VPGSettings.Journal.Limitation.HardLimitInPercent
$VPGJournalWarningThresholdInMB = $VPGSettings.Journal.Limitation.WarningThresholdInMB
$VPGJournalWarningThresholdInPercent = $VPGSettings.Journal.Limitation.WarningThresholdInPercent
# Setting Network IDs
$VPGFailoverNetworkID = $VPGSettings.Networks.Failover.Hypervisor.DefaultNetworkIdentifier
$VPGFailoverTestNetworkID = $VPGSettings.Networks.FailoverTest.Hypervisor.DefaultNetworkIdentifier
# Setting recovery info
$VPGDefaultDatastoreID = $VPGSettings.Recovery.DefaultDatastoreIdentifier
$VPGDefaultFolderID = $VPGSettings.Recovery.DefaultFolderIdentifier
$VPGDefaultHostClusterID = $VPGSettings.Recovery.DefaultHostClusterIdentifier
$VPGDefaultHostID = $VPGSettings.Recovery.DefaultHostIdentifier
$VPGResourcePoolID = $VPGSettings.Recovery.ResourcePoolIdentifier
# Setting scripting info
$VPGScriptingPreRecovery = $VPGSettings.Scripting.PreRecovery
$VPGScriptingPostRecovery = $VPGSettings.Scripting.PostRecovery
# Setting VM IDs in VPG
$VPGVMIDs = $VPGSettings.VMs.VmIdentifier
################################################
# Translating Zerto IDs from VPG settings to friendly vSphere names
################################################
# Getting site names
$VPGProtectedSite = $TargetSiteList |  Where-Object {$_.SiteIdentifier -eq $VPGProtectedSiteID} | select -ExpandProperty VirtualizationSiteName
$VPGRecoverySite = $TargetSiteList |  Where-Object {$_.SiteIdentifier -eq $VPGRecoverySiteID} | select -ExpandProperty VirtualizationSiteName
# Getting network names
$VPGFailoverNetwork = $PortGroupTable | Where-Object {$_.PortGroupID -eq $VPGFailoverNetworkID}  | Select -ExpandProperty PortGroup
$VPGFailoverTestNetwork = $PortGroupTable | Where-Object {$_.PortGroupID -eq $VPGFailoverTestNetworkID}  | Select -ExpandProperty PortGroup
# Getting datastore cluster name
$VPGJournalDatastoreCluster = $DatastoreClusterTable | Where-Object {$_.DatastoreClusterID -eq $VPGJournalDatastoreClusterID} | select -ExpandProperty DatastoreCluster
# Getting datastore names
$VPGDefaultDatastore = $DatastoreTable | Where-Object {$_.DatastoreID -eq $VPGDefaultDatastoreID} | select -ExpandProperty Datastore
$VPGJournalDatastore = $DatastoreTable | Where-Object {$_.DatastoreID -eq $VPGJournalDatastoreID} | select -ExpandProperty Datastore
# Getting folder name
$VPGDefaultFolder = $FolderTable | Where-Object {$_.FolderID -eq $VPGDefaultFolderID} | select -ExpandProperty Folder
# Getting cluster name
$VPGDefaultHostCluster = $ClusterTable | Where-Object {$_.ClusterID -eq $VPGDefaultHostClusterID} | select -ExpandProperty Cluster
# Getting host name
$VPGDefaultHost = $ESXiHostTable | Where-Object {$_.ESXiHostID -eq $VPGDefaultHostID} | select -ExpandProperty ESXiHost
# Getting resource pool name
$VPGResourcePool = $ResourcePoolTable | Where-Object {$_.ResourcePoolID -eq $VPGResourcePoolID} | select -ExpandProperty Resourcepool
################################################
# Adding all VPG setting info to $VPGArray
################################################
$VPGSettingsTableRow = New-Object PSObject
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPG" -Value $VPGName
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPGID" -Value $VPGID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Org" -Value $VPGOrg
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VMCount" -Value $VPGVMCount
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ProvisionedStorageInMB" -Value $VPGProvisionedStorageInMB
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "UsedStorageInMB" -Value $VPGUsedStorageInMB
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Priortiy" -Value $VPGPriortiy
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedSite" -Value $VPGProtectedSite
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedSiteID" -Value $VPGProtectedSiteID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "RecoverySite" -Value $VPGRecoverySite
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "RecoverySiteID" -Value $VPGRecoverySiteID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "RpoInSeconds" -Value $VPGRpoInSeconds
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ServiceProfileID" -Value $VPGServiceProfileID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "TestIntervalInMins" -Value $VPGTestIntervalInMinutes
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "UseWanCompression" -Value $VPGUseWanCompression
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ZorgID" -Value $VPGZorgID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroupCount" -Value $VPGBootGroupCount
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroups" -Value $VPGBootGroupNames
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroupDelays" -Value $VPGBootGroupDelays
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroupIDs" -Value $VPGBootGroupIDs
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalHistoryInHours" -Value $VPGJournalHistoryInHours
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastoreCluster" -Value $VPGJournalDatastoreCluster
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastoreClusterID" -Value $VPGJournalDatastoreClusterID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastore" -Value $VPGJournalDatastore
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastoreID" -Value $VPGJournalDatastoreID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalHardLimitInMB" -Value $VPGJournalHardLimitInMB
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalHardLimitInPercent" -Value $VPGJournalHardLimitInPercent
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalWarningThresholdInMB" -Value $VPGJournalWarningThresholdInMB
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalWarningThresholdInPercent" -Value $VPGJournalWarningThresholdInPercent
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverNetwork" -Value $VPGFailoverNetwork
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverNetworkID" -Value $VPGFailoverNetworkID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestNetwork" -Value $VPGFailoverTestNetwork
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestNetworkID" -Value $VPGFailoverTestNetworkID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultDatastore" -Value $VPGDefaultDatastore
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultDatastoreID" -Value $VPGDefaultDatastoreID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultFolder" -Value $VPGDefaultFolder
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultFolderID" -Value $VPGDefaultFolderID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultCluster" -Value $VPGDefaultHostCluster
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultClusterID" -Value $VPGDefaultHostClusterID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultHost" -Value $VPGDefaultHost
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DefaultHostID" -Value $VPGDefaultHostID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ResourcePool" -Value $VPGResourcePool
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ResourcePoolID" -Value $VPGResourcePoolID
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "PreRecoveryScript" -Value $VPGScriptingPreRecovery
$VPGSettingsTableRow | Add-Member -MemberType NoteProperty -Name "PostRecoveryScript" -Value $VPGScriptingPostRecovery
$VPGSettingsTable += $VPGSettingsTableRow
################################################
# Starting For Each VM ID action for collecting ZVM VM data
################################################
ForEach ($VPGVMID in $VPGVMIDs)
{
################################################
# Getting VM settings from API
################################################
$VMSettingsURL = $BaseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VPGVMID
Try 
{
$VMSettings = Invoke-RestMethod -Method GET -Uri $VMSettingsURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Setting VM settings from API
################################################
# Setting the VM name and disk usage
$VMSettingsList = $VMList | Where-Object {$_.VmIdentifier -eq $VPGVMID} 
$VMName = $VMSettingsList | Select -ExpandProperty VmName
$VMProvisionedStorageInMB = $VMSettingsList | Select -ExpandProperty ProvisionedStorageInMB
$VMUsedStorageInMB = $VMSettingsList | Select -ExpandProperty UsedStorageInMB
# Setting variables from the API
$VMVolumeCount = $VMSettings.Volumes.Count
$VMNICCount = $VMSettings.Nics.Count
$VMBootGroupID = $VMSettings.BootGroupIdentifier
$VMJournalDatastoreClusterID = $VMSettings.Journal.DatastoreClusterIdentifier
$VMJournalDatastoreID = $VMSettings.Journal.DatastoreIdentifier
$VMJournalHardLimitInMB = $VMSettings.Journal.Limitation.HardLimitInMB
$VMJournalHardLimitInPercent = $VMSettings.Journal.Limitation.HardLimitInPercent
$VMJournalWarningThresholdInMB = $VMSettings.Journal.Limitation.WarningThresholdInMB
$VMJournalWarningThresholdInPercent = $VMSettings.Journal.Limitation.WarningThresholdInPercent
$VMDatastoreClusterID = $VMSettings.Recovery.DatastoreClusterIdentifier
$VMDatastoreID = $VMSettings.Recovery.DatastoreIdentifier
$VMFolderID = $VMSettings.Recovery.FolderIdentifier
$VMClusterID = $VMSettings.Recovery.HostClusterIdentifier
$VMHostID = $VMSettings.Recovery.HostIdentifier
$VMResourcePoolID = $VMSettings.Recovery.ResourcePoolIdentifier
################################################
# Translating Zerto IDs from VM settings to friendly vSphere names
################################################
# Getting boot group
$VMBootGroupName = $VPGBootGroups | Where-Object {$_.BootGroupIdentifier -eq $VMBootGroupID} | select -ExpandProperty Name
$VMBootGroupDelay = $VPGBootGroups | Where-Object {$_.BootGroupIdentifier -eq $VMBootGroupID} | select -ExpandProperty BootDelayInSeconds
# Getting datastore cluster name
$VMJournalDatastoreCluster = $DatastoreClusterTable | Where-Object {$_.DatastoreClusterID -eq $VMJournalDatastoreClusterID} | select -ExpandProperty DatastoreCluster
$VMDatastoreCluster = $DatastoreClusterTable | Where-Object {$_.DatastoreClusterID -eq $VMDatastoreClusterID} | select -ExpandProperty DatastoreCluster
# Getting datastore  name
$VMJournalDatastore = $DatastoreTable | Where-Object {$_.DatastoreID -eq $VMJournalDatastoreID} | select -ExpandProperty Datastore
$VMDatastore = $DatastoreTable | Where-Object {$_.DatastoreID -eq $VMDatastoreID} | select -ExpandProperty Datastore
# Getting folder name
$VMFolder = $FolderTable | Where-Object {$_.FolderID -eq $VMFolderID} | select -ExpandProperty Folder
# Getting cluster name
$VMCluster = $ClusterTable | Where-Object {$_.ClusterID -eq $VMClusterID} | select -ExpandProperty Cluster
# Getting host name
$VMHost = $ESXiHostTable | Where-Object {$_.ESXiHostID -eq $VMHostID} | select -ExpandProperty ESXiHost
# Getting resource pool name
$VMResourcePool = $ResourcePoolTable | Where-Object {$_.ResourcePoolID -eq $VMResourcePoolID} | select -ExpandProperty Resourcepool
################################################
# Adding all VM setting info to $VMArray
################################################
$VMSettingsTableRow = New-Object PSObject
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VM" -Value $VMName
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VPGVMID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPG" -Value $VPGName
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPGID" -Value $VPGID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedSite" -Value $VPGProtectedSite
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedSiteID" -Value $VPGProtectedSiteID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "RecoverySite" -Value $VPGRecoverySite
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "RecoverySiteID" -Value $VPGRecoverySiteID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "NICCount" -Value $VMNICCount
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VolumeCount" -Value $VMVolumeCount
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ProvisionedStorageInMB" -Value $VMProvisionedStorageInMB
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "UsedStorageInMB" -Value $VMUsedStorageInMB
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroupName" -Value $VMBootGroupName
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroupDelay" -Value $VMBootGroupDelay
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "BootGroupID" -Value $VMBootGroupID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastoreCluster" -Value $VMJournalDatastoreCluster
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastoreClusterID" -Value $VMJournalDatastoreClusterID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastore" -Value $VMJournalDatastore
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalDatastoreID" -Value $VMJournalDatastoreID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalHardLimitInMB" -Value $VMJournalHardLimitInMB
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "JournalHardLimitInPercent" -Value $VMJournalHardLimitInPercent
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreCluster" -Value $VMDatastoreCluster
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreClusterID" -Value $VMDatastoreClusterID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Datastore" -Value $VMDatastore
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreID" -Value $VMDatastoreID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Folder" -Value $VMFolder
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FolderID" -Value $VMFolderID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Cluster" -Value $VMCluster
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ClusterID" -Value $VMClusterID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Host" -Value $VMHost
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "HostID" -Value $VMHostID
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ResourcePool" -Value $VMResourcePool
$VMSettingsTableRow | Add-Member -MemberType NoteProperty -Name "ResourcePoolID" -Value $VMResourcePoolID
$VMSettingsTable += $VMSettingsTableRow
################################################
# Getting VM Volume settings from API
################################################
$VMVolumeSettingsURL = $BaseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VPGVMID+"/volumes"
Try 
{
$VMVolumeSettings = Invoke-RestMethod -Method GET -Uri $VMVolumeSettingsURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Starting for each VM Volume ID action for collecting ZVM VM Volume data
################################################
ForEach ($VMVolume in $VMVolumeSettings)
{
$VMVolumeID = $VMVolume.VolumeIdentifier
################################################
# Setting VM Volume settings from API
################################################
$VMVolumeDatastoreClusterID = $VMVolume.Datastore.DatastoreClusterIdentifier
$VMVolumeDatastoreID = $VMVolume.Datastore.DatastoreIdentifier
$VMVolumeIsSWAP = $VMVolume.IsSwap
$VMVolumeIsThin = $VMVolume.Datastore.IsThin
################################################
# Translating Zerto IDs from VM settings to friendly vSphere names
################################################
$VMVolumeDatastoreCluster = $DatastoreClusterTable | Where-Object {$_.DatastoreClusterID -eq $VMVolumeDatastoreClusterID} | select -ExpandProperty DatastoreCluster
$VMVolumeDatastore = $DatastoreTable | Where-Object {$_.DatastoreID -eq $VMVolumeDatastoreID} | select -ExpandProperty Datastore
################################################
# Adding all VM Volume setting info to $VMVolumeArray
################################################
$VMVolumeSettingsTableRow = New-Object PSObject
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VM" -Value $VMName
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VPGVMID
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPG" -Value $VPGName
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPGID" -Value $VPGID
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VolumeID" -Value $VMVolumeID
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VolumeIsSWAP" -Value $VMVolumeIsSWAP
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VolumeIsThin" -Value $VMVolumeIsThin
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreCluster" -Value $VMVolumeDatastoreCluster
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreClusterID" -Value $VMVolumeDatastoreClusterID
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "Datastore" -Value $VMVolumeDatastore
$VMVolumeSettingsTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreID" -Value $VMVolumeDatastoreID
$VMVolumeSettingsTable += $VMVolumeSettingsTableRow
}
################################################
# Getting VM NIC List from API
################################################
$VMNICListURL = $BaseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VPGVMID+"/nics"
Try 
{
$VMNICList = Invoke-RestMethod -Method GET -Uri $VMNICListURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Starting For Each VM NIC ID action for collecting ZVM VM NIC data
################################################
ForEach ($VMNIC in $VMNICList)
{
$VMNICID = $VMNIC.NicIdentifier
################################################
# Getting VM NIC Settings from API
################################################
$VMNICSettingsURL = $BaseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VPGVMID+"/nics/"+$VMNICID
Try 
{
$VMNICSettings = Invoke-RestMethod -Method GET -Uri $VMNICSettingsURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Setting VM NIC Settings from API
################################################
$VMSettingNICIDArray1 = $VMNICSettings.Failover.Hypervisor
$VMSettingNICIDArray2 = $VMNICSettings.Failover.Hypervisor.IpConfig
$VMSettingNICIDArray3 = $VMNICSettings.FailoverTest.Hypervisor
$VMSettingNICIDArray4 = $VMNICSettings.FailoverTest.Hypervisor.IpConfig
# Setting failover values
$VMNICFailoverDNSSuffix = $VMSettingNICIDArray1.DnsSuffix
$VMNICFailoverNetworkID = $VMSettingNICIDArray1.NetworkIdentifier
$VMNICFailoverShouldReplaceMacAddress = $VMSettingNICIDArray1.ShouldReplaceMacAddress
$VMNICFailoverGateway = $VMSettingNICIDArray2.Gateway
$VMNIsFailoverDHCP = $VMSettingNICIDArray2.IsDhcp
$VMNICFailoverPrimaryDns = $VMSettingNICIDArray2.PrimaryDns
$VMNICFailoverSecondaryDns = $VMSettingNICIDArray2.SecondaryDns
$VMNICFailoverStaticIp = $VMSettingNICIDArray2.StaticIp
$VMNICFailoverSubnetMask = $VMSettingNICIDArray2.SubnetMask
# Nulling blank content
IF ($VMNICFailoverDNSSuffix.nil -eq $true){$VMNICFailoverDNSSuffix = $null}
IF ($VMNICFailoverGateway.nil -eq $true){$VMNICFailoverGateway = $null}
IF ($VMNICFailoverPrimaryDns.nil -eq $true){$VMNICFailoverPrimaryDns = $null}
IF ($VMNICFailoverSecondaryDns.nil -eq $true){$VMNICFailoverSecondaryDns = $null}
IF ($VMNICFailoverStaticIp.nil -eq $true){$VMNICFailoverStaticIp = $null}
IF ($VMNICFailoverSubnetMask.nil -eq $true){$VMNICFailoverSubnetMask = $null}
# Setting failover test values
$VMNICFailoverTestDNSSuffix = $VMSettingNICIDArray3.DnsSuffix
$VMNICFailoverTestNetworkID = $VMSettingNICIDArray3.NetworkIdentifier
$VMNICFailoverTestShouldReplaceMacAddress = $VMSettingNICIDArray3.ShouldReplaceMacAddress
$VMNICFailoverTestGateway = $VMSettingNICIDArray4.Gateway
$VMNIsFailoverTestDHCP = $VMSettingNICIDArray4.IsDhcp
$VMNICFailoverTestPrimaryDns = $VMSettingNICIDArray4.PrimaryDns
$VMNICFailoverTestSecondaryDns = $VMSettingNICIDArray4.SecondaryDns
$VMNICFailoverTestStaticIp = $VMSettingNICIDArray4.StaticIp
$VMNICFailoverTestSubnetMask = $VMSettingNICIDArray4.SubnetMask
# Nulling blank content
IF ($VMNICFailoverTestDNSSuffix.nil -eq $true){$VMNICFailoverTestDNSSuffix = $null}
IF ($VMNICFailoverTestGateway.nil -eq $true){$VMNICFailoverTestGateway = $null}
IF ($VMNICFailoverTestPrimaryDns.nil -eq $true){$VMNICFailoverTestPrimaryDns = $null}
IF ($VMNICFailoverTestSecondaryDns.nil -eq $true){$VMNICFailoverTestSecondaryDns = $null}
IF ($VMNICFailoverTestStaticIp.nil -eq $true){$VMNICFailoverTestStaticIp = $null}
IF ($VMNICFailoverTestSubnetMask.nil -eq $true){$VMNICFailoverTestSubnetMask = $null}
# Mapping Network IDs to Names
$VMNICFailoverNetwork = $PortGroupTable | Where-Object {$_.PortGroupID -eq $VMNICFailoverNetworkID}  | Select -ExpandProperty PortGroup 
$VMNICFailoverTestNetwork = $PortGroupTable | Where-Object {$_.PortGroupID -eq $VMNICFailoverTestNetworkID}  | Select -ExpandProperty PortGroup 
################################################
# Adding all VM NIC setting info to $VMNICArray
################################################
$VMNICSettingsTableRow = New-Object PSObject
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPG" -Value $VPGName
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VPGID" -Value $VPGID
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VM" -Value $VMName
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VPGVMID
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "NICID" -Value $VMNICID
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverNetwork" -Value $VMNICFailoverNetwork
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverNetworkID" -Value $VMNICFailoverNetworkID
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverDNSSuffix" -Value $VMNICFailoverDNSSuffix
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverShouldReplaceMacAddress" -Value $VMNICFailoverShouldReplaceMacAddress
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverGateway" -Value $VMNICFailoverGateway
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverDHCP" -Value $VMNIsFailoverDHCP
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverPrimaryDns" -Value $VMNICFailoverPrimaryDns
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverSecondaryDns" -Value $VMNICFailoverSecondaryDns
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverStaticIp" -Value $VMNICFailoverStaticIp
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverSubnetMask" -Value $VMNICFailoverSubnetMask
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestNetwork" -Value $VMNICFailoverTestNetwork
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestNetworkID" -Value $VMNICFailoverTestNetworkID
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestDNSSuffix" -Value $VMNICFailoverTestDNSSuffix
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestShouldReplaceMacAddress" -Value $VMNICFailoverTestShouldReplaceMacAddress
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestGateway" -Value $VMNICFailoverTestGateway
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestDHCP" -Value $VMNIsFailoverTestDHCP
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestPrimaryDns" -Value $VMNICFailoverTestPrimaryDns
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestSecondaryDns" -Value $VMNICFailoverTestSecondaryDns
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestStaticIp" -Value $VMNICFailoverTestStaticIp
$VMNICSettingsTableRow | Add-Member -MemberType NoteProperty -Name "FailoverTestSubnetMask" -Value $VMNICFailoverTestSubnetMask
$VMNICSettingsTable += $VMNICSettingsTableRow
# End of per VM NIC actions below
}
# End of per VM NIC actions above
################################################
# End of per VM actions below
################################################
}
# End of per VM actions above
################################################
# Deleting VPG edit settings ID (same as closing the edit screen on a VPG in the ZVM without making any changes)
################################################
Try
{
Invoke-RestMethod -Method DELETE -Uri $VPGSettingsURL -Headers $ZertoSessionHeader -ContentType $Type -SkipCertificateCheck
}
Catch [system.exception]
{
}
#
################################################
# End of check for valid VPG settings ID below
################################################
}
# End of check for valid VPG settings ID above
#
################################################
# End of per VPG actions below
################################################
}
# End of per VPG actions above
################################################################################################
# Final part of the script - summarizing the data
################################################################################################
################################################
# Datastore VPG usage 
################################################
$DatastoreVPGUsageTable = @()
ForEach ($Datastore in $DatastoreTable)
{
# Setting values
$DatastoreName = $Datastore.Datastore
$DatastoreID = $Datastore.DatastoreID
$DatastoreSite = $Datastore.Site
$DatastoreSiteID = $Datastore.SiteID
# Getting VPG usage
$VPGUsage1List = $VPGSettingsTable | Where-Object {$_.DefaultDatastoreID -eq $DatastoreID} | Select -ExpandProperty VPG -Unique
IF ($VPGUsage1List -ne $null){$VPGUsage1 = $VPGUsage1List -Join " "}ELSE{$VPGUsage1 = $null}
$VPGUsage2List = $VPGSettingsTable | Where-Object {$_.JournalDatastoreID -eq $DatastoreID} | Select -ExpandProperty VPG -Unique
IF ($VPGUsage2List -ne $null){$VPGUsage2 = $VPGUsage2List -Join " "}ELSE{$VPGUsage2 = $null}
# Adding row to table
$DatastoreVPGUsageTableRow = New-Object PSObject
$DatastoreVPGUsageTableRow | Add-Member -MemberType NoteProperty -Name "Datastore" -Value $DatastoreName
$DatastoreVPGUsageTableRow | Add-Member -MemberType NoteProperty -Name "DatastoreID" -Value $DatastoreID
$DatastoreVPGUsageTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $DatastoreSite
$DatastoreVPGUsageTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $DatastoreSiteID
$DatastoreVPGUsageTableRow | Add-Member -MemberType NoteProperty -Name "VPGDataDisks" -Value $VPGUsage1
$DatastoreVPGUsageTableRow | Add-Member -MemberType NoteProperty -Name "VPGJournalDisks" -Value $VPGUsage2
$DatastoreVPGUsageTable += $DatastoreVPGUsageTableRow
}
$DatastoreVPGUsageTable = $DatastoreVPGUsageTable | Where-Object {(($_.VPGDisks -ne $null) -or ($_.VPGJournalDisks -ne $null))} | Select -Unique
################################################
# Host VM load
################################################
ForEach ($ESXiHost in $ESXiHostTable)
{
$ESXiHostID = $ESXiHost.ESXiHostID
$ESXiHostName = $ESXiHost.ESXiHost
$ESXiHostSite = $ESXiHost.Site
$ESXiHostSiteID = $ESXiHost.SiteID
# Getting Host usage
$VMHostUsageCount = $VMSettingsTable | Where-Object {$_.HostID -eq $ESXiHostID} | Measure | Select -ExpandProperty Count
$VMHostVPGs = $VMSettingsTable | Where-Object {$_.HostID -eq $ESXiHostID} | Select -ExpandProperty VPG -Unique
IF ($VMHostVPGs -ne $null){$VMHostVPGs = $VMHostVPGs -Join " "}ELSE{$VMHostVPGs = $null}
# Adding row to table
$HostVMUsageTableRow = New-Object PSObject
$HostVMUsageTableRow | Add-Member -MemberType NoteProperty -Name "Host" -Value $ESXiHostName
$HostVMUsageTableRow | Add-Member -MemberType NoteProperty -Name "HostID" -Value $ESXiHostID
$HostVMUsageTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $ESXiHostSite
$HostVMUsageTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $ESXiHostSiteID
$HostVMUsageTableRow | Add-Member -MemberType NoteProperty -Name "VMCount" -Value $VMHostUsageCount
$HostVMUsageTableRow | Add-Member -MemberType NoteProperty -Name "VPGs" -Value $VMHostVPGs
$HostVMUsageTable += $HostVMUsageTableRow
}
################################################
# Site totals
################################################
ForEach ($TargetSite in $TargetSiteList)
{
$TargetSiteID = $TargetSite.SiteIdentifier
$TargetSiteName = $TargetSite.VirtualizationSiteName
# Getting totals for VPGs
$TotalProtectedVPGs = $VPGSettingsTable | Where-Object {$_.ProtectedSiteID -eq $TargetSiteID} | Measure | Select -ExpandProperty Count
$TotalRecoveryVPGs = $VPGSettingsTable | Where-Object {$_.RecoverySiteID -eq $TargetSiteID} | Measure | Select -ExpandProperty Count
$TotalProtectedStorageInMB = $VPGSettingsTable | Where-Object {$_.ProtectedSiteID -eq $TargetSiteID} | Select -ExpandProperty ProvisionedStorageInMB | Measure -Sum | Select -ExpandProperty Sum
$TotalUsedStorageInMB = $VPGSettingsTable | Where-Object {$_.RecoverySiteID -eq $TargetSiteID} | Select -ExpandProperty UsedStorageInMB | Measure -Sum | Select -ExpandProperty Sum
# Converting MB to TB
$TotalProtectedStorageInTB = $TotalProtectedStorageInMB / 1000 / 1000
$TotalUsedStorageInTB = $TotalUsedStorageInMB / 1000 / 1000
$TotalProtectedStorageInTB = [Math]::Round($TotalProtectedStorageInTB, 1)
$TotalUsedStorageInTB = [Math]::Round($TotalUsedStorageInTB, 1)
# Getting totals for VMs
$TotalProtectedVMs = $VMSettingsTable | Where-Object {$_.ProtectedSiteID -eq $TargetSiteID} | Measure | Select -ExpandProperty Count
$TotalUnProtectedVMs = $UnprotectedVMTable | Where-Object {$_.SiteID -eq $TargetSiteID} | Measure | Select -ExpandProperty Count
$TotalRecoveryVMs = $VMSettingsTable | Where-Object {$_.RecoverySiteID -eq $TargetSiteID} | Measure | Select -ExpandProperty Count
# Adding row to table
$SiteTotalsTableRow = New-Object PSObject
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "Site" -Value $TargetSiteName
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "SiteID" -Value $TargetSiteID
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "UnProtectedVMs" -Value $TotalUnProtectedVMs
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedVPGs" -Value $TotalProtectedVPGs
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedVMs" -Value $TotalProtectedVMs
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "RecoveryVPGs" -Value $TotalRecoveryVPGs
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "RecoveryVMs" -Value $TotalRecoveryVMs
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "ProtectedStorageInTB" -Value $TotalProtectedStorageInTB
$SiteTotalsTableRow | Add-Member -MemberType NoteProperty -Name "RecoveryStorageInTB" -Value $TotalUsedStorageInTB
$SiteTotalsTable += $SiteTotalsTableRow
}
################################################################################################
# Exporting data to CSVs
################################################################################################
# From APIs to table to CSV
$VPGTable | Sort-Object VpgName | Export-CSV $VPGExport -NoTypeInformation -Force
$VPGVMTable | Sort-Object VmName | Export-CSV $ProtectedVMExport -NoTypeInformation -Force
$VRATable | Sort-Object Vraname | Export-CSV $VRAVMExport -NoTypeInformation -Force
# From APIs to CSV
$TargetSiteList | Sort-Object VirtualizationSiteName | Export-CSV $TargetSiteExport -NoTypeInformation -Force
# From APIs to For Each TargetSite into table to CSV
$UnprotectedVMTable | Sort-Object Site,VM | Export-CSV $UnprotectedVMExport -NoTypeInformation -Force
$ClusterTable | Sort-Object Site,Cluster | Export-CSV $ClustersExport -NoTypeInformation -Force
$ESXiHostTable | Sort-Object Site,ESXiHost | Export-CSV $HostsExport -NoTypeInformation -Force
$DatastoreTable | Sort-Object Site,Datastore | Export-CSV $DatastoresExport -NoTypeInformation -Force
$DatastoreClusterTable | Sort-Object Site,DatastoreCluster | Export-CSV $DatastoreClustersExport -NoTypeInformation -Force
$PortGroupTable | Sort-Object Site,PortGroup | Export-CSV $PortGroupsExport -NoTypeInformation -Force
$FolderTable | Sort-Object Site,Folder | Export-CSV $FoldersExport -NoTypeInformation -Force
$ResourcePoolTable | Sort-Object Site,ResourcePool | Export-CSV $ResourcePoolsExport -NoTypeInformation -Force
$VPGSettingsTable | Sort-Object Site,VPG | Export-CSV $VPGSettingsExport -NoTypeInformation -Force
$VMSettingsTable | Sort-Object Site,VPG,VM | Export-CSV $VMSettingsExport -NoTypeInformation -Force
$VMVolumeSettingsTable | Sort-Object Site,VPG,VM,VolumeID | Export-CSV $VolumeSettingsExport -NoTypeInformation -Force
$VMNICSettingsTable | Sort-Object Site,VPG,VM,NICID | Export-CSV $NICSettingsExport -NoTypeInformation -Force
# From summary data to table to CSV
$DatastoreVPGUsageTable | Sort-Object Site,Datastore,VPGs | Export-CSV $DatastoreVPGUsageExport -NoTypeInformation -Force
$HostVMUsageTable | Sort-Object Site,Host,VMCount | Export-CSV $HostVPGUsageExport -NoTypeInformation -Force
$SiteTotalsTable | Sort-Object Site | Export-CSV $SiteTotalsExport -NoTypeInformation -Force
################################################
# End of bypass if Zerto not authenticated below
################################################
}
ELSE
{
"Not authenticated with Zerto, please check and try again"
}
################################################
# End of script
################################################