﻿#####################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
#####################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
##############################################
# Description:
# vSphere CMDB script 4 of 5. This connects to the SQL server/vCMDB specified then gives you some example queries/reports to run against your data.
##############################################
# Requirements:
# - Set-executionpolicy unrestricted on the computer running the script
# - Access to a SQL server instance and vCMDB with sufficient permissions to query the database already created
##############################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever. 
# Including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss arising out of the use of,
# or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##############################################
# Configure the variables below for connecting to the SQL database
##############################################
$SQLInstance = ".\SQLEXPRESS"
$SQLDatabase = "vSphereCMDB"
##############################################
# Importing the SQL credentials
##############################################
$SQLCredentials = IMPORT-CLIXML ".\SQLCredentials.xml"
$SQLUsername = $SQLCredentials.UserName
$SQLPassword = $SQLCredentials.GetNetworkCredential().Password
##############################################
# Checking to see if the SqlServer module is already installed, if not installing it
##############################################
$SQLModuleCheck = Get-Module -ListAvailable SqlServer
if ($SQLModuleCheck -eq $null)
{
write-host "SqlServer Module Not Found - Installing"
# Not installed, trusting PS Gallery to remove prompt on install
Set-PSRepository -Name PSGallery -InstallationPolicy Trusted
# Installing module
Install-Module -Name SqlServer –Scope AllUsers -Confirm:$false -AllowClobber
}
##############################################
# Importing the SqlServer module
##############################################
Import-Module SqlServer
##############################################
# Querying all the tables and assigning them to variables for subsequent use
##############################################
# VM table 
$VMsTableQuery = "USE $SQLDatabase
SELECT * FROM VMs"
$VMsTable = invoke-sqlcmd -query $VMsTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# VMDisks table 
$VMDisksTableQuery = "USE $SQLDatabase
SELECT * FROM VMDisks"
$VMDisksTable = invoke-sqlcmd -query $VMDisksTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# VMDiskUsage table 
$VMDiskUsageTableQuery = "USE $SQLDatabase
SELECT * FROM VMDisks"
$VMDiskUsageTable = invoke-sqlcmd -query $VMDiskUsageTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# VMNICs table 
$VMNICsTableQuery = "USE $SQLDatabase
SELECT * FROM VMNICs"
$VMNICsTable = invoke-sqlcmd -query $VMNICsTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# Datastores table 
$DatastoresTableQuery = "USE $SQLDatabase
SELECT * FROM VMNICs"
$DatastoresTable = invoke-sqlcmd -query $DatastoresTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# PortGroups table 
$PortGroupsTableQuery = "USE $SQLDatabase
SELECT * FROM PortGroups"
$PortGroupsTable = invoke-sqlcmd -query $PortGroupsTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# Hosts table 
$HostsTableQuery = "USE $SQLDatabase
SELECT * FROM Hosts"
$HostsTable = invoke-sqlcmd -query $HostsTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# Clusters table 
$ClustersTableQuery = "USE $SQLDatabase
SELECT * FROM Clusters"
$ClustersTable = invoke-sqlcmd -query $ClustersTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
# Datacenters table 
$DatacentersTableQuery = "USE $SQLDatabase
SELECT * FROM Datacenters"
$DatacentersTable = invoke-sqlcmd -query $DatacentersTableQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
##############################################
# Example queries from the VM table
##############################################
# Show all configurations of a VM
$VMHistory = $VMsTable | Where-Object {$_.Name -eq "SQLServer-VM01"}
$VMHistory | Sort-Object LastUpdated | Out-GridView -Title "VM History"
# List all unique dates in the VM table from which to take a sample
$VMDatesAvailable = $VMsTable | select LastUpdated -Unique
# Taking oldest date available and showing all VM configs
$VMOldestDateSelected = $VMDatesAvailable | Sort-Object LastUpdated | select -ExpandProperty LastUpdated -First 1
$VMsTable | Where-Object {$_.LastUpdated -eq $VMOldestDateSelected} | Out-GridView -Title "VM Point In Time Config"
# Taking most recent data available and showing total VM size provisioned and used (useful for Rubrik sizing)
$VMNewestDateSelected = $VMDatesAvailable | Sort-Object LastUpdated | select -ExpandProperty LastUpdated -Last 1
$VMProvisionedSpaceGB = $VMsTable | Where-Object {$_.LastUpdated -eq $VMNewestDateSelected} | select -ExpandProperty ProvisionedSpaceGB | Measure-Object -Sum | Select-Object -ExpandProperty Sum
write-host "Total VM ProvisionedSpace (GB): $VMProvisionedSpaceGB"
$VMUsedSpaceGB = $VMsTable | Where-Object {$_.LastUpdated -eq $VMNewestDateSelected} | select -ExpandProperty UsedSpaceGB | Measure-Object -Sum | Select-Object -ExpandProperty Sum
write-host "Total VM UsedSpace (GB): $VMUsedSpaceGB"
# Taking most recent and oldest data available and then calculating growth over the period (also useful for Rubrik sizing!)
$VMOldestUsedSpaceGB = $VMsTable | Where-Object {$_.LastUpdated -eq $VMOldestDateSelected} | select -ExpandProperty UsedSpaceGB | Measure-Object -Sum | Select-Object -ExpandProperty Sum
$VMNewestUsedSpaceGB = $VMsTable | Where-Object {$_.LastUpdated -eq $VMNewestDateSelected} | select -ExpandProperty UsedSpaceGB | Measure-Object -Sum | Select-Object -ExpandProperty Sum
# Calculating time sample in days
$TimeSpan = New-Timespan –Start $VMOldestDateSelected –End $VMNewestDateSelected | Select -ExpandProperty TotalDays 
# Calculating growth percentage
$Diff = $VMNewestUsedSpaceGB - $VMOldestUsedSpaceGB
$PercentDiff = ($Diff / $VMOldestUsedSpaceGB) * 100
write-host "Time Sample in Days: $TimeSpan
Total VM UsedSpace Growth (%): $PercentDiff"
##############################################
# Example queries from the VMDisks table
##############################################
# Show all VMDisk configurations for a VM
$VMDiskConfigHistory = $VMDisksTable | Where-Object {$_.Parent -eq "HCHLvCenter"}
$VMDiskConfigHistory | Sort-Object LastUpdated | Out-GridView -Title "VMDisk Config History"
# Show all configurations for a VMDK
$VMDiskConfigHistory = $VMDisksTable | Where-Object {($_.Parent -eq "HCHLvCenter") -and ($_.Name -eq "Hard disk 1")}
$VMDiskConfigHistory | Sort-Object LastUpdated | Out-GridView -Title "VMDisk Config History"
##############################################
# Example queries from the VMNICs table
##############################################
# Show all VMNIC configurations for a VM
$VMNICConfigHistory = $VMNICsTable | Where-Object {$_.Parent -eq "DC1vCenter"}
$VMNICConfigHistory | Sort-Object LastUpdated | Out-GridView -Title "VMNIC Config History"
# Show all configurations for a VMNIC
$VMNICConfigHistory = $VMNICsTable | Where-Object {($_.Parent -eq "DC1vCenter") -and ($_.Name -eq "Network adapter 1")}
$VMNICConfigHistory | Sort-Object LastUpdated | Out-GridView -Title "VMNIC Config History"
##############################################
# Example queries from the PortGroupsTable table
##############################################
# Show all configurations for a Port Group on a host
$PortGroupHistory = $PortGroupsTable | Where-Object {($_.Name -eq "VM Network") -and ($_.VMHostId -eq "HostSystem-host-74")}
$PortGroupHistory | Sort-Object LastUpdated | Out-GridView -Title "Host Port Group History"
##############################################
# Example queries from the Hosts table
##############################################
# Show all configurations of a Host
$HostHistory = $HostsTable | Where-Object {$_.Name -eq "192.168.0.13"}
$HostHistory | Sort-Object LastUpdated | Out-GridView -Title "Host History"
# List all unique dates in the Host table from which to take a sample
$HostDatesAvailable = $HostsTable | select LastUpdated -Unique
# Taking oldest date available and showing all Host configs
$HostDateSelected = $HostDatesAvailable | Sort-Object LastUpdated | select -ExpandProperty LastUpdated -First 1
$HostsTable | Where-Object {$_.LastUpdated -eq $HostDateSelected} | Out-GridView -Title "Host Point In Time Config"
##############################################
# Example queries from the Clusters table
##############################################
# Show all configurations of a Cluster
$ClusterHistory = $ClustersTable | Where-Object {$_.Name -eq "ProdCluster1"}
$ClusterHistory | Sort-Object LastUpdated | Out-GridView -Title "Cluster History"
# List all unique dates in the Cluster table from which to take a sample
$ClusterDatesAvailable = $ClustersTable | select LastUpdated -Unique
# Taking oldest date available and showing all Host configs
$ClusterDateSelected = $ClusterDatesAvailable | Sort-Object LastUpdated | select -ExpandProperty LastUpdated -First 1
$ClustersTable | Where-Object {$_.LastUpdated -eq $ClusterDateSelected} | Out-GridView -Title "Cluster Point In Time Config"
##############################################
# Example queries from the Datacenters table
##############################################
# Show all stats for a Datacenter
$DatacenterHistory = $DatacentersTable | Where-Object {$_.Name -eq "Datacenter1"}
$DatacenterHistory | Sort-Object LastUpdated | Out-GridView -Title "Datacenter History"
# List all unique dates in the Datacenter table from which to take a sample
$DatacenterDatesAvailable = $DatacentersTable | select LastUpdated -Unique
# Taking oldest date available and showing all Datacenter stats
$DatacenterDateSelected = $DatacenterDatesAvailable | Sort-Object LastUpdated | select -ExpandProperty LastUpdated -First 1
$DatacentersTable | Where-Object {$_.LastUpdated -eq $DatacenterDateSelected} | Out-GridView -Title "Datacenter Point In Time Config"
#######################
# End of script
#######################