﻿#####################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
#####################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
##############################################
# Description:
# This script deploys Rubrik EDGE/AIR to 1 host specified
##############################################
# Requirements:
# - Internet access on the computer running this script to download the latest module
# - Run the script as administrator to enable the uninstall of the existing module and to allow the install of the new module for all users
##############################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever. 
# Including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss arising out of the use of,
# or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
#######################
# Configure the variables below
#######################
# File & folder settings
$ScriptDirectory = "S:\RubrikOVFDeployerv1"
$LogDirectory = "S:\RubrikOVFDeployerv1\Logs"
# CSV File
$CSVToImport = "S:\RubrikOVFDeployerv1\RubrikOVFsToDeploy.csv"
# OVF locations
$EDGEOVF = $ScriptDirectory + "\rubrik-edge.ovf"
$AIROVF = $ScriptDirectory + "\rubrik-air.ovf"
# OVF type deciding factor
$UseAIRAboveGB = 4096
# Time delays
$OVFBootDelaySecs = 60 # Time to wait for VM after power on before attempting bootstrap, recommended 1-2 mins
$RubrikPreBootstrapDelaySecs = 120 # Time to wait for bootstrap cluster service to become available after cluster bootstrap command, recommended 2-3 mins
$RubrikPostBootstrapDelaySecs = 1200 # Time to wait for bootstrap to finish, recommended 20 mins, range in test lab 11-19 mins
# Rubrik cluster settings
$RemoveDefaultSLAs = $TRUE
$AddvCenter = $TRUE
# Login banner
$AddLoginBanner = $TRUE
$LoginBannerText = "Welcome to Rubrik! <br></br> This VM was deployed end to end using a script from virtuallysober.com"
# SMTP settings
$AddSMTPServer = $TRUE
$SMTPServer = "192.168.1.198"
$SMTPPort = "25"
$SMTPEncryption = "NONE" # Valid SMTP encryption settings are NONE, SSL, STARTTLS
#####################################################################
# Nothing to change below this line, commented throughout
#####################################################################
##################################
# Creating directories if not exists
##################################
$LogDirectoryTest = Test-Path $LogDirectory
IF ($LogDirectoryTest -eq $False)
{
New-Item -Path $LogDirectory -ItemType "directory" | Out-Null
}
##################################
# Importing CSV
##################################
$CSVToImportTest = Test-Path $CSVToImport
IF ($CSVToImportTest -eq $True)
{
$CSVImport = Import-Csv $CSVToImport
}
ELSE
{
Write-Host "CSVNotFound: $CSVToImport"
Sleep 10
Exit
}
# Removing null lines
$CSVImport = $CSVImport | Where {$_.vCenter -ne ""}
#######################
# OVF defaults
#######################
$OVFSizeMB = 4354
$OVFUser = "admin"
$OVFPassword = "rubrik"
#######################
# Adding Assembly Required
#######################
Add-Type -AssemblyName System.Web
#######################
# Adding certificate exception and TLS 1.2 to prevent API errors
#######################
Add-Type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
#####################################################################
# Functions Required (Thanks to @lamw for Set-VMKeystrokes)
#####################################################################
Function Set-VMKeystrokes {
<#
    .NOTES
    ===========================================================================
     Created by:    William Lam
     Organization:  VMware
     Blog:          www.virtuallyghetto.com
     Twitter:       @lamw
    ===========================================================================
    .DESCRIPTION
        This function sends a series of character keystrokse to a particular VM
    .PARAMETER VMName
        The name of a VM to send keystrokes to
    .PARAMETER StringInput
        The string of characters to send to VM
    .PARAMETER DebugOn
        Enable debugging which will output input charcaters and their mappings
    .EXAMPLE
        Set-VMKeystrokes -VMName $VM -StringInput "root"
        Push "root" to VM $VM
    .EXAMPLE
        Set-VMKeystrokes -VMName $VM -StringInput "root" -ReturnCarriage $true
        Push "root" with return line to VM $VM
    .EXAMPLE
        Set-VMKeystrokes -VMName $VM -StringInput "root" -DebugOn $true
        Push "root" to VM $VM with some debug
    ===========================================================================
     Modified by:   David Rodriguez
     Organization:  Sysadmintutorials
     Blog:          www.sysadmintutorials.com
     Twitter:       @systutorials
    ===========================================================================
    .MODS
        Made $StringInput Optional
        Added a $SpecialKeyInput - See PARAMETER SpecialKeyInput below
        Added description to write-hosts [SCRIPTINPUT] OR [SPECIALKEYINPUT]
    .PARAMETER StringInput
        The string of single characters to send to the VM
    .PARAMETER SpecialKeyInput
        All Function Keys i.e. F1 - F12
        Keyboard TAB, ESC, BACKSPACE, ENTER
        Keyboard Up, Down, Left Right
    .EXAMPLE
        Set-VMKeystrokes -VMName $VM -SpecialKeyInput "F2"
        Push SpecialKeyInput F2 to VM $VM
#>
    param(
        [Parameter(Mandatory = $true)][String]$VMName,
        [Parameter(Mandatory = $false)][String]$StringInput,
        [Parameter(Mandatory = $false)][String]$SpecialKeyInput,
        [Parameter(Mandatory = $false)][Boolean]$ReturnCarriage,
        [Parameter(Mandatory = $false)][Boolean]$DebugOn
    )

    # Map subset of USB HID keyboard scancodes
    # https://gist.github.com/MightyPork/6da26e382a7ad91b5496ee55fdc73db2
    $hidCharacterMap = @{
        "a"            = "0x04";
        "b"            = "0x05";
        "c"            = "0x06";
        "d"            = "0x07";
        "e"            = "0x08";
        "f"            = "0x09";
        "g"            = "0x0a";
        "h"            = "0x0b";
        "i"            = "0x0c";
        "j"            = "0x0d";
        "k"            = "0x0e";
        "l"            = "0x0f";
        "m"            = "0x10";
        "n"            = "0x11";
        "o"            = "0x12";
        "p"            = "0x13";
        "q"            = "0x14";
        "r"            = "0x15";
        "s"            = "0x16";
        "t"            = "0x17";
        "u"            = "0x18";
        "v"            = "0x19";
        "w"            = "0x1a";
        "x"            = "0x1b";
        "y"            = "0x1c";
        "z"            = "0x1d";
        "1"            = "0x1e";
        "2"            = "0x1f";
        "3"            = "0x20";
        "4"            = "0x21";
        "5"            = "0x22";
        "6"            = "0x23";
        "7"            = "0x24";
        "8"            = "0x25";
        "9"            = "0x26";
        "0"            = "0x27";
        "!"            = "0x1e";
        "@"            = "0x1f";
        "#"            = "0x20";
        "$"            = "0x21";
        "%"            = "0x22";
        "^"            = "0x23";
        "&"            = "0x24";
        "*"            = "0x25";
        "("            = "0x26";
        ")"            = "0x27";
        "_"            = "0x2d";
        "+"            = "0x2e";
        "{"            = "0x2f";
        "}"            = "0x30";
        "|"            = "0x31";
        ":"            = "0x33";
        "`""           = "0x34";
        "~"            = "0x35";
        "<"            = "0x36";
        ">"            = "0x37";
        "?"            = "0x38";
        "-"            = "0x2d";
        "="            = "0x2e";
        "["            = "0x2f";
        "]"            = "0x30";
        "\"            = "0x31";
        "`;"           = "0x33";
        "`'"           = "0x34";
        ","            = "0x36";
        "."            = "0x37";
        "/"            = "0x38";
        " "            = "0x2c";
        "F1"           = "0x3a";
        "F2"           = "0x3b";
        "F3"           = "0x3c";
        "F4"           = "0x3d";
        "F5"           = "0x3e";
        "F6"           = "0x3f";
        "F7"           = "0x40";
        "F8"           = "0x41";
        "F9"           = "0x42";
        "F10"          = "0x43";
        "F11"          = "0x44";
        "F12"          = "0x45";
        "TAB"          = "0x2b";
        "KeyUp"        = "0x52";
        "KeyDown"      = "0x51";
        "KeyLeft"      = "0x50";
        "KeyRight"     = "0x4f";
        "KeyESC"       = "0x29";
        "KeyBackSpace" = "0x2a";
        "KeyEnter"     = "0x28";
    }

    $vm = Get-View -ViewType VirtualMachine -Filter @{"Name" = "^$($VMName)$" }

    # Verify we have a VM or fail
    if (!$vm) {
        Write-host "Unable to find VM $VMName"
        return
    }

    #Code for -StringInput
    if ($StringInput) {
        $hidCodesEvents = @()
        foreach ($character in $StringInput.ToCharArray()) {
            # Check to see if we've mapped the character to HID code
            if ($hidCharacterMap.ContainsKey([string]$character)) {
                $hidCode = $hidCharacterMap[[string]$character]

                $tmp = New-Object VMware.Vim.UsbScanCodeSpecKeyEvent

                # Add leftShift modifer for capital letters and/or special characters
                if ( ($character -cmatch "[A-Z]") -or ($character -match "[!|@|#|$|%|^|&|(|)|_|+|{|}|||:|~|<|>|?|*]") ) {
                    $modifer = New-Object Vmware.Vim.UsbScanCodeSpecModifierType
                    $modifer.LeftShift = $true
                    $tmp.Modifiers = $modifer
                }

                # Convert to expected HID code format
                $hidCodeHexToInt = [Convert]::ToInt64($hidCode, "16")
                $hidCodeValue = ($hidCodeHexToInt -shl 16) -bor 0007

                $tmp.UsbHidCode = $hidCodeValue
                $hidCodesEvents += $tmp

                if ($DebugOn) {
                    Write-Host "[StringInput] Character: $character -> HIDCode: $hidCode -> HIDCodeValue: $hidCodeValue"
                }
            }
            else {
                Write-Host "[StringInput] The following character `"$character`" has not been mapped, you will need to manually process this character"
                break
            }

        }
    }

    #Code for -SpecialKeyInput
    if ($SpecialKeyInput) {
        if ($hidCharacterMap.ContainsKey([string]$SpecialKeyInput)) {
            $hidCode = $hidCharacterMap[[string]$SpecialKeyInput]
            $tmp = New-Object VMware.Vim.UsbScanCodeSpecKeyEvent
            $hidCodeHexToInt = [Convert]::ToInt64($hidCode, "16")
            $hidCodeValue = ($hidCodeHexToInt -shl 16) -bor 0007

            $tmp.UsbHidCode = $hidCodeValue
            $hidCodesEvents += $tmp

            if ($DebugOn) {
                Write-Host "[SpecialKeyInput] Character: $character -> HIDCode: $hidCode -> HIDCodeValue: $hidCodeValue"
            }
        }
        else {
            Write-Host "[SpecialKeyInput] The following character `"$character`" has not been mapped, you will need to manually process this character"
            break
        }
    }

    # Add return carriage to the end of the string input (useful for logins or executing commands)
    if ($ReturnCarriage) {
        # Convert return carriage to HID code format
        $hidCodeHexToInt = [Convert]::ToInt64("0x28", "16")
        $hidCodeValue = ($hidCodeHexToInt -shl 16) + 7

        $tmp = New-Object VMware.Vim.UsbScanCodeSpecKeyEvent
        $tmp.UsbHidCode = $hidCodeValue
        $hidCodesEvents += $tmp
    }

    # Call API to send keystrokes to VM
    $spec = New-Object Vmware.Vim.UsbScanCodeSpec
    $spec.KeyEvents = $hidCodesEvents
    Write-Host "Entering: $StringInput"
    $results = $vm.PutUsbScanCodes($spec)
}
# Timer function
Function Wait-Timer {
Param ($Seconds,$Activity)
1..$Seconds |
ForEach-Object { $percent = $_ * 100 / $Seconds;
Write-Progress -Activity "   $Activity" -Status "$($Seconds - $_) seconds remaining..." -PercentComplete $percent;
Start-Sleep -Seconds 1}
}
#####################################################################
# End of Functions
#####################################################################
#######################
# Importing Rubrik support credentials
#######################
# Setting credential file
$RubrikSupportCredentialsFile = $ScriptDirectory + "\RubrikSupportCredentials.xml"
# Testing if file exists
$RubrikSupportCredentialsFileTest =  Test-Path $RubrikSupportCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikSupportCredentialsFileTest -eq $False)
{
$RubrikSupportCredentials = Get-Credential -Message "Enter Rubrik Support login credentials"
$RubrikSupportCredentials | EXPORT-CLIXML $RubrikSupportCredentialsFile -Force
}
ELSE
{
# Importing credentials
$RubrikSupportCredentials = IMPORT-CLIXML $RubrikSupportCredentialsFile
}
#######################
# Installing then importing PowerCLI module
#######################
$PowerCLIModuleCheck = Get-Module -ListAvailable VMware.PowerCLI
IF ($PowerCLIModuleCheck -eq $null)
{
Install-Module -Name VMware.PowerCLI -Scope CurrentUser -Confirm:$false -AllowClobber
Import-Module VMware.PowerCLI | Out-Null
Set-PowerCLIConfiguration -Scope User -ParticipateInCEIP $true
}
ELSE
{
# Importing PowerCLI
Import-Module VMware.PowerCLI | Out-Null
}
# Ignoring certs if not already configured
$PowerCLIConfig = Get-PowerCLIConfiguration | Where {$_.Scope -eq "Session"} | Select -ExpandProperty InvalidCertificateAction
IF ($PowerCLIConfig -ne "Ignore"){Set-PowerCLIConfiguration -InvalidCertificateAction Ignore}
#######################
# Starting Process (can change this to a for each if deploying multiple)
#######################
$ScriptStart = Get-Date
$ScriptResults = [System.Collections.ArrayList]@()
##############################################
# Getting vCenters From CSV & Processing For Each
##############################################
$UniquevCenters = $CSVImport | Select -ExpandProperty vCenter -Unique
# For Each
ForEach ($vCenter in $UniquevCenters)
{
#######################
# Importing vCenter credentials
#######################
# Setting credential file
$vCenterCredentialsFile = $ScriptDirectory + "\" + $vCenter + "-Credentials.xml"
# Testing if file exists
$vCenterCredentialsFileTest =  Test-Path $vCenterCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($vCenterCredentialsFileTest -eq $False)
{
$vCenterCredentials = Get-Credential -Message "Enter vCenter login credentials for $vCenter"
$vCenterCredentials | EXPORT-CLIXML $vCenterCredentialsFile -Force
}
ELSE
{
# Importing credentials
$vCenterCredentials = IMPORT-CLIXML $vCenterCredentialsFile
}
# Getting user
$vCenterUser = $vCenterCredentials.UserName
$vCenterPassword = $vCenterCredentials.GetNetworkCredential().Password
#######################
# 0. Connecting to vCenter 
#######################
# Logging
Write-Host "------------------------------------------------------------
0. Connecting to vCenter using PowerCLI Connect-VIServer
------------------------------------------------------------
vCenter: $vCenter"
# Connecting to the vCenter
Try
{
Connect-VIServer -Server $vCenter -Credential $vCenterCredentials | Out-Null
}
Catch
{
$OVFDeployed = $FALSE
}
##############################################
# 1. Selecting OVFs Then Deploying
##############################################
$OVFsToDeploy = $CSVImport | Where {$_.vCenter -eq $vCenter}
$OVFsToDeployCount = $OVFsToDeploy | Measure | Select -ExpandProperty Count
$OVFsToDeployCounter = 0
#######################
# For Each OVF
#######################
# Deploying each OVF
ForEach ($OVFToDeploy in $OVFsToDeploy)
{
# Incrementing counter
$OVFsToDeployCounter ++
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 1. Deploying OVF using PowerCLI Import-VApp 
------------------------------------------------------------"
# vCenter Settings
$ESXiHost = $OVFToDeploy.ESXiHost
$Datastore = $OVFToDeploy.Datastore
$VMFolder = $OVFToDeploy.VMFolder
# OVF settings
$OVFVMNIC = $OVFToDeploy.VMNIC
$OVFVMName = $OVFToDeploy.VMName
$OVFDataDiskGB = $OVFToDeploy.DataDiskGB
# Rubrik bootstrap settings
$RubrikEnableEncrypton = $OVFToDeploy.EnableEncryption
$RubrikEmail = $OVFToDeploy.DefaultEmail
$RubrikClusterName = $OVFToDeploy.RubrikClusterName
$RubrikDNSServers = $OVFToDeploy.DNSServers
$RubrikDNSearchDomains = $OVFToDeploy.DNSSearchDomains
$RubrikNTPServers = $OVFToDeploy.NTPServers
$RubrikGateway = $OVFToDeploy.DefaultGateway
$RubrikSubnet = $OVFToDeploy.Subnet
$RubrikIPAddress = $OVFToDeploy.IPAddress
# Cluster settings
$TimeZone = $OVFToDeploy.Timezone
$GeoLocation = $OVFToDeploy.Location
# SMTP settings
$SMTPFrom = $OVFToDeploy.DefaultEmail
#######################
# Deciding Which OVF & setting data disk size
#######################
IF ([int]$OVFDataDiskGB -gt [int]$UseAIRAboveGB)
{
$OVFImage = $AIROVF
}
ELSE
{
$OVFImage = $EDGEOVF
}
#######################
# Configuring OVF 
#######################
$OVFConfig = Get-OvfConfiguration -Ovf $OVFImage
# Configuring settings
$OVFConfig.IpAssignment.IpAllocationPolicy.Value = "fixedPolicy"
$OVFConfig.IpAssignment.IpProtocol.Value = "IPv4"
$OVFConfig.NetworkMapping.VM_Network.Value = "$OVFVMNIC"
# Benching deploy
$OVFDeployStart = Get-Date
# Logging
Write-Host "OVFToDeploy: $OVFImage
VMName: $OVFVMName
ESXiHost: $ESXiHost
Datastore: $Datastore
IPAddress: $RubrikIPAddress
Subnet: $RubrikSubnet
Gateway: $RubrikGateway"
# Deploying OVF
Try
{
Import-VApp -Source $OVFImage -Force -OvfConfiguration $OVFConfig -Name $OVFVMName -VMHost $ESXiHost -Datastore $Datastore -DiskStorageFormat "Thin" -Confirm:$false
$OVFDeployed = $TRUE
}
Catch
{
$OVFDeployed = $FALSE
}
# Logging
Write-Host "OVFDeployed: $OVFDeployed"
# Benching
$OVFDeployEnd = Get-Date
# Time taken
$OVFDeployTimespan = New-Timespan -Start $OVFDeployStart -End $OVFDeployEnd
$OVFDeploySeconds = $OVFDeployTimespan | Select -ExpandProperty TotalSeconds
$OVFDeployTimeTaken = "{0:g}" -f $OVFDeployTimespan
IF ($OVFDeployTimeTaken -match "."){$OVFDeployTimeTaken = $OVFDeployTimeTaken.Split(".") | Select -First 1}
# Calc throughput
IF (($OVFDeploySeconds -gt 0) -and ($OVFDeployed -eq $TRUE))
{
$OVFDeployThroughputMBSec = $OVFSizeMB / $OVFDeploySeconds
$OVFDeployThroughputMBSec = [Math]::Round($OVFDeployThroughputMBSec,2)
}
ELSE
{
$OVFDeployThroughputMBSec = $null
}
# Logging
Write-Host "OVFDeployTimeTaken: $OVFDeployTimeTaken
OVFDeployThroughoutMBSec: $OVFDeployThroughputMBSec"
# Waiting 10 seconds before trying re-size
Sleep 10
#######################
# Step 2. Changing Data Disk Size
#######################
# Only re-sizing if more than the default
IF ([int]$OVFDataDiskGB -gt 1024)
{
Try
{
Get-VM $OVFVMName | Get-HardDisk | Where {$_.Name -eq "Hard disk 1"} | Set-HardDisk -CapacityGB $OVFDataDiskGB -Confirm:$false | Out-Null
$OVFResized = $TRUE
}
Catch
{
$OVFResized = $FALSE
}
# Changing VM spec if above 10TB
IF ([int]$OVFDataDiskGB -gt 10240)
{
Get-VM $OVFVMName | Set-VM -NumCpu 8 -MemoryGB​​ 32 -Confirm:$false | Out-Null
}
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 2. Resizing Data Disk & VM CPU/RAM (if required) using PowerCLI Set-HardDisk & Set-VM
------------------------------------------------------------
OVFResized: $OVFResized"
}
ELSE
{
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 2. Skipping re-size, as size is equal to or below default, will be thin provisioned as 1024GB no matter what
------------------------------------------------------------"
}
#######################
# Step 3.1 Moving VM Folder
#######################
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 3.1 Moving VM Folder using PowerCLI Move-VM
------------------------------------------------------------"
Try
{
# Move-VM -VM $OVFVMName -Destination $VMFolder | Out-Null
Get-VM $OVFVMName | Move-Inventory -Destination $VMFolder | Out-Null
$OVFMovedToFolder = $TRUE
}
Catch
{
$OVFMovedToFolder = $FALSE
}
#######################
# Step 3.2 Powering on VM & Waiting For Boot
#######################
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 3.2 Powering On VM using PowerCLI Start-VM
------------------------------------------------------------"
# Power on
Try
{
Start-VM $OVFVMName -Confirm:$false | Out-Null
$OVFPoweredOn = $TRUE
}
Catch
{
$OVFPoweredOn = $FALSE
}
# Logging
Write-Host "OVFPoweredOn: $OVFPoweredOn"
# Logging
Wait-Timer -Seconds $OVFBootDelaySecs -Activity "Waiting for VM to boot and services start"
#######################
# Step 4. Logging Into VM & Generating Passwords
#######################
IF ($OVFPoweredOn -eq $TRUE)
{
# Logging 
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 4. Logging Into VM using Set-VMKeyStrokes
------------------------------------------------------------"
# Entering
Set-VMKeyStrokes -VM $OVFVMName -StringInput $OVFUser -ReturnCarriage $True; Sleep 3
# Password, have to enter it twice for some strange reason on mine
Set-VMKeyStrokes -VM $OVFVMName -StringInput $OVFPassword -ReturnCarriage $True; Sleep 5
Set-VMKeyStrokes -VM $OVFVMName -StringInput $OVFPassword -ReturnCarriage $True; Sleep 20
# Generating New admin password
$RubrikPassword = [System.Web.Security.Membership]::GeneratePassword(30, 3)
# Generating Encryption password, if required
IF ($RubrikEnableEncrypton -eq $TRUE)
{
$RubrikEncryptionPassword = [System.Web.Security.Membership]::GeneratePassword(30, 3)
}
ELSE
{
$RubrikEncryptionPassword = "n/a"
}
#######################
# Step 5. Bootstrapping
#######################
# Logging 
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 5. Bootstrapping VM using Set-VMKeyStrokes
------------------------------------------------------------"
# 5.1 Start bootstrap
Set-VMKeyStrokes -VM $OVFVMName -StringInput "cluster bootstrap" -ReturnCarriage $True
# Logging
Wait-Timer -Seconds $RubrikPreBootstrapDelaySecs -Activity "Waiting for bootstrap service to become available"
# 5.2 Email & Password
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikEmail -ReturnCarriage $True; Sleep 3
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikPassword -ReturnCarriage $True; Sleep 3
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikPassword -ReturnCarriage $True; Sleep 3
# 5.3 Cluster Name
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikClusterName -ReturnCarriage $True; Sleep 3
# 5.4 DNS Settings 
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikDNSServers -ReturnCarriage $True; Sleep 3
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikDNSearchDomains -ReturnCarriage $True; Sleep 3
# 5.5 NTP config
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikNTPServers -ReturnCarriage $True; Sleep 3
# 5.6 IP Gateway
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikGateway -ReturnCarriage $True; Sleep 3
# 5.7 IP Subnet
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikSubnet -ReturnCarriage $True; Sleep 3
# 5.8 Bypass VLAN
Set-VMKeyStrokes -VM $OVFVMName -StringInput "" -ReturnCarriage $True; Sleep 3
# 5.9 Enable Software Encryption, and 2nd confirmation prompt, random password twice, if enabled
IF ($RubrikEnableEncrypton -eq $TRUE)
{
Set-VMKeyStrokes -VM $OVFVMName -StringInput "y" -ReturnCarriage $True; Sleep 3
Set-VMKeyStrokes -VM $OVFVMName -StringInput "y" -ReturnCarriage $True; Sleep 3
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikEncryptionPassword -ReturnCarriage $True; Sleep 3
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikEncryptionPassword -ReturnCarriage $True; Sleep 3
}
ELSE
{
Set-VMKeyStrokes -VM $OVFVMName -StringInput "n" -ReturnCarriage $True; Sleep 3
}
# 5.10 IP Address
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikIPAddress -ReturnCarriage $True; Sleep 3
# 5.11 Proceed with bootstrap
Set-VMKeyStrokes -VM $OVFVMName -StringInput "y" -ReturnCarriage $True; Sleep 3
# Logging
Wait-Timer -Seconds $RubrikPostBootstrapDelaySecs -Activity "Waiting for bootstrap to complete"
#######################
# Step 6. Registering Cluster with Polaris
#######################
# Logging 
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 6. Registering VM With Polaris using Set-VMKeyStrokes
------------------------------------------------------------"
# 6.1 Selecting method 1
Set-VMKeyStrokes -VM $OVFVMName -StringInput "1" -ReturnCarriage $True; Sleep 3
# 6.2 Entering username
$RubrikSupportUser = $RubrikSupportCredentials.UserName
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikSupportUser -ReturnCarriage $True; Sleep 3
# 6.3 Entering password
$RubrikSupportPassword = $RubrikSupportCredentials.GetNetworkCredential().Password 
Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikSupportPassword -ReturnCarriage $True; Sleep 20
# 6.3 Entering password
# $RubrikSupportPassword = $RubrikSupportCredentials.GetNetworkCredential().Password 
# Set-VMKeyStrokes -VM $OVFVMName -StringInput $RubrikSupportPassword -ReturnCarriage $True; Sleep 3
#######################
# Step 7. Testing Authentication & Bootstrap
#######################
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 7. Validating Bootstrap & Testing Auth using Invoke-RestMethod to Rubrik REST APIs
------------------------------------------------------------"
# Building URLs
$v1BaseURL = "https://" + $RubrikIPAddress + "/api/v1/"
$v2BaseURL = "https://" + $RubrikIPAddress + "/api/v1/"
$InternalURL = "https://" + $RubrikIPAddress + "/api/internal/"
$ClusterURL = "https://" + $RubrikIPAddress + "/web/bin/index.html#/dashboard"
$RubrikSessionURL = $v1BaseURL + "session"
# Building header & setting data type
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($OVFUser+":"+$RubrikPassword))}
$Type = "application/json"
# Testing bootstrap
Try
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $ClusterURL -Method GET -ContentType $Type
$RubrikBootstrapped = $TRUE
}
Catch
{
$RubrikBootstrapped = $FALSE
}
# Testing authentication
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
$RubrikAuthenticated = $TRUE
}
Catch 
{
$RubrikAuthenticated = $FALSE
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
# Logging
Write-Host "RubrikBootstrapped: $RubrikBootstrapped
RubrikAuthenticated: $RubrikAuthenticated"
###############################################
# Step 8.1 - Configuring Cluster Location & Timezone
###############################################
$ClusterSettingsApplied = 0
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 8.1 Configuring Cluster Location & TimeZone using Invoke-RestMethod to Rubrik REST APIs
------------------------------------------------------------
TimeZone: $TimeZone
Location: $GeoLocation"
# Building JSON
$ClusterSettingJSON =
"{
  ""timezone"": {
        ""timezone"": ""$TimeZone""
    },
  ""geolocation"": {
        ""address"": ""$GeoLocation""
    }
}"
# Building URL
$ClusterSettingURL = $v1BaseURL+"cluster/me"
# PATCH to URL
Try 
{
$ClusterSetting = Invoke-RestMethod -Method PATCH -Uri $ClusterSettingURL -Body $ClusterSettingJSON -Headers $RubrikSessionHeader -ContentType $Type
$ClusterSettingsApplied = $TRUE
}
Catch 
{
$ClusterSettingsApplied = $FALSE
}
###############################################
# Step 8.2 - Removing Default SLAs
###############################################
IF ($RemoveDefaultSLAs -eq $True)
{
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 8.2 Removing Default SLA Domains using Invoke-RestMethod to Rubrik REST APIs
------------------------------------------------------------"
# Getting SLAs
$ExistingSLADomainsURL = $v2BaseURL+"sla_domain"
# GET to URL
Try 
{
$ExistingSLADomainsJSON = Invoke-RestMethod -Method GET -Uri $ExistingSLADomainsURL -Headers $RubrikSessionHeader -ContentType $Type
$ExistingSLADomains = $ExistingSLADomainsJSON.data
}
Catch 
{
$SLADomainsDeleted = $FALSE
}
# Filtering names just in case this is run against a production Rubrik as you wouldn't want it to delete everything!
$ExistingSLADomains = $ExistingSLADomains | Where-Object {(($_.name -eq "Gold") -or ($_.name -eq "Silver") -or ($_.name -eq "Bronze") -or ($_.name -eq "DeleteMe"))}
# Deleting SLAs
ForEach ($ExistingSLADomain in $ExistingSLADomains)
{
# Setting variable
$SLADomainID = $ExistingSLADomain.id
$SLADomainName = $ExistingSLADomain.name
# Host output
"DeletingSLA: $SLADomainName"
# Building URL
$SLADomainDeleteURL = $v2BaseURL+"sla_domain/"+$SLADomainID
# DELETE to URL
Try 
{
$SLADomainDelete = Invoke-RestMethod -Method DELETE -Uri $SLADomainDeleteURL -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainsDeleted = $TRUE
}
Catch 
{
$SLADomainsDeleted = $FALSE
}
# End of for each SLA delete action below
}
# End of for each SLA delete action above
#
# End of IF $RemoveDefaultSLAs -eq $True below
}
ELSE
{
$SLADomainsDeleted = $FALSE
}
# End of IF $RemoveDefaultSLAs -eq $True above
###############################################
# Step 8.3 - Adding vCenter
###############################################
IF ($AddvCenter -eq $TRUE)
{
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 8.3 Adding vCenter using Invoke-RestMethod to Rubrik REST APIs
------------------------------------------------------------
vCenter: $vCenter
User: $vCenterUser"
# Building JSON
$AddvCenterJSON =
"{
  ""hostname"": ""$vCenter"",
  ""username"": ""$vCenterUser"",
  ""password"": ""$vCenterPassword"",
  ""conflictResolutionAuthz"": ""AllowAutoConflictResolution""
}"
# Building URL
$AddvCenterURL = $v1BaseURL+"vmware/vcenter"
# PATCH to URL
Try 
{
$AddvCenter = Invoke-RestMethod -Method POST -Uri $AddvCenterURL -Body $AddvCenterJSON -Headers $RubrikSessionHeader -ContentType $Type
$vCenterAdded = $TRUE
}
Catch 
{
$vCenterAdded = $FALSE
}
# End of adding vCenter below
}
ELSE
{
$vCenterAdded = $FALSE
}
# End of adding vCenter above
###############################################
# Step 8.4 - Configuring Login Banner
###############################################
IF ($AddLoginBanner -eq $TRUE)
{
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 8.4 Setting Login Banner using Invoke-RestMethod to Rubrik REST APIs
------------------------------------------------------------
Banner: $LoginBannerText"
# Creating JSON
$LoginBannerJSON =
"{
  ""loginBanner"": ""$LoginBannerText""
}"
# Building URL
$LoginBannerURL = $InternalURL+"cluster/me/login_banner"
# PUT to URL
Try 
{
$LoginBanner = Invoke-RestMethod -Method PUT -Uri $LoginBannerURL -Body $LoginBannerJSON -Headers $RubrikSessionHeader -ContentType $Type
$LoginBannerSet = $TRUE
}
Catch 
{
$LoginBannerSet = $FALSE
}
# End of adding login banner below
}
ELSE
{
$LoginBannerSet = $FALSE
}
###############################################
# Step 8.5 - Configuring SMTP Settings
###############################################
IF ($AddSMTPServer -eq $TRUE)
{
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - 8.5 Configuring SMTP Settings using Invoke-RestMethod to Rubrik REST APIs
------------------------------------------------------------
SMTPServer: $SMTPServer
Port: $SMTPPort
From: $SMTPFrom"
# Building JSON
$SMTPSettingJSON =
"{
  ""smtpHostname"": ""$SMTPServer"",
  ""smtpPort"": $SMTPPort,
  ""smtpSecurity"": ""$SMTPEncryption"",
  ""fromEmailId"": ""$SMTPFrom""
}"
# Building URL
$SMTPSettingURL = $InternalURL+"smtp_instance"
# PUT to URL
Try 
{
$SMTPSetting = Invoke-RestMethod -Method POST -Uri $SMTPSettingURL -Body $SMTPSettingJSON -Headers $RubrikSessionHeader -ContentType $Type
$SMTPServerSet = $TRUE
}
Catch 
{
$SMTPServerSet = $FALSE
}
}
ELSE
{
$SMTPServerSet = $FALSE
}
# End of bypass for not powered on below
}
# End of bypass for not powered on above
#######################
# Finishing Script
#######################
$ScriptEnd = Get-Date
# Time taken
$ScriptTimespan = New-Timespan -Start $ScriptStart -End $ScriptEnd
$ScriptTimeMinutes = $ScriptTimespan | Select -ExpandProperty TotalMinutes
$ScriptTimeMinutes = [Math]::Round($ScriptTimeMinutes,2)
$ScriptTimeTaken = "{0:g}" -f $ScriptTimespan
IF ($ScriptTimeTaken -match "."){$ScriptTimeTaken = $ScriptTimeTaken.Split(".") | Select -First 1}
# Logging
Write-Host "------------------------------------------------------------
OVF:$OVFsToDeployCounter/$OVFsToDeployCount - Finished OVF Deployment & Bootstrap
------------------------------------------------------------
TimeTaken: $ScriptTimeTaken"
#######################
# Summarizing Result
#######################
$Object = New-Object PSObject
# OVF Settings and deployment
$Object | Add-Member -MemberType NoteProperty -Name "VM" -Value $OVFVMName
$Object | Add-Member -MemberType NoteProperty -Name "ESXiHost" -Value $ESXiHost
$Object | Add-Member -MemberType NoteProperty -Name "Datastore" -Value $Datastore
$Object | Add-Member -MemberType NoteProperty -Name "OVF" -Value $OVFImage
$Object | Add-Member -MemberType NoteProperty -Name "Deployed" -Value $OVFDeployed
$Object | Add-Member -MemberType NoteProperty -Name "DataDiskResized" -Value $OVFResized
$Object | Add-Member -MemberType NoteProperty -Name "MovedToFolder" -Value $OVFMovedToFolder
$Object | Add-Member -MemberType NoteProperty -Name "PoweredOn" -Value $OVFPoweredOn
$Object | Add-Member -MemberType NoteProperty -Name "Bootstrapped" -Value $RubrikBootstrapped
$Object | Add-Member -MemberType NoteProperty -Name "Authenticated" -Value $RubrikAuthenticated
# Total runtime
$Object | Add-Member -MemberType NoteProperty -Name "Start" -Value $ScriptStart
$Object | Add-Member -MemberType NoteProperty -Name "End" -Value $ScriptEnd
$Object | Add-Member -MemberType NoteProperty -Name "TimeTaken" -Value $ScriptTimeTaken
$Object | Add-Member -MemberType NoteProperty -Name "TimeTakenMins" -Value $ScriptTimeMinutes
# Cluster settings
$Object | Add-Member -MemberType NoteProperty -Name "SettingsApplied" -Value $ClusterSettingsApplied
$Object | Add-Member -MemberType NoteProperty -Name "SLADomainsDeleted" -Value $SLADomainsDeleted
$Object | Add-Member -MemberType NoteProperty -Name "vCenterAdded" -Value $vCenterAdded
$Object | Add-Member -MemberType NoteProperty -Name "LoginBanner" -Value $LoginBannerSet
$Object | Add-Member -MemberType NoteProperty -Name "SMTPServer" -Value $SMTPServerSet
# Deploy stats
$Object | Add-Member -MemberType NoteProperty -Name "DeployTimeTaken" -Value $OVFDeployTimeTaken
$Object | Add-Member -MemberType NoteProperty -Name "DeployThroughputMBSec" -Value $OVFDeployThroughputMBSec
$Object | Add-Member -MemberType NoteProperty -Name "DataDiskGB" -Value $OVFDataDiskGB
# Bootstrap settings
$Object | Add-Member -MemberType NoteProperty -Name "DefaultEmail" -Value $RubrikEmail
$Object | Add-Member -MemberType NoteProperty -Name "DNSServers" -Value $RubrikDNSServers
$Object | Add-Member -MemberType NoteProperty -Name "DNSSearchDomains" -Value $RubrikDNSearchDomains
$Object | Add-Member -MemberType NoteProperty -Name "NTPServers" -Value $RubrikNTPServers
$Object | Add-Member -MemberType NoteProperty -Name "Gateway" -Value $RubrikGateway
$Object | Add-Member -MemberType NoteProperty -Name "Subnet" -Value $RubrikSubnet
$Object | Add-Member -MemberType NoteProperty -Name "IPAddress" -Value $RubrikIPAddress
$Object | Add-Member -MemberType NoteProperty -Name "AdminPassword" -Value $RubrikPassword
$Object | Add-Member -MemberType NoteProperty -Name "EncryptionPassword" -Value $RubrikEncryptionPassword
$Object | Add-Member -MemberType NoteProperty -Name "SupportUser" -Value $RubrikSupportUser
# Adding object
$ScriptResults.Add($Object) | Out-Null
##############################################
# End of for each OVF below
##############################################
}
# End of for each OVF above
##############################################
# End of for each vCenter below - Disconnecting from PowerCLI session
##############################################
Disconnect-VIServer * -Confirm:$false
}
# End of for each vCenter above

##################################
# Creating CSVs
##################################
# Creating the file names
$ScriptCSVFile = $LogDirectory + "\RubrikOVFDeployerLog-" + $ScriptStart.ToString("yyyy-MM-dd") + "@" + $ScriptStart.ToString("HH-mm-ss") + ".csv"
# Exporting to CSV
$ScriptResults | Export-Csv -Path $ScriptCSVFile -NoTypeInformation -Force
# Logging
"------------------------------------------------------------
CreatingCSV: $ScriptCSVFile
------------------------------------------------------------"
#######################
# Showing Result
#######################
$ScriptResults | Out-GridView
#######################
# End of script
#######################