﻿##################################
# Idera Geek Sync Webinar - PowerShell Top 10 Tips & Tricks (New & Improved Examples)
# Created by Joshua Stenhouse @ virtuallysober.com - Check out my blog to find more scripts!
# Last Updated 09/28/18
##################################
# Tip 1 - General Advice - Keep your scripts as simple as possible, iterate & improve, unit test as you go along, use the right rools, build your own library of scripts with versioning.
# Tip 2 - Use comments like these, split up your scripts into sections like the below, explain complex elements in detail, consider an index in larger scripts.
##################################
# Tip 3 - Variables - Use them at the start of your script as your building blocks, to store results, and don't overwrite variables, better to use new.
##################################
$Variable1 = 1
$Variable2 = 2
$Variable3 = $Variable1 + $Variable2
"Result: $Variable3"
##################################
# Tip 4 - Logging - Use to catch errors in headless/scheduled scripts, good for auditors/compliance, not always needed for short scripts. 
##################################
$Now = Get-Date
$Log = "C:\Test\Log" + $Now.ToString(" yyyy-MM-dd_HH-mm-ss") + ".log"
Start-Transcript-Path $Log
# Your script here
"Hello World"
Stop-Transcript
##################################
# Tip 5 - Try & Catch - Use to gracefully handle errors in scripts, improve resilience, nest to try again.
##################################
# Run the below to see an error not handled gracefully
$Host = "Test"
# Now try the same command inside a Try & Catch by selecting lines 27 to 37. See the difference? Change $Host on line 30 to $Host1 and watch it succeed.
Try
{
$TrySuccess = $True
$Host = "Test"
}
Catch
{
"Don't try to overwrite the host variable dummy!"
$TrySuccess = $False
}
"TrySuccess:$TrySuccess"
# Now try the below and see the full error, combine with the transcript logging in your scripts for extra power!
Try
{
$TrySuccess = $True
$Host = "Test"
}
Catch
{
"Don't try to overwrite the host variable dummy!"
$TrySuccess = $False
$_.Exception.ToString()
$error[0] | Format-List -Force
}
"TrySuccess:$TrySuccess" 
##################################
# Tip 6 - Use CSVs - For importing and exporting data nothing more powerful, universally accepted file format, make your scripts user and machine friendly
##################################
$CSVFileName = "C:\Test\ProcessList.csv"
$ProcessList = Get-Process
$ProcessList | Export-CSV $CSVFileName -NoTypeInformation
$ProcessCSV = Import-CSV$CSVFileName 
$ProcessCSV | Format-Table
##################################
# Tip 7 - Conditional IF statements - Allows decision making, bypass actions, nest multiple for even more power, but don't go too crazy, use comments at the end of closing brackets on big IF statements to make editing/troubleshooting easier.
##################################
$Variable1 = 1
IF ($Variable1 –eq 1)
{
"Yes"
}
ELSE
{
"No"
}
##################################
# Tip 8 - ForEach - Repeat blocks of script for multiple objects. Processed serially, use sleep x to pause between if needed, combine with CSVs for power.
##################################
$Servers = “Server1”,”Server2”,”Server3”,”Server4”
ForEach ($Server in $Servers)
{
“Running script for Server: $Server”
}
# If you've run section 56-60 already try the below to give you an idea of combining Import-CSV, ForEach, and IF
ForEach ($Process in $ProcessCSV)
{
$ProcessName = $Process.Name
$ProcessPath = $Process.Path
"ProcessName:$ProcessName"
If ($ProcessPath -ne "")
{
"ProcessPath:$ProcessPath"
}
}
##################################
# Tip 9 - Use PowerShell Modules - Pre-biuilt to extend functionality, save time, over 3000 available
##################################
$AllModules = Find-Module | Sort-Object Name
"ShowingModules:"
$AllModules | Format-Table
"TotalModules:"
$AllModules.count
# Install, import, then show all commands from the SqlServer PowerShell module
Install-Module -Name SqlServer -Scope CurrentUser -Confirm:$false
Import-Module SqlServer
Get-Command -Module SqlServer | Format-Table
##################################
# Tip 10 - Use REST APIs - Open up your world to interacting with any REST API, not limtied to pre-build modules, diff APIs require diff auth header structures, look for documentation or examples online.
##################################
# This is an example using the Google maps API key, I beleive its still free for 25k requests per day but annoyingly you now have to enter billing info
# To get your key go to https://developers.google.com/maps/documentation/geocoding/get-api-key and select places
# I recommend restricting your API key usage to your IP address and I used my hometown of Todmorden UK as an example lookup
$GoogleMapsAPIKey = "yourgooglemapsAPIkeyhere"
$City = "Todmorden"
# Building URI (a URL is a type of URI)
$GoogleMapQueryURI = "https://maps.googleapis.com/maps/api/geocode/json?address=" + $City + "&key=" + $GoogleMapsAPIKey
# Invoking the REST API call
Try 
{
$GoogleMapsResponse = Invoke-RestMethod -Method GET -Uri $GoogleMapQueryURI -ContentType "application/json"
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# Extracting the data from the response
$GoogleMapResults = $GoogleMapsResponse.results
$Location = $GoogleMapResults.address_components.long_name | Select -Skip 1
# Showing result
"The town/city $City is in:"
$Location
##################################
# Bonus Tip 11 - Create arrays (tables) - Store and combine results from multiple sources, use for reporting, output to CSV etc. I use these all the time!
##################################
# Creating table object
$Table = @()
# Showing you an example within a ForEach loop
$Rows = "Row1","Row2","Row3"
ForEach ($Row in $Rows)
{
# Adding Row
$TableRow = New-Object PSObject
$TableRow | Add-Member -MemberType NoteProperty -Name "Name" -Value "$Row"
$TableRow | Add-Member -MemberType NoteProperty -Name "DataField" -Value "100"
$Table += $TableRow
}
# Showing table after ForEach section
$Table | Sort-Object Name | Format-Table
# Adding the values together
$TotalDataField = $Table | Select -ExpandProperty DataField | Measure -Sum | Select -ExpandProperty Sum
$TotalDataField
##################################
# End of script
##################################