﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script automatically verifies each Managed Volume snapshot is within the Business SLA you define. 
# Ensure the ScriptDirectory is configured to the correct location as all the settings are derived from RubrikAutoTicketingv1-Settings.ps1 in it.
# Run the RubrikAutoTicketingv1-Auth.ps1 script first to create the credential files needed.
# It can be configured with the settings script to automatically email reports, auto remediate objects with on-demand snapshots, and create helpdesk tickets with a consolidated failure report.
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Configure the $ScriptDirectory to a location which contains the below files
# - Before running Configure RubrikAutoTicketingv1-Settings.ps1
# - Before running execute RubrikAutoTicketingv1-Auth.ps1 to store your Rubrik and SMTP server creds
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##################################
# Global settings file location
##################################
$ScriptDirectory = "C:\RubrikAutoTicketingv1\"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
$GlobalSettingsFile = $ScriptDirectory + "RubrikAutoTicketingv1-Settings.ps1"
$SettingsTest = Test-Path $GlobalSettingsFile 
IF ($SettingsTest -eq $FALSE)
{
"No RubrikAutoTicketingv1-Settings.ps1 found in directory:" 
$ScriptDirectory
"Verify the ScriptDirectory variable at the start of each script is configured to the correct directory containing the settings and credential files"
sleep 10
# Terminating script as no valid settings file found
kill $pid
}
##################################
# Global settings import of the variables required
##################################
. $GlobalSettingsFile
##################################
# Importing Rubrik credentials
##################################
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##########################
# Building Rubrik API string & invoking REST API
##########################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##########################
# Calculating timezone of machine running script
##########################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##################################
# Getting list of SLA Domains
##################################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of Managed Volumes
##################################
$MVListURL = $InternalURL+"managed_volume"
Try 
{
$MVListJSON = Invoke-RestMethod -Uri $MVListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$MVList = $MVListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# If $MVList is null, likely that the JSON was too big for the default 2MB MaxJsonLength, fixing with JavaScriptSerializer
IF ($MVList -eq $null)
{
$MVListJSONSerialized = ParseItem ((New-Object -TypeName System.Web.Script.Serialization.JavaScriptSerializer -Property @{MaxJsonLength=67108864}).DeserializeObject($MVListJSON))
$MVList = $MVListJSONSerialized.data
}
##########################
# Building list of Protected Managed Volumes
##################################
$ProtectedMVList = $MVList | Where-Object {($_.effectiveSlaDomainName -ne "UNPROTECTED") -and ($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.isRelic -ne "True")}
##################################
# Building Report per VM
##################################
# Creating array to store report info
$MVComplianceReport=@()
# Performing per VM action
ForEach ($MV in $ProtectedMVList)
{
# Setting variables required
$MVName = $MV.name
$MVID = $MV.id
$MVSLADomain = $MV.effectiveSlaDomainName
$MVSLADomainID = $MV.effectiveSlaDomainId
$MVUsedSizeBytes = $MV.usedSize
$MVVolumeSizeBytes = $MV.volumeSize
##########################
# Converting Bytes to GB
##########################
$MVVolumeSizeGB = $MVVolumeSizeBytes / 1000 / 1000 / 1000
$MVUsedSizeGB = $MVUsedSizeBytes / 1000 / 1000 / 1000
$MVVolumeSizeGB = [Math]::Round($MVVolumeSizeGB, 0)
$MVUsedSizeGB = [Math]::Round($MVUsedSizeGB, 1)
##########################
# Getting VM detail
##########################
$MVInfoURL = $InternalURL+"managed_volume/"+$MVID
Try 
{
$MVInfo = Invoke-RestMethod -Uri $MVInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling VM detail
$MVSnapshotCount = $MVInfo.snapshotCount
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($MVSnapshotCount -eq 0)
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
# Resetting values to ensure data isn't carried over between rows
$MVLatestSnapshotAdjusted = $null
$SnapshotAdjustedGapInHours = $null
}
ELSE
{
# Snapshot exists, so performing actions
##########################
# Getting VM snapshot info
##########################
$MVSnapshotURL = $InternalURL+"managed_volume/"+$MVID+"/snapshot"
Try 
{
$MVSnapshotJSON = Invoke-RestMethod -Uri $MVSnapshotURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$MVSnapshot = $MVSnapshotJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
##################################
# Selecting snapshots
##################################
$MVLatestSnapshot1 = $MVSnapshot | Sort-Object -Descending date | select -ExpandProperty date -First 1
$MVOldestSnapshot1 = $MVSnapshot | Sort-Object date | select -ExpandProperty date -First 1
##################################
# Converting Latest Snapshot
##################################
# Step 1 - Removing characters and trimming snapshot string for conversion
$MVLatestSnapshot2 = $MVLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$MVLatestCharCount = $MVLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$MVLatestCharSubtract = $MVLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$MVLatestSnapshot3 = $MVLatestSnapshot2.Substring(0,$MVLatestSnapshot2.Length-$MVLatestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$MVLatestSnapshot = ([datetime]::ParseExact($MVLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$MVOldestSnapshot2 = $MVOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$MVOldestCharCount = $MVOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$MVOldestCharSubtract = $MVOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$MVOldestSnapshot3 = $MVOldestSnapshot2.Substring(0,$MVOldestSnapshot2.Length-$MVOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$MVOldestSnapshot = ([datetime]::ParseExact($MVOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Calculating SLA compliance
##########################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $MVLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$MVLatestSnapshotAdjusted = $MVLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $MVLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
# End of ELSE action to IF snapshot count equals 0 below
}
# End of ELSE action to IF snapshot count equals 0 above
##########################
# Summarizing Managed Volume info into report
##########################
$MVComplianceReportLine = new-object PSObject
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$MVSLADomain"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "ManagedVolume" -Value "$MVName"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastBackup" -Value "$MVLatestSnapshotAdjusted"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "VolumeSizeGB" -Value "$MVVolumeSizeGB"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "VolumeUsedGB" -Value "$MVUsedSizeGB"
$MVComplianceReportLine | Add-Member -MemberType NoteProperty -Name "TotalBackups" -Value "$MVSnapshotCount"
# Adding row to array
$MVComplianceReport += $MVComplianceReportLine
# End of For Each VM below
}
# End of For Each VM above
#
##########################
# Show Managed Volume SLA complaince reports
##########################
"----------------------------------------------"
"SLA Compliance Report"
"----------------------------------------------"
$MVComplianceReport | Sort-Object SLACompliance,ManagedVolume | Format-Table
# Uncomment the below if you want a pop up interactive display of the results
# $MVComplianceReport | Sort-Object SLACompliance,ManagedVolume | Out-GridView -Title "Managed Volume Business SLA Compliance Report"
##################################
# Output compliance report to CSV if enabled
##################################
IF ($EnableCSVOutput -eq $TRUE)
{
$CSVFile = $CSVOutputDirectory + "\Rubrik-BusinessSLAReport-ManagedVolumes-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$MVComplianceReport | Sort-Object SLACompliance,ManagedVolume | Export-Csv $CSVFile -Force -NoTypeInformation
}
################################################
# SMTP Email Settings
################################################
IF ($EnableEmail -eq $TRUE)
{
##################################
# SMTP Body - HTML Email style settings
##################################
$TableFont = "#FFFFFF"
$TableBackground = "#00B2A9"
$TableBorder = "#00B2A9"
$HTMLTableStyle = @"
<style type="text/css">
.tg  {border-collapse:collapse;border-spacing:0;border-color:#aaa;}
.tg td{font-family:Arial, sans-serif;font-size:10px;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:#333;background-color:#ffffff;border-top-width:1px;border-bottom-width:1px;}
.tg th{font-family:Arial, sans-serif;font-size:10px;font-weight:bold;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:$TableFont;background-color:$TableBorder;border-top-width:1px;border-bottom-width:1px;}
.tg .tg-tabletop{background-color:$TableBackground;vertical-align:top;text-align:left}
.tg .tg-tablerow{vertical-align:top}
.caption {font-family:Arial, sans-serif;font-size:11px;font-weight:bold;color:$TableFont;}</style>
"@
##################################
# Creating HTML table structure
##################################
$HTMLTableStart = @"
<table class="tg">
<caption><span class="caption">Managed Volume SLA Compliance - $BusinessSLAInHours hours</span></caption>
  <tr>
    <th class="tg-tabletop">SLADomain</th>
    <th class="tg-tabletop">ManagedVolume</th>
    <th class="tg-tabletop">SLACompliance</th>
    <th class="tg-tabletop">LastBackup</th>
    <th class="tg-tabletop">HoursSince</th>
    <th class="tg-tabletop">VolumeSizeGB</th>
    <th class="tg-tabletop">VolumeUsedGB</th>
    <th class="tg-tabletop">TotalBackups</th>
  </tr>
"@
$HTMLTableEnd = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
$ReportList = $MVComplianceReport | Sort-Object SLACompliance,ManagedVolume
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $ReportList) 
{
# Setting values
$HTML1SLADomain = $Row.SLADomain
$HTML1ManagedVolume = $Row.ManagedVolume
$HTML1SLACompliance = $Row.SLACompliance
$HTML1LastBackup = $Row.LastBackup
$HTML1HoursSince = $Row.HoursSince
$HTML1VolumeSizeGB = $Row.VolumeSizeGB
$HTML1VolumeUsedGB = $Row.VolumeUsedGB
$HTML1TotalBackups = $Row.TotalBackups
# Building HTML table row
$HTMLReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML1SLADomain</td>
    <td class=""tg-tablerow"">$HTML1ManagedVolume</td>
    <td class=""tg-tablerow"">$HTML1SLACompliance</td>
    <td class=""tg-tablerow"">$HTML1LastBackup</td>
    <td class=""tg-tablerow"">$HTML1HoursSince</td>
    <td class=""tg-tablerow"">$HTML1VolumeSizeGB</td>
    <td class=""tg-tablerow"">$HTML1VolumeUsedGB</td>
    <td class=""tg-tablerow"">$HTML1TotalBackups</td>
  </tr>
"
# Adding row to table
$HTMLReportTableMiddle += $HTMLReportTableRow
}
##################################
# Creating the Failure Email Body
##################################
$FailureList = $MVComplianceReport | Where-Object{$_.SLACompliance -eq "NotMeetingSLA"} | Sort-Object SLACompliance,ManagedVolume
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLFailureReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $FailureList) 
{
# Setting values
$HTML2SLADomain = $Row.SLADomain
$HTML2ManagedVolume = $Row.ManagedVolume
$HTML2SLACompliance = $Row.SLACompliance
$HTML2LastBackup = $Row.LastBackup
$HTML2HoursSince = $Row.HoursSince
$HTML2VolumeSizeGB = $Row.VolumeSizeGB
$HTML2VolumeUsedGB = $Row.VolumeUsedGB
$HTML2TotalBackups = $Row.TotalBackups
# Building HTML table row
$HTMLFailureReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML2SLADomain</td>
    <td class=""tg-tablerow"">$HTML2ManagedVolume</td>
    <td class=""tg-tablerow"">$HTML2SLACompliance</td>
    <td class=""tg-tablerow"">$HTML2LastBackup</td>
    <td class=""tg-tablerow"">$HTML2HoursSince</td>
    <td class=""tg-tablerow"">$HTML2VolumeSizeGB</td>
    <td class=""tg-tablerow"">$HTML2VolumeUsedGB</td>
    <td class=""tg-tablerow"">$HTML2TotalBackups</td>
  </tr>
"
# Adding row to table
$HTMLFailureReportTableMiddle += $HTMLFailureReportTableRow
}
##################################
# Creating Emails
##################################
# Report email
$HTMLReport = $HTMLTableStyle + $HTMLTableStart + $HTMLReportTableMiddle + $HTMLTableEnd
# Failure email
$HTMLFailureReport = $HTMLTableStyle + $HTMLTableStart + $HTMLFailureReportTableMiddle + $HTMLTableEnd
##################################
# SMTPAuthRequired $TRUE section - Importing credentials if required
##################################
IF ($SMTPAuthRequired -eq $TRUE)
{
##################################
# Importing the SMTP credentials, ensure already created by running: GET-CREDENTIAL –Credential (Get-Credential) | EXPORT-CLIXML ".\SecureCredentials.xml"
##################################
$SMTPCredentialsFile = $ScriptDirectory + "SMTPCredentials.xml"
$SMTPCredentials = IMPORT-CLIXML $SMTPCredentialsFile
##################################
# Sending Report email
##################################
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
# End of auto IT ticket email below
}
# End of auth required below
}
##################################
# End of auth required above, sending the same emails without authentication if specified in the settings
##################################
ELSE
{
##################################
# Sending Report email
##################################
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort
}
}
# End of email section below
}
# End of email section above
###############################################
# End of script
###############################################