﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script allows you to programmitcally create tickets for any protected NAS share that do not have a snapshot in the last x hours (configurable)
# Ensure the ScriptDirectory is configured to the correct location as all the settings are derived from RubrikAutoTicketingv1-Settings.ps1 in it.
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Configure the $ScriptDirectory to a location which contains the below files
# - Before running Configure RubrikAutoTicketingv1-Settings.ps1
# - Before running execute RubrikAutoTicketingv1-Auth.ps1 to store your Rubrik and SMTP server creds
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##################################
# Global settings file location
##################################
$ScriptDirectory = "C:\RubrikAutoTicketingv1\"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
$GlobalSettingsFile = $ScriptDirectory + "RubrikAutoTicketingv1-Settings.ps1"
$SettingsTest = Test-Path $GlobalSettingsFile 
IF ($SettingsTest -eq $FALSE)
{
"No RubrikAutoTicketingv1-Settings.ps1 found in directory:" 
$ScriptDirectory
"Verify the ScriptDirectory variable at the start of each script is configured to the correct directory containing the settings and credential files"
sleep 10
# Terminating script as no valid settings file found
kill $pid
}
##################################
# Global settings import of the variables required
##################################
. $GlobalSettingsFile
##################################
# Importing Rubrik credentials
##################################
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##########################
# Calculating timezone of machine running script
##########################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##########################
# Getting list of SLA Domains
##########################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##########################
# Getting list of Shares
##########################
$ShareListURL = $InternalURL+"host/share?limit=5000"
Try 
{
$ShareListJSON = Invoke-RestMethod -Uri $ShareListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$ShareList = $ShareListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##########################
# Getting list of Filesets
##########################
$FilesetListURL = $BaseURL+"fileset?limit=5000"
Try 
{
$FilesetListJSON = Invoke-RestMethod -Uri $FilesetListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$FilesetList = $FilesetListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# If $FilesetList is null, likely that the JSON was too big for the default 2MB MaxJsonLength, fixing with JavaScriptSerializer
IF ($FilesetList -eq $null)
{
$FilesetListJSONSerialized = ParseItem ((New-Object -TypeName System.Web.Script.Serialization.JavaScriptSerializer -Property @{MaxJsonLength=67108864}).DeserializeObject($FilesetListJSON))
$FilesetList = $FilesetListJSONSerialized.data
}
##########################
# Building list of  Hosts, excluding NAS (separate script)
##########################
$NASFilesetList = $FilesetList | Where-Object {($_.operatingSystemType -eq $null) -and ($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.effectiveSlaDomainId -ne "UNPROTECTED") -and ($_.isRelic -ne "True")} | Sort-Object name |select *
##########################
# Building Report per Fileset
##########################
# Creating array to store report info
$NASComplianceReport=@()
# Performing per Fileset action
ForEach ($Fileset in $NASFilesetList)
{
# Setting variables required
$HostName = $Fileset.hostName
$HostID = $Fileset.hostId
$ShareID = $Fileset.shareId
$FilesetName = $Fileset.name
$FilesetID = $Fileset.id
$SLADomainID = $Fileset.effectiveSlaDomainId
$Includes = $Fileset.includes
$Excludes = $Fileset.excludes
$Exceptions = $Fileset.exceptions
##########################
# Getting SLA Domain name
##########################
$SLADomain = $SLAList  | Where-Object {$_.id -eq $SLADomainID} | Select -ExpandProperty name
##########################
# Getting Share info
##########################
$ShareInfo = $ShareList | Where-Object {$_.id -eq $ShareID} | Select * -First 1
$ShareName = $ShareInfo.exportPoint
$ShareStatus = $ShareInfo.status
$ShareType = $ShareInfo.shareType
##########################
# Getting Fileset detail
##########################
$FilesetInfoURL = $BaseURL+"fileset/"+$FilesetID
Try 
{
$FilesetInfo = Invoke-RestMethod -Uri $FilesetInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$HostSnapshotCount = $FilesetInfo.snapshotCount
$HostSnapshots = $FilesetInfo.snapshots
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($HostSnapshotCount -eq 0)
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
# Resetting values to ensure data isn't carried over between rows
$HostLatestSnapshotAdjusted = $null
$SnapshotAdjustedGapInHours = $null
$HostOnDemandRun = $FALSE
$HostSnapshotFileCount = 0
$HostSnapshotSizeGB = 0
}
ELSE
{
# Snapshot exists, so performing actions
##################################
# Selecting snapshots
##################################
$HostLatestSnapshot1 = $HostSnapshots | Sort-Object -Descending date | select -ExpandProperty date -First 1
$HostLatestSnapshotID = $HostSnapshots | Sort-Object -Descending date | select -ExpandProperty id -First 1
$HostOldestSnapshot1 = $HostSnapshots | Sort-Object date | select -ExpandProperty date -First 1
##########################
# Converting Latest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$HostLatestSnapshot2 = $HostLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$HostLatestCharCount = $HostLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$HostLatestCharSubtract = $HostLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$HostLatestSnapshot3 = $HostLatestSnapshot2.Substring(0,$HostLatestSnapshot2.Length-$HostLatestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$HostLatestSnapshot = ([datetime]::ParseExact($HostLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$HostOldestSnapshot2 = $HostOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$HostOldestCharCount = $HostOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$HostOldestCharSubtract = $HostOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$HostOldestSnapshot3 = $HostOldestSnapshot2.Substring(0,$HostOldestSnapshot2.Length-$HostOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$HostOldestSnapshot = ([datetime]::ParseExact($HostOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Getting Snapshot detail
##########################
$SnapshotInfoURL = $BaseURL+"fileset/snapshot/"+$HostLatestSnapshotID
Try 
{
$SnapshotInfo = Invoke-RestMethod -Uri $SnapshotInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$HostSnapshotFileCount = $SnapshotInfo.fileCount
$HostSnapshotSizeBytes = $SnapshotInfo.size
$HostSnapshotSizeGB = $HostSnapshotSizeBytes / 1000 / 1000 / 1000
$HostSnapshotSizeGB = [Math]::Round($HostSnapshotSizeGB, 0)
##########################
# Calculating SLA compliance
##########################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $HostLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$HostLatestSnapshotAdjusted = $HostLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $HostLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
##################################
# Automatically running an on-demand snapshot for any backups that are not meeting the SLA compliance if $EnableAutoOnDemandSnapshots equals $TRUE
##################################
IF (($EnableAutoOnDemandSnapshots -eq $TRUE) -AND ($SLACompliance -eq "NotMeetingSLA"))
{
# Building POST URL
$FilesetOnDemandSnapURL = $baseURL+"fileset/"+$FilesetID+"/snapshot"
# Building JSON
$FilesetOnDemandSnapJSON =
"{
""slaId"": ""$SLADomainID""
}"
# POST to URL with JSON
Try 
{
Invoke-RestMethod -Uri $FilesetOnDemandSnapURL -Method POST -Body $FilesetOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$HostOnDemandRun = $TRUE
}
Catch 
{
$HostOnDemandRun = $FALSE
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Waiting 30 seconds before processing next object if on-demand snap was posted to stop overloading in case of multiple objects not meeting
sleep 30
}
ELSE
{
# Didn't run on demand snapshot, setting variable
$HostOnDemandRun = $FALSE
}
# End of ELSE action to IF snapshot count equals 0 below
}
# End of ELSE action to IF snapshot count equals 0 above
##########################
# Summarizing info into report
##########################
$NASComplianceReportLine = New-Object PSObject
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SLADomain"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Host" -Value "$HostName"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Share" -Value "$ShareName"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Status" -Value "$ShareStatus"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Type" -Value "$ShareType"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Fileset" -Value "$FilesetName"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastBackup" -Value "$HostLatestSnapshotAdjusted"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "StorageUsedGB" -Value "$HostSnapshotSizeGB"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastFileCount" -Value "$HostSnapshotFileCount"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "TotalBackups" -Value "$HostSnapshotCount"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "OnDemandRun" -Value "$HostOnDemandRun"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Includes" -Value "$Includes"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Excludes" -Value "$Excludes"
$NASComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Exceptions" -Value "$Exceptions"
# Adding row to array
$NASComplianceReport += $NASComplianceReportLine
# End of For Each Fileset below
}
# End of For Each Fileset above
#
##########################
# Show SLA complaince reports
##########################
"----------------------------------------------"
"SLA Compliance Report"
"----------------------------------------------"
$NASComplianceReport | Sort-Object SLACompliance,SLADomain,Share,Fileset | Format-Table
# Uncomment the below if you want a pop up interactive display of the results
# $NASComplianceReport | Sort-Object SLACompliance,SLADomain,Share,Fileset | Out-GridView -Title "NAS Business SLA Compliance Report"
##################################
# Output compliance report to CSV if enabled
##################################
IF ($EnableCSVOutput -eq $TRUE)
{
$CSVFile = $CSVOutputDirectory + "\Rubrik-BusinessSLAReport-NAS-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$NASComplianceReport | Sort-Object SLACompliance,SLADomain,Share,Fileset | Export-Csv $CSVFile -Force -NoTypeInformation
}
################################################
# SMTP Email Settings
################################################
IF ($EnableEmail -eq $TRUE)
{
##################################
# Importing the SMTP credentials, ensure already created by running: GET-CREDENTIAL –Credential (Get-Credential) | EXPORT-CLIXML ".\SecureCredentials.xml"
##################################
$SMTPCredentialsFile = $ScriptDirectory + "SMTPCredentials.xml"
$SMTPCredentials = IMPORT-CLIXML $SMTPCredentialsFile
##################################
# SMTP Body - HTML Email style settings
##################################
$TableFont = "#FFFFFF"
$TableBackground = "#00B2A9"
$TableBorder = "#00B2A9"
$HTMLTableStyle = @"
<style type="text/css">
.tg  {border-collapse:collapse;border-spacing:0;border-color:#aaa;}
.tg td{font-family:Arial, sans-serif;font-size:10px;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:#333;background-color:#ffffff;border-top-width:1px;border-bottom-width:1px;}
.tg th{font-family:Arial, sans-serif;font-size:10px;font-weight:bold;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:$TableFont;background-color:$TableBorder;border-top-width:1px;border-bottom-width:1px;}
.tg .tg-tabletop{background-color:$TableBackground;vertical-align:top;text-align:left}
.tg .tg-tablerow{vertical-align:top}
.caption {font-family:Arial, sans-serif;font-size:11px;font-weight:bold;color:$TableFont;}</style>
"@
##################################
# Creating HTML table structure
##################################
$HTMLTableStart = @"
<table class="tg">
<caption><span class="caption">NAS SLA Compliance - $BusinessSLAInHours hours</span></caption>
  <tr>
    <th class="tg-tabletop">SLADomain</th>
    <th class="tg-tabletop">Host</th>
    <th class="tg-tabletop">Share</th>
    <th class="tg-tabletop">Status</th>
    <th class="tg-tabletop">Type</th>
    <th class="tg-tabletop">Fileset</th>
    <th class="tg-tabletop">SLACompliance</th>
    <th class="tg-tabletop">LastBackup</th>
    <th class="tg-tabletop">HoursSince</th>
    <th class="tg-tabletop">StorageUsedGB</th>
    <th class="tg-tabletop">LastFileCount</th>
    <th class="tg-tabletop">TotalBackups</th>
    <th class="tg-tabletop">OnDemandRun</th>
    <th class="tg-tabletop">Includes</th>
    <th class="tg-tabletop">Excludes</th>
    <th class="tg-tabletop">Exceptions</th>
  </tr>
"@
$HTMLTableEnd = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
$ReportList = $NASComplianceReport | Sort-Object SLACompliance,SLADomain,Share,Fileset
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $ReportList) 
{
# Setting values
$HTML1SLADomain = $Row.SLADomain
$HTML1Host = $Row.Host
$HTML1Share = $Row.Share
$HTML1Status = $Row.Status
$HTML1Type = $Row.Type
$HTML1Fileset = $Row.Fileset
$HTML1SLACompliance = $Row.SLACompliance
$HTML1LastBackup= $Row.LastBackup
$HTML1HoursSince = $Row.HoursSince
$HTML1StorageUsedGB = $Row.StorageUsedGB
$HTML1LastFileCount = $Row.LastFileCount
$HTML1TotalBackups = $Row.TotalBackups
$HTML1OnDemandRun = $Row.OnDemandRun
$HTML1Includes = $Row.Includes
$HTML1Excludes = $Row.Excludes
$HTML1Exceptions = $Row.Exceptions
# Building HTML table row
$HTMLReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML1SLADomain</td>
    <td class=""tg-tablerow"">$HTML1Host</td>
    <td class=""tg-tablerow"">$HTML1Share</td>
    <td class=""tg-tablerow"">$HTML1Status</td>
    <td class=""tg-tablerow"">$HTML1Type</td>
    <td class=""tg-tablerow"">$HTML1Fileset</td>
    <td class=""tg-tablerow"">$HTML1SLACompliance</td>
    <td class=""tg-tablerow"">$HTML1LastBackup</td>
    <td class=""tg-tablerow"">$HTML1HoursSince</td>
    <td class=""tg-tablerow"">$HTML1StorageUsedGB</td>
    <td class=""tg-tablerow"">$HTML1LastFileCount</td>
    <td class=""tg-tablerow"">$HTML1TotalBackups</td>
    <td class=""tg-tablerow"">$HTML1OnDemandRun</td>
    <td class=""tg-tablerow"">$HTML1Includes</td>
    <td class=""tg-tablerow"">$HTML1Excludes</td>
    <td class=""tg-tablerow"">$HTML1Exceptions</td>
  </tr>
"
# Adding row to table
$HTMLReportTableMiddle += $HTMLReportTableRow
}
##################################
# Creating the Failure Email Body
##################################
$FailureList = $NASComplianceReport | Where-Object {$_.SLACompliance -eq "NotMeetingSLA"} | Sort-Object SLACompliance,SLADomain,Share,Fileset
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLFailureReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $FailureList) 
{
# Setting values
$HTML2SLADomain = $Row.SLADomain
$HTML2Host = $Row.Host
$HTML2Share = $Row.Share
$HTML2Status = $Row.Status
$HTML2Type = $Row.Type
$HTML2Fileset = $Row.Fileset
$HTML2SLACompliance = $Row.SLACompliance
$HTML2LastBackup= $Row.LastBackup
$HTML2HoursSince = $Row.HoursSince
$HTML2StorageUsedGB = $Row.StorageUsedGB
$HTML2LastFileCount = $Row.LastFileCount
$HTML2TotalBackups = $Row.TotalBackups
$HTML2OnDemandRun = $Row.OnDemandRun
$HTML2Includes = $Row.Includes
$HTML2Excludes = $Row.Excludes
$HTML2Exceptions = $Row.Exceptions
# Building HTML table row
$HTMLFailureReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML2SLADomain</td>
    <td class=""tg-tablerow"">$HTML2Host</td>
    <td class=""tg-tablerow"">$HTML2Share</td>
    <td class=""tg-tablerow"">$HTML2Status</td>
    <td class=""tg-tablerow"">$HTML2Type</td>
    <td class=""tg-tablerow"">$HTML2Fileset</td>
    <td class=""tg-tablerow"">$HTML2SLACompliance</td>
    <td class=""tg-tablerow"">$HTML2LastBackup</td>
    <td class=""tg-tablerow"">$HTML2HoursSince</td>
    <td class=""tg-tablerow"">$HTML2StorageUsedGB</td>
    <td class=""tg-tablerow"">$HTML2LastFileCount</td>
    <td class=""tg-tablerow"">$HTML2TotalBackups</td>
    <td class=""tg-tablerow"">$HTML2OnDemandRun</td>
    <td class=""tg-tablerow"">$HTML2Includes</td>
    <td class=""tg-tablerow"">$HTML2Excludes</td>
    <td class=""tg-tablerow"">$HTML2Exceptions</td>
  </tr>
"
# Adding row to table
$HTMLFailureReportTableMiddle += $HTMLFailureReportTableRow
}
##################################
# Creating Emails
##################################
# Report email
$HTMLReport = $HTMLTableStyle + $HTMLTableStart + $HTMLReportTableMiddle + $HTMLTableEnd
# Failure email
$HTMLFailureReport = $HTMLTableStyle + $HTMLTableStart + $HTMLFailureReportTableMiddle + $HTMLTableEnd
##################################
# SMTPAuthRequired $TRUE section - Importing credentials if required
##################################
IF ($SMTPAuthRequired -eq $TRUE)
{
##################################
# Importing the SMTP credentials, ensure already created by running: GET-CREDENTIAL –Credential (Get-Credential) | EXPORT-CLIXML ".\SecureCredentials.xml"
##################################
$SMTPCredentialsFile = $ScriptDirectory + "SMTPCredentials.xml"
$SMTPCredentials = IMPORT-CLIXML $SMTPCredentialsFile
##################################
# Sending Report email
##################################
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
# End of auto IT ticket email below
}
# End of auth required below
}
##################################
# End of auth required above, sending the same emails without authentication if specified in the settings
##################################
ELSE
{
##################################
# Sending Report email
##################################
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort
}
}
# End of email section below
}
# End of email section above
###############################################
# End of script
###############################################