﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script automatically verifies each VM snapshot is within the Business SLA you define. 
# Ensure the ScriptDirectory is configured to the correct location as all the settings are derived from RubrikAutoTicketingv1-Settings.ps1 in it.
# Run the RubrikAutoTicketingv1-Auth.ps1 script first to create the credential files needed.
# It can be configured with the settings script to automatically email reports, auto remediate objects with on-demand snapshots, and create helpdesk tickets with a consolidated failure report.
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Configure the $ScriptDirectory to a location which contains the below files
# - Before running Configure RubrikAutoTicketingv1-Settings.ps1
# - Before running execute RubrikAutoTicketingv1-Auth.ps1 to store your Rubrik and SMTP server creds
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##################################
# Global settings file location
##################################
$ScriptDirectory = "C:\RubrikAutoTicketingv1\"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
$GlobalSettingsFile = $ScriptDirectory + "RubrikAutoTicketingv1-Settings.ps1"
$SettingsTest = Test-Path $GlobalSettingsFile 
IF ($SettingsTest -eq $FALSE)
{
"No RubrikAutoTicketingv1-Settings.ps1 found in directory:" 
$ScriptDirectory
"Verify the ScriptDirectory variable at the start of each script is configured to the correct directory containing the settings and credential files"
sleep 10
# Terminating script as no valid settings file found
kill $pid
}
##################################
# Global settings import of the variables required
##################################
. $GlobalSettingsFile
##################################
# Importing Rubrik credentials
##################################
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##########################
# Building Rubrik API string & invoking REST API
##########################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##########################
# Calculating timezone of machine running script
##########################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##################################
# Getting list of SLA Domains
##################################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of VMs
##################################
$VMListURL = $BaseURL+"vmware/vm?limit=5000"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# If $VMList is null, likely that the JSON was too big for the default 2MB MaxJsonLength, fixing with JavaScriptSerializer
IF ($VMList -eq $null)
{
$VMListJSONSerialized = ParseItem ((New-Object -TypeName System.Web.Script.Serialization.JavaScriptSerializer -Property @{MaxJsonLength=67108864}).DeserializeObject($VMListJSON))
$VMList = $VMListJSONSerialized.data
}
##################################
# Building list of UnProtected, but Powered On, VMs
##################################
$UnprotectedVMList = $VMList | Where-Object {($_.effectiveSlaDomainName -eq "UNPROTECTED") -and ($_.PowerStatus -eq "poweredOn") -and ($_.isRelic -ne "True")} | Sort-Object name |select name,powerstatus,moid,clusterName,ipAddress,vmwareToolsInstalled
##########################
# Building list of Protected VMs, will use this to create a list of VMs meeting SLA compliance
##################################
$ProtectedVMList = $VMList | Where-Object {($_.effectiveSlaDomainName -ne "UNPROTECTED") -and ($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.isRelic -ne "True")}
##################################
# Building Report per VM
##################################
# Creating array to store report info
$VMComplianceReport=@()
# Performing per VM action
ForEach ($VM in $ProtectedVMList)
{
# Setting variables required
$VMName = $VM.name
$VMIP = $VM.ipAddress
$VMClusterName = $VM.clusterName
$VMPowerStatus = $VM.powerStatus
$VMSLADomain = $VM.effectiveSlaDomainName
$VMSLADomainID = $VM.effectiveSlaDomainId
$VMID = $VM.id
##########################
# Getting VM detail
##########################
$VMInfoURL = $baseURL+"vmware/vm/"+$VMID
Try 
{
$VMInfo = Invoke-RestMethod -Uri $VMInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling VM detail
$VMSnapshotCount = $VMInfo.snapshotCount
$VMVirtualDisks = $VMInfo.virtualDiskIds.Count
$VMToolInstalled = $VMInfo.vmwareToolsInstalled
$VMGuestOS = $VMInfo.guestOsType
$VMPhysicalStorage = $VMInfo.physicalStorage
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($VMSnapshotCount -eq 0)
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
# Resetting values to ensure data isn't carried over between rows
$VMLatestSnapshotAdjusted = $null
$SnapshotAdjustedGapInHours = $null
$VMOnDemandRun = $FALSE
}
ELSE
{
# Snapshot exists, so performing actions
##########################
# Getting VM snapshot info
##########################
$VMSnapshotURL = $baseURL+"vmware/vm/"+$VMID+"/snapshot"
Try 
{
$VMSnapshotJSON = Invoke-RestMethod -Uri $VMSnapshotURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMSnapshot = $VMSnapshotJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
##################################
# Selecting snapshots
##################################
$VMLatestSnapshot1 = $VMSnapshot | Sort-Object -Descending date | select -ExpandProperty date -First 1
$VMOldestSnapshot1 = $VMSnapshot | Sort-Object date | select -ExpandProperty date -First 1
##################################
# Converting Latest Snapshot
##################################
# Step 1 - Removing characters and trimming snapshot string for conversion
$VMLatestSnapshot2 = $VMLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$VMLatestCharCount = $VMLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$VMLatestCharSubtract = $VMLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$VMLatestSnapshot3 = $VMLatestSnapshot2.Substring(0,$VMLatestSnapshot2.Length-$VMLatestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$VMLatestSnapshot = ([datetime]::ParseExact($VMLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$VMOldestSnapshot2 = $VMOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$VMOldestCharCount = $VMOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$VMOldestCharSubtract = $VMOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$VMOldestSnapshot3 = $VMOldestSnapshot2.Substring(0,$VMOldestSnapshot2.Length-$VMOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$VMOldestSnapshot = ([datetime]::ParseExact($VMOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Calculating SLA compliance
##########################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $VMLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$VMLatestSnapshotAdjusted = $VMLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $VMLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
##################################
# Automatically running an on-demand snapshot for any backups that are not meeting the SLA compliance if $EnableAutoOnDemandSnapshots equals $TRUE
##################################
IF (($EnableAutoOnDemandSnapshots -eq $TRUE) -AND ($SLACompliance -eq "NotMeetingSLA"))
{
# Building POST URL
$VMOnDemandSnapURL = $baseURL+"vmware/vm/"+$VMID+"/snapshot"
# Building JSON
$VMOnDemandSnapJSON =
"{
""slaId"": ""$VMSLADomainID""
}"
# POST to URL with JSON
Try 
{
Invoke-RestMethod -Uri $VMOnDemandSnapURL -Method POST -Body $VMOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMOnDemandRun = $TRUE
}
Catch 
{
$VMOnDemandRun = $FALSE
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Waiting 30 seconds before processing next object if on-demand snap was posted to stop overloading in case of multiple objects not meeting
sleep 30
}
ELSE
{
# Didn't run on demand snapshot, setting variable
$VMOnDemandRun = $FALSE
}
# End of ELSE action to IF snapshot count equals 0 below
}
# End of ELSE action to IF snapshot count equals 0 above
##########################
# Summarizing VM info into report
##########################
$VMComplianceReportLine = new-object PSObject
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$VMSLADomain"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Cluster" -Value "$VMClusterName"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "VM" -Value "$VMName"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastBackup" -Value "$VMLatestSnapshotAdjusted"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "TotalBackups" -Value "$VMSnapshotCount"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "OnDemandRun" -Value "$VMOnDemandRun"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "PowerStatus" -Value "$VMPowerStatus"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "VMTools" -Value "$VMToolInstalled"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "GuestOS" -Value "$VMGuestOS"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "GuestIP" -Value "$VMIP"
$VMComplianceReportLine | Add-Member -MemberType NoteProperty -Name "DiskCount" -Value "$VMVirtualDisks"
# Adding row to array
$VMComplianceReport += $VMComplianceReportLine
# End of For Each VM below
}
# End of For Each VM above
#
##########################
# Show VM SLA complaince reports
##########################
"----------------------------------------------"
"SLA Compliance Report"
"----------------------------------------------"
$VMComplianceReport | Sort-Object SLACompliance,SLADomain,Cluster,VM | Format-Table
# $VMComplianceReport | Sort-Object SLACompliance,SLADomain,Cluster,VM | Out-GridView -Title "VM Business SLA Compliance Report"
"----------------------------------------------"
##################################
# Output compliance report to CSV if enabled
##################################
IF ($EnableCSVOutput -eq $TRUE)
{
$CSVFile = $CSVOutputDirectory + "\Rubrik-BusinessSLAReport-VMwareVMs-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$VMComplianceReport | Sort-Object SLACompliance,SLADomain,Cluster,VM | Export-Csv $CSVFile -Force -NoTypeInformation
}
################################################
# SMTP Email Settings
################################################
IF ($EnableEmail -eq $TRUE)
{
##################################
# SMTP Body - HTML Email style settings
##################################
$TableFont = "#FFFFFF"
$TableBackground = "#00B2A9"
$TableBorder = "#00B2A9"
$HTMLTableStyle = @"
<style type="text/css">
.tg  {border-collapse:collapse;border-spacing:0;border-color:#aaa;}
.tg td{font-family:Arial, sans-serif;font-size:10px;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:#333;background-color:#ffffff;border-top-width:1px;border-bottom-width:1px;}
.tg th{font-family:Arial, sans-serif;font-size:10px;font-weight:bold;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:$TableFont;background-color:$TableBorder;border-top-width:1px;border-bottom-width:1px;}
.tg .tg-tabletop{background-color:$TableBackground;vertical-align:top;text-align:left}
.tg .tg-tablerow{vertical-align:top}
.caption {font-family:Arial, sans-serif;font-size:11px;font-weight:bold;color:$TableFont;}</style>
"@
##################################
# Creating HTML table structure
##################################
$HTMLTableStart = @"
<table class="tg">
<caption><span class="caption">VM SLA Compliance - $BusinessSLAInHours hours</span></caption>
  <tr>
    <th class="tg-tabletop">SLADomain</th>
    <th class="tg-tabletop">Cluster</th>
    <th class="tg-tabletop">VM</th>
    <th class="tg-tabletop">SLACompliance</th>
    <th class="tg-tabletop">LastBackup</th>
    <th class="tg-tabletop">HoursSince</th>
    <th class="tg-tabletop">TotalBackups</th>
    <th class="tg-tabletop">OnDemandRun</th>
    <th class="tg-tabletop">PowerStatus</th>
    <th class="tg-tabletop">VMTools</th>
    <th class="tg-tabletop">GuestOS</th>
    <th class="tg-tabletop">GuestIP</th>
    <th class="tg-tabletop">DiskCount</th>
  </tr>
"@
$HTMLTableEnd = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
$ReportList = $VMComplianceReport | Sort-Object SLACompliance,SLADomain,Cluster,VM
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $ReportList) 
{
# Setting values
$HTML1SLADomain = $Row.SLADomain
$HTML1Cluster = $Row.Cluster
$HTML1VM = $Row.VM
$HTML1SLACompliance = $Row.SLACompliance
$HTML1LastBackup = $Row.LastBackup
$HTML1HoursSince = $Row.HoursSince
$HTML1TotalBackups = $Row.TotalBackups
$HTML1OnDemandRun = $Row.OnDemandRun
$HTML1PowerStatus = $Row.PowerStatus
$HTML1VMTools = $Row.VMTools
$HTML1GuestOS = $Row.GuestOS
$HTML1GuestIP = $Row.GuestIP
$HTML1DiskCount = $Row.DiskCount
# Building HTML table row
$HTMLReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML1SLADomain</td>
    <td class=""tg-tablerow"">$HTML1Cluster</td>
    <td class=""tg-tablerow"">$HTML1VM</td>
    <td class=""tg-tablerow"">$HTML1SLACompliance</td>
    <td class=""tg-tablerow"">$HTML1LastBackup</td>
    <td class=""tg-tablerow"">$HTML1HoursSince</td>
    <td class=""tg-tablerow"">$HTML1TotalBackups</td>
    <td class=""tg-tablerow"">$HTML1OnDemandRun</td>
    <td class=""tg-tablerow"">$HTML1PowerStatus</td>
    <td class=""tg-tablerow"">$HTML1VMTools</td>
    <td class=""tg-tablerow"">$HTML1GuestOS</td>
    <td class=""tg-tablerow"">$HTML1GuestIP</td>
    <td class=""tg-tablerow"">$HTML1DiskCount</td>
  </tr>
"
# Adding row to table
$HTMLReportTableMiddle += $HTMLReportTableRow
}
##################################
# Creating the Failure Email Body
##################################
$FailureList = $VMComplianceReport | Where-Object{$_.SLACompliance -eq "NotMeetingSLA"} | Sort-Object SLACompliance,SLADomain,Cluster,VM
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLFailureReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $FailureList) 
{
# Setting values
$HTML2SLADomain = $Row.SLADomain
$HTML2Cluster = $Row.Cluster
$HTML2VM = $Row.VM
$HTML2SLACompliance = $Row.SLACompliance
$HTML2LastBackup = $Row.LastBackup
$HTML2HoursSince = $Row.HoursSince
$HTML2TotalBackups = $Row.TotalBackups
$HTML2OnDemandRun = $Row.OnDemandRun
$HTML2PowerStatus = $Row.PowerStatus
$HTML2VMTools = $Row.VMTools
$HTML2GuestOS = $Row.GuestOS
$HTML2GuestIP = $Row.GuestIP
$HTML2DiskCount = $Row.DiskCount
# Building HTML table row
$HTMLFailureReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML2SLADomain</td>
    <td class=""tg-tablerow"">$HTML2Cluster</td>
    <td class=""tg-tablerow"">$HTML2VM</td>
    <td class=""tg-tablerow"">$HTML2SLACompliance</td>
    <td class=""tg-tablerow"">$HTML2LastBackup</td>
    <td class=""tg-tablerow"">$HTML2HoursSince</td>
    <td class=""tg-tablerow"">$HTML2TotalBackups</td>
    <td class=""tg-tablerow"">$HTML2OnDemandRun</td>
    <td class=""tg-tablerow"">$HTML2PowerStatus</td>
    <td class=""tg-tablerow"">$HTML2VMTools</td>
    <td class=""tg-tablerow"">$HTML2GuestOS</td>
    <td class=""tg-tablerow"">$HTML2GuestIP</td>
    <td class=""tg-tablerow"">$HTML2DiskCount</td>
  </tr>
"
# Adding row to table
$HTMLFailureReportTableMiddle += $HTMLFailureReportTableRow
}
##################################
# Creating Emails
##################################
# Report email
$HTMLReport = $HTMLTableStyle + $HTMLTableStart + $HTMLReportTableMiddle + $HTMLTableEnd
# Failure email
$HTMLFailureReport = $HTMLTableStyle + $HTMLTableStart + $HTMLFailureReportTableMiddle + $HTMLTableEnd
##################################
# SMTPAuthRequired $TRUE section - Importing credentials if required
##################################
IF ($SMTPAuthRequired -eq $TRUE)
{
##################################
# Importing the SMTP credentials, ensure already created by running: GET-CREDENTIAL –Credential (Get-Credential) | EXPORT-CLIXML ".\SecureCredentials.xml"
##################################
$SMTPCredentialsFile = $ScriptDirectory + "SMTPCredentials.xml"
$SMTPCredentials = IMPORT-CLIXML $SMTPCredentialsFile
##################################
# Sending Report email
##################################
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
# End of auto IT ticket email below
}
# End of auth required below
}
##################################
# End of auth required above, sending the same emails without authentication if specified in the settings
##################################
ELSE
{
##################################
# Sending Report email
##################################
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort
}
}
# End of email section below
}
# End of email section above
###############################################
# End of script
###############################################