﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script allows you to programmitcally create tickets for any protected Windows or Linux hosts that do not have a snapshot in the last x hours (configurable)
# Ensure the ScriptDirectory is configured to the correct location as all the settings are derived from RubrikAutoTicketingv1-Settings.ps1 in it.
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Configure the $ScriptDirectory to a location which contains the below files
# - Before running Configure RubrikAutoTicketingv1-Settings.ps1
# - Before running execute RubrikAutoTicketingv1-Auth.ps1 to store your Rubrik and SMTP server creds
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##################################
# Global settings file location
##################################
$ScriptDirectory = "C:\RubrikAutoTicketingv1\"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
$GlobalSettingsFile = $ScriptDirectory + "RubrikAutoTicketingv1-Settings.ps1"
$SettingsTest = Test-Path $GlobalSettingsFile 
IF ($SettingsTest -eq $FALSE)
{
"No RubrikAutoTicketingv1-Settings.ps1 found in directory:" 
$ScriptDirectory
"Verify the ScriptDirectory variable at the start of each script is configured to the correct directory containing the settings and credential files"
sleep 10
# Terminating script as no valid settings file found
kill $pid
}
##################################
# Global settings import of the variables required
##################################
. $GlobalSettingsFile
##################################
# Importing Rubrik credentials
##################################
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##########################
# Calculating timezone of machine running script
##########################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##########################
# Getting list of SLA Domains
##########################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##########################
# Getting list of Volume Groups
##########################
$VolumeGroupListURL = $InternalURL+"volume_group?limit=5000"
Try 
{
$VolumeGroupListJSON = Invoke-RestMethod -Uri $VolumeGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VolumeGroupList = $VolumeGroupListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# If $FilesetList is null, likely that the JSON was too big for the default 2MB MaxJsonLength, fixing with JavaScriptSerializer
IF ($VolumeGroupList -eq $null)
{
$VolumeGroupListJSONSerialized = ParseItem ((New-Object -TypeName System.Web.Script.Serialization.JavaScriptSerializer -Property @{MaxJsonLength=67108864}).DeserializeObject($FilesetListJSON))
$VolumeGroupList = $VolumeGroupListJSONSerialized.data
}
##########################
# Building list of Protected Volume Groups
##########################
$ProtectedVolumeGroupList = $VolumeGroupList | Where-Object {($_.effectiveSlaDomainId -ne "UNPROTECTED") -and ($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.isRelic -ne "True")} | Sort-Object name |select *
##########################
# Building Report per Volume Group
##########################
# Creating array to store report info
$HostComplianceReport=@()
# Performing per Fileset action
ForEach ($VolumeGroup in $ProtectedVolumeGroupList)
{
# Setting variables required
$HostName = $VolumeGroup.hostName
$HostID = $VolumeGroup.hostId
$VolumeGroupName = $VolumeGroup.name
$VolumeGroupID = $VolumeGroup.id
$SLADomainID = $VolumeGroup.effectiveSlaDomainId
##########################
# Getting SLA Domain name
##########################
$SLADomain = $SLAList  | Where-Object {$_.id -eq $SLADomainID} | Select -ExpandProperty name
##########################
# Getting Volume Group detail
##########################
$VolumeGroupInfoURL = $InternalURL+"volume_group/"+$VolumeGroupID
Try 
{
$VolumeGroupInfo = Invoke-RestMethod -Uri $VolumeGroupInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$VolumeGroupVolumes = $VolumeGroupInfo.volumes
$VolumeGroupVolumeCount = $VolumeGroupVolumes.Count
$VolumeMountPoints = $VolumeGroupVolumes.mountPoints
##########################
# Getting Volume Group Snapshots
##########################
$VolumeGroupSnapshotsURL = $InternalURL+"volume_group/"+$VolumeGroupID+"/snapshot"
Try 
{
$VolumeGroupSnapshotsJSON = Invoke-RestMethod -Uri $VolumeGroupSnapshotsURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VolumeGroupSnapshots = $VolumeGroupSnapshotsJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Counting snapshots
$VolumeGroupSnapshotCount = $VolumeGroupSnapshots.count
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($VolumeGroupSnapshotCount -eq 0)
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
# Resetting values to ensure data isn't carried over between rows
$VoumeGroupLatestSnapshotAdjusted = $null
$VolumeGroupOnDemandRun = $FALSE
$SnapshotAdjustedGapInHours = $null
$VolumeGroupSnapshotSizeGB = 0
}
ELSE
{
# Snapshot exists, so performing actions
##################################
# Selecting snapshots
##################################
$VolumeGroupLatestSnapshot1 = $VolumeGroupSnapshots | Sort-Object -Descending date | select -ExpandProperty date -First 1
$VolumeGroupLatestSnapshotID = $VolumeGroupSnapshots | Sort-Object -Descending date | select -ExpandProperty id -First 1
$VolumeGroupOldestSnapshot1 = $VolumeGroupSnapshots | Sort-Object date | select -ExpandProperty date -First 1
##########################
# Converting Latest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$VolumeGroupLatestSnapshot2 = $VolumeGroupLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$VolumeGroupLatestCharCount = $VolumeGroupLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$VolumeGroupCharSubtract = $VolumeGroupLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$VolumeGroupLatestSnapshot3 = $VolumeGroupLatestSnapshot2.Substring(0,$VolumeGroupLatestSnapshot2.Length-$VolumeGroupCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$VolumeGroupLatestSnapshot = ([datetime]::ParseExact($VolumeGroupLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$VolumeGroupOldestSnapshot2 = $VolumeGroupOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$VolumeGroupOldestCharCount = $VolumeGroupOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$VolumeGroupOldestCharSubtract = $VolumeGroupOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$VolumeGroupOldestSnapshot3 = $VolumeGroupOldestSnapshot2.Substring(0,$VolumeGroupOldestSnapshot2.Length-$VolumeGroupOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$VolumeGroupOldestSnapshot = ([datetime]::ParseExact($VolumeGroupOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Getting Snapshot detail
##########################
$SnapshotInfoURL = $InternalURL+"volume_group/snapshot/"+$VolumeGroupLatestSnapshotID
Try 
{
$SnapshotInfo = Invoke-RestMethod -Uri $SnapshotInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$SnapshotVolumeSizes = $SnapshotInfo.includedVolumes | Select -ExpandProperty Size
# Calculating total
$TotalSnapshotVolumeSize = 0
$SnapshotVolumeSizes | Foreach {$TotalSnapshotVolumeSizeBytes += $_}
$VolumeGroupSnapshotSizeGB = $TotalSnapshotVolumeSizeBytes / 1000 / 1000 / 1000
$VolumeGroupSnapshotSizeGB = [Math]::Round($VolumeGroupSnapshotSizeGB, 0)
##########################
# Calculating SLA compliance
##########################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $VolumeGroupLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$VolumeGroupLatestSnapshotAdjusted = $VolumeGroupLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $VolumeGroupLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
##################################
# Automatically running an on-demand snapshot for any backups that are not meeting the SLA compliance if $EnableAutoOnDemandSnapshots equals $TRUE
##################################
IF (($EnableAutoOnDemandSnapshots -eq $TRUE) -AND ($SLACompliance -eq "NotMeetingSLA"))
{
# Building POST URL
$VolumeGroupOnDemandSnapURL = $InternalURL+"volume_group/"+$VolumeGroupID+"/snapshot"
# Building JSON
$VolumeGroupOnDemandSnapJSON =
"{
""slaId"": ""$SLADomainID""
}"
# POST to URL with JSON
Try 
{
Invoke-RestMethod -Uri $VolumeGroupOnDemandSnapURL -Method POST -Body $VolumeGroupOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$HostOnDemandRun = $TRUE
}
Catch 
{
$VolumeGroupOnDemandRun = $FALSE
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Waiting 30 seconds before processing next object if on-demand snap was posted to stop overloading in case of multiple objects not meeting
sleep 30
}
ELSE
{
# Didn't run on demand snapshot, setting variable
$VolumeGroupOnDemandRun = $FALSE
}
# End of ELSE action to IF snapshot count equals 0 below
}
# End of ELSE action to IF snapshot count equals 0 above
##########################
# Summarizing info into report
##########################
$HostComplianceReportLine = New-Object PSObject
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SLADomain"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Host" -Value "$HostName"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Volumes" -Value "$VolumeGroupVolumeCount"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "VolumeGroup" -Value "$VolumeGroupName"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastBackup" -Value "$VolumeGroupLatestSnapshotAdjusted"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SizeGB" -Value "$VolumeGroupSnapshotSizeGB"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Snapshots" -Value "$VolumeGroupSnapshotCount"
$HostComplianceReportLine | Add-Member -MemberType NoteProperty -Name "OnDemandRun" -Value "$VolumeGroupOnDemandRun"
# Adding row to array
$HostComplianceReport += $HostComplianceReportLine
# End of For Each Fileset below
}
# End of For Each Fileset above
#
##########################
# Show SLA complaince reports
##########################
"----------------------------------------------"
"SLA Compliance Report"
"----------------------------------------------"
$HostComplianceReport | Sort-Object SLACompliance,SLADomain,OS,HostName
# Uncomment the below if you want a pop up interactive display of the results
# $HostComplianceReport | Sort-Object SLACompliance,SLADomain,OS,HostName | Out-GridView -Title "Host SLA Compliance Report"
##################################
# Output compliance report to CSV if enabled
##################################
IF ($EnableCSVOutput -eq $TRUE)
{
$CSVFile = $CSVOutputDirectory + "/Rubrik-BusinessSLAReport-PhysicalHosts-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$HostComplianceReport | Sort-Object SLACompliance,SLADomain,OS,HostName | Export-Csv $CSVFile -Force -NoTypeInformation
}
################################################
# SMTP Email Settings
################################################
IF ($EnableEmail -eq $TRUE)
{
##################################
# SMTP Body - HTML Email style settings
##################################
$TableFont = "#FFFFFF"
$TableBackground = "#00B2A9"
$TableBorder = "#00B2A9"
$HTMLTableStyle = @"
<style type="text/css">
.tg  {border-collapse:collapse;border-spacing:0;border-color:#aaa;}
.tg td{font-family:Arial, sans-serif;font-size:10px;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:#333;background-color:#ffffff;border-top-width:1px;border-bottom-width:1px;}
.tg th{font-family:Arial, sans-serif;font-size:10px;font-weight:bold;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:$TableFont;background-color:$TableBorder;border-top-width:1px;border-bottom-width:1px;}
.tg .tg-tabletop{background-color:$TableBackground;vertical-align:top;text-align:left}
.tg .tg-tablerow{vertical-align:top}
.caption {font-family:Arial, sans-serif;font-size:11px;font-weight:bold;color:$TableFont;}</style>
"@
##################################
# Creating HTML table structure
##################################
$HTMLTableStart = @"
<table class="tg">
<caption><span class="caption">Physical Host SLA Compliance - $BusinessSLAInHours hours</span></caption>
  <tr>
    <th class="tg-tabletop">SLADomain</th>
    <th class="tg-tabletop">Host</th>
    <th class="tg-tabletop">Volumes</th>
    <th class="tg-tabletop">VolumeGroup</th>
    <th class="tg-tabletop">SLACompliance</th>
    <th class="tg-tabletop">LastBackup</th>
    <th class="tg-tabletop">HoursSince</th>
    <th class="tg-tabletop">SizeGB</th>
    <th class="tg-tabletop">Snapshots</th>
    <th class="tg-tabletop">OnDemandRun</th>
  </tr>
"@
$HTMLTableEnd = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
$ReportList = $HostComplianceReport | Sort-Object SLACompliance,SLADomain,OS,HostName
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTableMiddle = $null
# Creating table row for each line
ForEach ($Row in $ReportList) 
{
# Setting values
$HTML1SLADomain = $Row.SLADomain
$HTML1Host = $Row.Host
$HTML1Volumes = $Row.Volumes
$HTML1VolumeGroup = $Row.VolumeGroup
$HTML1SLACompliance = $Row.SLACompliance
$HTML1LastBackup = $Row.LastBackup
$HTML1HoursSince = $Row.HoursSince
$HTML1SizeGB = $Row.SizeGB
$HTML1Snapshots = $Row.Snapshots
$HTML1OnDemandRun = $Row.OnDemandRun
# Building HTML table row
$HTMLReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML1SLADomain</td>
    <td class=""tg-tablerow"">$HTML1Host</td>
    <td class=""tg-tablerow"">$HTML1Volumes</td>
    <td class=""tg-tablerow"">$HTML1VolumeGroup</td>
    <td class=""tg-tablerow"">$HTML1SLACompliance</td>
    <td class=""tg-tablerow"">$HTML1LastBackup</td>
    <td class=""tg-tablerow"">$HTML1HoursSince</td>
    <td class=""tg-tablerow"">$HTML1SizeGB</td>
    <td class=""tg-tablerow"">$HTML1Snapshots</td>
    <td class=""tg-tablerow"">$HTML1OnDemandRun</td>
  </tr>
"
# Adding row to table
$HTMLReportTableMiddle += $HTMLReportTableRow
}
##################################
# Creating the Failure Email Body
##################################
$FailureList = $HostComplianceReport | Where-Object {$_.SLACompliance -eq "NotMeetingSLA"} | Sort-Object SLACompliance,SLADomain,OS,HostName
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLFailureReportTableMiddle = $null
# Creating table row for each line
ForEach ($Row in $FailureList) 
{
# Setting values
$HTML2SLADomain = $Row.SLADomain
$HTML2Host = $Row.Host
$HTML2Volumes = $Row.Volumes
$HTML2VolumeGroup = $Row.VolumeGroup
$HTML2SLACompliance = $Row.SLACompliance
$HTML2LastBackup = $Row.LastBackup
$HTML2HoursSince = $Row.HoursSince
$HTML2SizeGB = $Row.SizeGB
$HTML2Snapshots = $Row.Snapshots
$HTML2OnDemandRun = $Row.OnDemandRun
# Building HTML table row
$HTMLFailureReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML2SLADomain</td>
    <td class=""tg-tablerow"">$HTML2Host</td>
    <td class=""tg-tablerow"">$HTML2Volumes</td>
    <td class=""tg-tablerow"">$HTML2VolumeGroup</td>
    <td class=""tg-tablerow"">$HTML2SLACompliance</td>
    <td class=""tg-tablerow"">$HTML2LastBackup</td>
    <td class=""tg-tablerow"">$HTML2HoursSince</td>
    <td class=""tg-tablerow"">$HTML2SizeGB</td>
    <td class=""tg-tablerow"">$HTML2Snapshots</td>
    <td class=""tg-tablerow"">$HTML2OnDemandRun</td>
  </tr>
"
# Adding row to table
$HTMLFailureReportTableMiddle += $HTMLFailureReportTableRow
}
##################################
# Creating Emails
##################################
# Report email
$HTMLReport = $HTMLTableStyle + $HTMLTableStart + $HTMLReportTableMiddle + $HTMLTableEnd
# Failure email
$HTMLFailureReport = $HTMLTableStyle + $HTMLTableStart + $HTMLFailureReportTableMiddle + $HTMLTableEnd
##################################
# SMTPAuthRequired $TRUE section - Importing credentials if required
##################################
IF ($SMTPAuthRequired -eq $TRUE)
{
##################################
# Importing the SMTP credentials, ensure already created by running: GET-CREDENTIAL –Credential (Get-Credential) | EXPORT-CLIXML ".\SecureCredentials.xml"
##################################
$SMTPCredentialsFile = $ScriptDirectory + "SMTPCredentials.xml"
$SMTPCredentials = IMPORT-CLIXML $SMTPCredentialsFile
##################################
# Sending Report email
##################################
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
# End of auto IT ticket email below
}
# End of auth required below
}
##################################
# End of auth required above, sending the same emails without authentication if specified in the settings
##################################
ELSE
{
##################################
# Sending Report email
##################################
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort
}
}
# End of email section below
}
# End of email section above
###############################################
# End of script
###############################################