﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script allows you to programmitcally create tickets and report on all protected MS SQL DBs that do not have a snapshot/backup in the last x hours (configurable). 
# It also shows log backups, but does not pass/fail based on these
# Ensure the ScriptDirectory is configured to the correct location as all the settings are derived from RubrikAutoTicketingv1-Settings.ps1 in it.
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Configure the $ScriptDirectory to a location which contains the below files
# - Before running Configure RubrikAutoTicketingv1-Settings.ps1
# - Before running execute RubrikAutoTicketingv1-Auth.ps1 to store your Rubrik and SMTP server creds
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##################################
# Global settings file location
##################################
$ScriptDirectory = "C:\RubrikAutoTicketingv1\"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
$GlobalSettingsFile = $ScriptDirectory + "RubrikAutoTicketingv1-Settings.ps1"
$SettingsTest = Test-Path $GlobalSettingsFile 
IF ($SettingsTest -eq $FALSE)
{
"No RubrikAutoTicketingv1-Settings.ps1 found in directory:" 
$ScriptDirectory
"Verify the ScriptDirectory variable at the start of each script is configured to the correct directory containing the settings and credential files"
sleep 10
# Terminating script as no valid settings file found
kill $pid
}
##################################
# Global settings import of the variables required
##################################
. $GlobalSettingsFile
##################################
# Importing Rubrik credentials
##################################
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Calculating timezone of machine running script
##################################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##################################
# Getting list of SLA Domains
##################################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of MS SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceListJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# If $SQLDBList is null, likely that the JSON was too big for the default 2MB MaxJsonLength, fixing with JavaScriptSerializer
IF ($SQLDBList -eq $null)
{
$SQLDBListJSONSerialized = ParseItem ((New-Object -TypeName System.Web.Script.Serialization.JavaScriptSerializer -Property @{MaxJsonLength=67108864}).DeserializeObject($SQLDBListJSON))
$SQLDBList = $SQLDBListJSONSerialized.data
}
##################################
# Building list of SQL DBs
##################################
$SQLDBs = ($SQLDBList | Where-Object {(($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.state -eq "ONLINE") -and ($_.isRelic -ne "True"))} | select *
##################################
# Building Report per SQL DB
##################################
# Creating array to store report info
$SQLDBComplianceReport=@()
# Performing per Fileset action
ForEach ($SQLDB in $SQLDBs)
{
# Setting variables required
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SLADomainName = $SQLDB.effectiveSlaDomainName
$SQLInAG = $SQLDB.isInAvailabilityGroup
$SQLInLM = $SQLDB.isLiveMount
$SQLDBInstanceName = $SQLDB.replicas.instanceName
$SQLDBHostName = $SQLDB.replicas.rootProperties.rootName
$SQLDBState = $SQLDB.state
$SQLDBEffectiveSLADomainID = $SQLDB.effectiveSlaDomainId
##################################
# Getting SQL DB detail
##################################
$SQLDBInfoURL = $baseURL+"mssql/db/"+$SQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$SQLDBSnapshotCount = $SQLDBInfo.snapshotCount
$SQLDBInstanceID = $SQLDBInfo.primaryClusterId
$SQLDBSLADomainID = $SQLDBInfo.effectiveSlaDomainId
$SQLDBStorageUsed = $SQLDBInfo.localStorage
$SQLDBStorageUsedGB = $SQLDBStorageUsed / 1000 / 1000 / 1000
# Rounding differently depending on size
IF ($SQLDBStorageUsedGB -lt 1)
{
$SQLDBStorageUsedGB = [Math]::Round($SQLDBStorageUsedGB, 3)
}
ELSE
{
$SQLDBStorageUsedGB = [Math]::Round($SQLDBStorageUsedGB, 0)
} 
# The below is the latest recoverable point in time including logs (if logs are being protected)
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Getting the recovery model
$SQLDBRecoveryModel = $SQLDBInfo.replicas.recoveryModel
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($SQLDBSnapshotCount -eq 0)
{
# If DB is unprotected setting SLA compliance to "UNPROTECTED"
IF ($SQLDBEffectiveSLADomainID -eq "UNPROTECTED")
{
$SLACompliance = "Unprotected"
}
ELSE
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
}
# Resetting values to ensure data isn't carried over between rows
$SQLDBLatestSnapshotAdjusted = $null
$SnapshotAdjustedGapInHours = $null
$SQLDBLatestRecoveryPointAdjusted = $null
$LogAdjustedGapInMinutes = $null
$SQLDBOnDemandRun = $FALSE
}
ELSE
{
# Snapshot exists, so performing actions
##################################
# Getting SQL DB snapshot detail
##################################
$SQLDBSnapshotsURL = $baseURL+"mssql/db/"+$SQLDBID+"/snapshot"
Try 
{
$SQLDBSnapshotsJSON = Invoke-RestMethod -Uri $SQLDBSnapshotsURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBSnapshots = $SQLDBSnapshotsJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
##################################
# Selecting snapshots
##################################
$SQLDBLatestSnapshot1 = $SQLDBSnapshots | Sort-Object -Descending date | select -ExpandProperty date -First 1
$SQLDBLatestSnapshotID = $SQLDBSnapshots | Sort-Object -Descending date | select -ExpandProperty id -First 1
$SQLDBOldestSnapshot1 = $SQLDBSnapshots | Sort-Object date | select -ExpandProperty date -First 1
##################################
# Converting Latest Snapshot
##################################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBLatestSnapshot2 = $SQLDBLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBLatestCharCount = $SQLDBLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBLatestCharSubtract = $SQLDBLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBLatestSnapshot3 = $SQLDBLatestSnapshot2.Substring(0,$SQLDBLatestSnapshot2.Length-$SQLDBLatestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBLatestSnapshot = ([datetime]::ParseExact($SQLDBLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBOldestSnapshot2 = $SQLDBOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBOldestCharCount = $SQLDBOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBOldestCharSubtract = $SQLDBOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBOldestSnapshot3 = $SQLDBOldestSnapshot2.Substring(0,$SQLDBOldestSnapshot2.Length-$SQLDBOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBOldestSnapshot = ([datetime]::ParseExact($SQLDBOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Latest Log Backup
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBLatestRecoveryPoint1 = $SQLDBLatestRecoveryPoint.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBLatestRPCharCount = $SQLDBLatestRecoveryPoint1 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBLatestRPCharSubtract = $SQLDBLatestRPCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBLatestRecoveryPoint2 = $SQLDBLatestRecoveryPoint1.Substring(0,$SQLDBLatestRecoveryPoint1.Length-$SQLDBLatestRPCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBLatestRecoveryPoint = ([datetime]::ParseExact($SQLDBLatestRecoveryPoint2,”yyyy-MM-dd HH:mm:ss”,$null))
##################################
# Calculating SLA compliance
##################################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $SQLDBLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$SQLDBLatestSnapshotAdjusted = $SQLDBLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $SQLDBLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
# Adjusting Log backup
$SQLDBLatestRecoveryPointAdjusted = $SQLDBLatestRecoveryPoint.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$LogAdjustedGap = NEW-TIMESPAN –Start $SQLDBLatestRecoveryPointAdjusted –End $SystemDateTime
$LogAdjustedGapInMinutes = $LogAdjustedGap.TotalMinutes
$LogAdjustedGapInMinutes = [Math]::Round($LogAdjustedGapInMinutes, 0)
##################################
# Setting log values to NULL if DB is in SIMPLE $SQLDBRecoveryModel
##################################
IF ($SQLDBRecoveryModel -eq "SIMPLE")
{
$SQLDBLatestRecoveryPointAdjusted = $null
$LogAdjustedGapInMinutes= $null
}
##################################
# Automatically running an on-demand snapshot for any backups that are not meeting the SLA compliance if $EnableAutoOnDemandSnapshots equals $TRUE
##################################
IF (($EnableAutoOnDemandSnapshots -eq $TRUE) -AND ($SLACompliance -eq "NotMeetingSLA"))
{
# Building POST URL
$SQLDBOnDemandSnapURL = $baseURL+"mssql/db/"+$SQLDBID+"/snapshot"
# Building JSON
$SQLDBOnDemandSnapJSON =
"{
""slaId"": ""$SQLDBSLADomainID"",
""forceFullSnapshot"": true
}"
# POST to URL with JSON
Try 
{
Invoke-RestMethod -Uri $SQLDBOnDemandSnapURL -Method POST -Body $SQLDBOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBOnDemandRun = $TRUE
}
Catch 
{
$SQLDBOnDemandRun = $FALSE
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Waiting 30 seconds before processing next object if on-demand snap was posted to stop overloading in case of multiple objects not meeting
sleep 30
}
ELSE
{
# Didn't run on demand snapshot, setting variable
$SQLDBOnDemandRun = $FALSE
}
# End of ELSE action to IF snapshot count equals 0 above
}
# End of ELSE action to IF snapshot count equals 0 above
##################################
# Summarizing info into report
##################################
$SQLDBComplianceReportLine = new-object PSObject
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SLADomainName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Host" -Value "$SQLDBHostName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLDBInstanceName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "DB" -Value "$SQLDBName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastFullBackup" -Value "$SQLDBLatestSnapshotAdjusted"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "RecoveryModel" -Value "$SQLDBRecoveryModel"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastLogBackup" -Value "$SQLDBLatestRecoveryPointAdjusted"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "MinutesSince" -Value "$LogAdjustedGapInMinutes"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "TotalBackups" -Value "$SQLDBSnapshotCount"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "StorageUsedGB" -Value "$SQLDBStorageUsedGB"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "OnDemandRun" -Value "$SQLDBOnDemandRun"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LiveMounted" -Value "$SQLInLM"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "InAvGroup" -Value "$SQLInAG"
# Adding row to array
$SQLDBComplianceReport += $SQLDBComplianceReportLine
# End of For Each SQL DB below
}
# End of For Each SQL DB above
#
##################################
# Show SLA compliance reports
##################################
"----------------------------------------------"
"SLA Compliance Report"
"----------------------------------------------"
$SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB | Format-Table
# Uncomment the below if you want a pop up interactive display of the results
# $SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB | Out-GridView -Title "SQL DB SLA Compliance Report"
##################################
# Output compliance report to CSV if enabled
##################################
IF ($EnableCSVOutput -eq $TRUE)
{
$CSVFile = $CSVOutputDirectory + "\Rubrik-BusinessSLAReport-SQL-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB | Export-Csv $CSVFile -Force -NoTypeInformation
}
################################################
# SMTP Email Settings
################################################
IF ($EnableEmail -eq $TRUE)
{
##################################
# SMTP Body - HTML Email style settings
##################################
$TableFont = "#FFFFFF"
$TableBackground = "#00B2A9"
$TableBorder = "#00B2A9"
$HTMLTableStyle = @"
<style type="text/css">
.tg  {border-collapse:collapse;border-spacing:0;border-color:#aaa;}
.tg td{font-family:Arial, sans-serif;font-size:10px;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:#333;background-color:#ffffff;border-top-width:1px;border-bottom-width:1px;}
.tg th{font-family:Arial, sans-serif;font-size:10px;font-weight:bold;padding:10px 5px;border-style:solid;border-width:0px;overflow:hidden;word-break:normal;border-color:#aaa;color:$TableFont;background-color:$TableBorder;border-top-width:1px;border-bottom-width:1px;}
.tg .tg-tabletop{background-color:$TableBackground;vertical-align:top;text-align:left}
.tg .tg-tablerow{vertical-align:top}
.caption {font-family:Arial, sans-serif;font-size:11px;font-weight:bold;color:$TableFont;}</style>
"@
##################################
# Creating HTML table structure
##################################
$HTMLTableStart = @"
<table class="tg">
<caption><span class="caption">SQL DB SLA Compliance - $BusinessSLAInHours hours</span></caption>
  <tr>
    <th class="tg-tabletop">SLADomain</th>
    <th class="tg-tabletop">Host</th>
    <th class="tg-tabletop">Instance</th>
    <th class="tg-tabletop">DB</th>
    <th class="tg-tabletop">SLACompliance</th>
    <th class="tg-tabletop">LastFullBackup</th>
    <th class="tg-tabletop">HoursSince</th>
    <th class="tg-tabletop">RecoveryModel</th>
    <th class="tg-tabletop">LastLogBackup</th>
    <th class="tg-tabletop">MinutesSince</th>
    <th class="tg-tabletop">TotalBackups</th>
    <th class="tg-tabletop">StorageUsedGB</th>
    <th class="tg-tabletop">OnDemandRun</th>
    <th class="tg-tabletop">LiveMounted</th>
    <th class="tg-tabletop">InAvGroup</th>
  </tr>
"@
$HTMLTableEnd = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
$ReportList = $SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $ReportList) 
{
# Setting values
$HTML1SLADomain = $Row.SLADomain
$HTML1Host = $Row.Host
$HTML1Instance = $Row.Instance
$HTML1DB = $Row.DB
$HTML1SLACompliance = $Row.SLACompliance
$HTML1LastFullBackup = $Row.LastFullBackup
$HTML1HoursSince = $Row.HoursSince
$HTML1StorageUsedGB = $Row.StorageUsedGB
$HTML1RecoveryModel = $Row.RecoveryModel
$HTML1LastLogBackup = $Row.LastLogBackup
$HTML1MinutesSince = $Row.MinutesSince
$HTML1TotalBackups = $Row.TotalBackups
$HTML1OnDemandRun = $Row.OnDemandRun
$HTML1LiveMounted = $Row.LiveMounted
$HTML1InAvGroup = $Row.InAvGroup
# Building HTML table row
$HTMLReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML1SLADomain</td>
    <td class=""tg-tablerow"">$HTML1Host</td>
    <td class=""tg-tablerow"">$HTML1Instance</td>
    <td class=""tg-tablerow"">$HTML1DB</td>
    <td class=""tg-tablerow"">$HTML1SLACompliance</td>
    <td class=""tg-tablerow"">$HTML1LastFullBackup</td>
    <td class=""tg-tablerow"">$HTML1HoursSince</td>
    <td class=""tg-tablerow"">$HTML1RecoveryModel</td>
    <td class=""tg-tablerow"">$HTML1LastLogBackup</td>
    <td class=""tg-tablerow"">$HTML1MinutesSince</td>
    <td class=""tg-tablerow"">$HTML1TotalBackups</td>
    <td class=""tg-tablerow"">$HTML1StorageUsedGB</td>
    <td class=""tg-tablerow"">$HTML1OnDemandRun</td>
    <td class=""tg-tablerow"">$HTML1LiveMounted</td>
    <td class=""tg-tablerow"">$HTML1InAvGroup</td>
  </tr>
"
# Adding row to table
$HTMLReportTableMiddle += $HTMLReportTableRow
}
##################################
# Creating the Failure Email Body
##################################
$FailureList = $SQLDBComplianceReport | Where-Object{$_.SLACompliance -eq "NotMeetingSLA"} | Sort-Object SLACompliance,SLADomain,Host,Instance,DB
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLFailureReportTableMiddle = $null
# Creating table row for each line
foreach ($Row in $FailureList) 
{
# Setting values
$HTML2SLADomain = $Row.SLADomain
$HTML2Host = $Row.Host
$HTML2Instance = $Row.Instance
$HTML2DB = $Row.DB
$HTML2SLACompliance = $Row.SLACompliance
$HTML2LastFullBackup = $Row.LastFullBackup
$HTML2HoursSince = $Row.HoursSince
$HTML2StorageUsedGB = $Row.StorageUsedGB
$HTML2RecoveryModel = $Row.RecoveryModel
$HTML2LastLogBackup = $Row.LastLogBackup
$HTML2MinutesSince = $Row.MinutesSince
$HTML2TotalBackups = $Row.TotalBackups
$HTML2OnDemandRun = $Row.OnDemandRun
$HTML2LiveMounted = $Row.LiveMounted
$HTML2InAvGroup = $Row.InAvGroup
# Building HTML table row
$HTMLFailureReportTableRow = "
<tr>
    <td class=""tg-tablerow"">$HTML2SLADomain</td>
    <td class=""tg-tablerow"">$HTML2Host</td>
    <td class=""tg-tablerow"">$HTML2Instance</td>
    <td class=""tg-tablerow"">$HTML2DB</td>
    <td class=""tg-tablerow"">$HTML2SLACompliance</td>
    <td class=""tg-tablerow"">$HTML2LastFullBackup</td>
    <td class=""tg-tablerow"">$HTML2HoursSince</td>
    <td class=""tg-tablerow"">$HTML2RecoveryModel</td>
    <td class=""tg-tablerow"">$HTML2LastLogBackup</td>
    <td class=""tg-tablerow"">$HTML2MinutesSince</td>
    <td class=""tg-tablerow"">$HTML2TotalBackups</td>
    <td class=""tg-tablerow"">$HTML2StorageUsedGB</td>
    <td class=""tg-tablerow"">$HTML2OnDemandRun</td>
    <td class=""tg-tablerow"">$HTML2LiveMounted</td>
    <td class=""tg-tablerow"">$HTML2InAvGroup</td>
  </tr>
"
# Adding row to table
$HTMLFailureReportTableMiddle += $HTMLFailureReportTableRow
}
##################################
# Creating Emails
##################################
# Report email
$HTMLReport = $HTMLTableStyle + $HTMLTableStart + $HTMLReportTableMiddle + $HTMLTableEnd
# Failure email
$HTMLFailureReport = $HTMLTableStyle + $HTMLTableStart + $HTMLFailureReportTableMiddle + $HTMLTableEnd
##################################
# SMTPAuthRequired $TRUE section - Importing credentials if required
##################################
IF ($SMTPAuthRequired -eq $TRUE)
{
##################################
# Importing the SMTP credentials, ensure already created by running: GET-CREDENTIAL –Credential (Get-Credential) | EXPORT-CLIXML ".\SecureCredentials.xml"
##################################
$SMTPCredentialsFile = $ScriptDirectory + "SMTPCredentials.xml"
$SMTPCredentials = IMPORT-CLIXML $SMTPCredentialsFile
##################################
# Sending Report email
##################################
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
IF ($SMTPSSLEnabled -eq $True)
{
# Using SSL if $SMTPSSLEnabled equals TRUE
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials -UseSsl
}
ELSE
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort -Credential $SMTPCredentials
}
# End of auto IT ticket email below
}
# End of auth required below
}
##################################
# End of auth required above, sending the same emails without authentication if specified in the settings
##################################
ELSE
{
##################################
# Sending Report email
##################################
Send-MailMessage -To $ReportEmailTo -From $EmailFrom -Subject $ReportEmailSubject -BodyAsHtml -Body $HTMLReport -SmtpServer $SMTPServer -Port $SMTPPort
##################################
# Sending Auto IT Ticket email, but only if objects found to be NotMeetingSLA
##################################
IF ($FailureList -ne $null)
{
Send-MailMessage -To $FailureEmailTo -From $EmailFrom -Subject $FailureEmailSubject -BodyAsHtml -Body $HTMLFailureReport -SmtpServer $SMTPServer -Port $SMTPPort
}
}
# End of email section below
}
# End of email section above
###############################################
# End of script
###############################################