﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates an on-demand snapshot for the target DB specified, then refreshes it using an export from the source DB
# WARNING: this script uses the source DB path on the destination server, if this doesn't exist then change the variable below
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A SQL database ready to protect, it doesn't have to have an existing SLA domain assigned
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
# Either remove the below and pass them into the script as variables or hard code
# Source DB settings
$SQLSourceHost = "SQL16-VM01.lab.local\INSTANCE01"
$SQLSourceDB = "SQL16-VM01-DemoDB01"
# Target DB settings
$SQLTargetHost = "SQL16-VM02.lab.local\MSSQLSERVER"
$SQLTargetDB = "ExportDB01"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
####################################################################
# Section 1 - Connecting to API and building table of DBs
####################################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainListURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainListJSON = Invoke-RestMethod -Uri $SLADomainListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainList = $SLADomainListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000&is_relic=false"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
##################################
$SQLInstanceArray = @()
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Combining server name and instance
$SQLInstanceServer = $SQLInstanceHostName + "\" + $SQLInstanceName 
# Adding to array
$SQLInstanceArrayLine = New-Object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
ForEach ($SQLDB in $SQLDBList)
{
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
$SQLDBSLADomain = $SQLDB.effectiveSlaDomainName
$SQLDBSLADomainID = $SQLDB.effectiveSlaDomainId
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
$SQLInstanceServer = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Server
}
# Adding to array
$SQLDBArrayLine = New-Object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SQLDBSLADomain"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomainID" -Value "$SQLDBSLADomainID"
$SQLDBArray += $SQLDBArrayLine
}
####################################################################
# Section 2 - Performing backup of target DBs (if they exist)
####################################################################
##################################
# Creating array required
##################################
$SQLDBBackupRequests = @()
##################################
# Selecting the DB ID from the SQLDBArray and SLA domain ID from SLADomainList 
##################################
$SQLBackupDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLSourceHost) -AND ($_.DatabaseName -eq $SQLSourceDB))} | Select -ExpandProperty DatabaseID
# Selecting SLA domain ID
$SQLBackupSLADomainID = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupDBID} | Select -ExpandProperty SLADomainID
$SQLBackupSLADomain = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupDBID} | Select -ExpandProperty SLADomain
# If the Source DB exists, but has no SLA, selecting the target DB SLA
IF ($SQLBackupSLADomainID -eq $null)
{
# Selecting source DB ID
$SQLBackupSourceDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.DatabaseName -eq $SQLTargetDB))} | Select -ExpandProperty DatabaseID
# Selecting source SLA
$SQLBackupSLADomainID = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupSourceDBID} | Select -ExpandProperty SLADomainID
}
##################################
# Checking for Required IDs
##################################
IF (($SQLBackupDBID -eq $null) -or ($SQLBackupSLADomainID -eq $null))
{
"--------------------------------------
Starting Backup
SQLDBNotFound: $SQLSourceDB
SQLServer: $SQLSourceHost
"
}
ELSE
{
##################################
# IF DB Existing Taking Backup
##################################
# Creating URL
$SQLDBOnDemandSnapURL = $BaseURL+"mssql/db/"+$SQLBackupDBID+"/snapshot"
# Creating JSON body
$SQLDBOnDemandSnapJSON = 
"{
  ""slaId"": ""$SQLBackupSLADomainID""
}"
# Output to host
"--------------------------------------
Starting Backup
SQLDB: $SQLSourceDB
SQLDBID: $SQLBackupDBID
SQLServer: $SQLSourceHost
SLADomain: $SQLBackupSLADomain
SLADomainID: $SQLBackupSLADomainID"
# Posting to the API
Try 
{
$SQLDBOnDemandSnapResponse = Invoke-RestMethod -Method Post -Uri $SQLDBOnDemandSnapURL -Body $SQLDBOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBOnDemandSnapResponse.links.href
# Adding to array
$SQLDBBackupRequest = New-Object PSObject
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "DB" -Value "$SQLSourceDB"
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value "$SQLJobStatusURL"
$SQLDBBackupRequests += $SQLDBBackupRequest
##################################
# End of bypass for no target DB found below
##################################
}
# End of bypass for no target DB found above
#
##################################
# Getting Job status on a loop, if requests exist
##################################
IF ($SQLDBBackupRequests -ne $null)
{
# Output to host
"--------------------------------------
Checking Backups Complete
"
# Starting status check
ForEach ($SQLDBBackupRequest in $SQLDBBackupRequests)
{
# Setting counter
$SQLJobStatusCount = 0
# Setting variables
$SQLDBBackupRequestDB = $SQLDBBackupRequest.DB
$SQLDBBackupRequestServer = $SQLDBBackupRequest.Server
$SQLDBBackupRequestURL = $SQLDBBackupRequest.JobURL
# Looping status check
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLDBBackupRequestURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
$SQLJobProgress  = $SQLJobStatusResponse.progress
# Setting progress if success, as API returns null
IF ($SQLJobStatus -eq "SUCCEEDED")
{
$SQLJobProgress = "100"
}
# Output to host
"BackupDB: $SQLDBBackupRequestDB 
Server: $SQLDBBackupRequestServer
BackupJob: $SQLJobStatus
Progress: $SQLJobProgress
"
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Will run until it succeeds, fails, or hits 1 hour
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "Failed") -OR ($SQLJobStatusCount -eq 10800))
# End of per job check below
}
# End of per job check above
#
# End of checking backup jobs below
}
# End of checking backup jobs above
####################################################################
# Section 3 - Performing export to refresh target DBs
####################################################################
##################################
# Creating array required
##################################
$SQLDBRefreshRequests = @()
######################
# Converting the paths to usable format for JSON, if not default
######################
# Converting paths if not default to correct format
IF ($DBPath -ne "Default")
{
$DBPath = $DBPath.Replace("\","\\")
}
IF ($LogPath -ne "Default")
{
$LogPath = $LogPath.Replace("\","\\")
}
######################
# Selecting IDs required
######################
# Selecting the database ID
$SQLRefreshSourceDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLSourceHost) -AND ($_.DatabaseName -eq $SQLSourceDB))} | Select -ExpandProperty DatabaseID
# Selecting the target instance ID
$SQLTargetInstanceID = $SQLInstanceArray | Where-Object {($_.Server -eq $SQLTargetHost)} | Select -ExpandProperty InstanceID
# Selecting ID for refresh of host in Rubrik
$SQLTargetHostID = $SQLInstanceArray | Where-Object {($_.Server -eq $SQLTargetHost)} | Select -ExpandProperty HostID
# Selecting target DB ID
$SQLRefreshTargetDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.DatabaseName -eq $SQLTargetDB))} | Select -ExpandProperty DatabaseID
###############################################
# POST to API to Refresh Host before export (prevents errors if DB was only just exported and Rubrik hasn't refreshed yet)
###############################################
$HostRefreshURL = $BaseURL+"host/"+$SQLTargetHostID+"/refresh"
Try 
{
$HostRefreshResponse = Invoke-RestMethod -Uri $HostRefreshURL -Method POST -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Getting DB info
######################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLRefreshSourceDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting latest point in time
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Converting latestRecoveryPoint to values needed
$SQLDBLatestRecoveryPointURL= $SQLDBLatestRecoveryPoint.Replace(":","%3A")
$SQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($SQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
######################
# Getting DB File info
######################
$SQLDBFilesURL = $InternalURL+"mssql/db/"+$SQLRefreshSourceDBID+"/restore_files?time="+$SQLDBLatestRecoveryPointURL
Try 
{
$SQLDBFiles = Invoke-RestMethod -Uri $SQLDBFilesURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Starting JSON build
######################
$SQLDBJSONStart = 
"{
""recoveryPoint"":
  {
  ""timestampMs"":$SQLDBLatestRecoveryPointMS
  },
""targetInstanceId"": ""$SQLTargetInstanceID"",
""targetDatabaseName"": ""$SQLTargetDB"",
""targetFilePaths"":[
"
######################
# Middle JSON build
######################
$SQLDBJSONMiddle = $null
# Selecting random ID to prevent file conflict
$Range = 1..10000
$RandomID = $Range | Get-Random
# Starting counters from ID
$SQLDBFileCounter = 1
$SQLNDFFileCounter = 1
$SQLDBLogFileCounter = 1
# Creating JSON VM array for all the VMs in the VPG
ForEach ($SQLDBFile in $SQLDBFiles)
{
# Setting the DB file variables
$LogicalName = $SQLDBFile.logicalName
$OriginalPath = $SQLDBFile.originalPath
$OriginalName = $SQLDBFile.originalName
# Converting original path
$OriginalPathConverted = $OriginalPath.Replace("\","\\")
######################
# MDF settings
######################
IF ($OriginalName -match ".mdf")
{
# Setting DB path
If ($DBPath -eq "Default")
{
$ExportPath = $OriginalPathConverted
}
ELSE
{
$ExportPath = $DBPath
}
# Setting New DB File Name
$NewFilename = $SQLTargetDB + "_ID" + $RandomID +"_" + $SQLDBFileCounter + ".mdf"
# Incrementing DB counter to prevent duplicates
$SQLDBFileCounter ++
}
######################
# NDF settings
######################
IF ($OriginalName -match ".ndf")
{
# Setting DB path
If ($DBPath -eq "Default")
{
$ExportPath = $OriginalPathConverted
}
ELSE
{
$ExportPath = $DBPath
}
# Setting New DB File Name
$NewFilename = $SQLTargetDB + "_ID" + $RandomID +"_" + $SQLNDFFileCounter + ".ndf"
# Incrementing DB counter to prevent duplicates
$SQLNDFFileCounter ++
}
######################
# LDF settings
######################
IF ($OriginalName -match ".ldf")
{
# Setting DB path
If ($LogPath -eq "Default")
{
$ExportPath = $OriginalPathConverted
}
ELSE
{
$ExportPath = $LogPath
}
# Setting New DB File Name
$NewFilename = $SQLTargetDB +"_ID" + $RandomID + "_Log" + $SQLDBLogFileCounter + ".ldf"
# Incrementing DB counter to prevent duplicates
$SQLDBLogFileCounter ++
}
######################
# Middle JSON
######################
$SQLDBJSONFile = 
"{
""exportPath"":""$ExportPath"",
""logicalName"":""$logicalName"",
""newFilename"":""$NewFilename""
}"
# Running if statement to check if this is the first file in the array, if not then a comma is added to string
IF ($SQLDBJSONMiddle -ne $null)
{
$SQLDBJSONFile = ",
" + $SQLDBJSONFile
}
# Adding the DB file to the JSON middle
$SQLDBJSONMiddle = $SQLDBJSONMiddle + $SQLDBJSONFile
}
######################
# End JSON & Compiling
######################
# Creating the end of the JSON request
$SQLDBJSONEnd = "],
""finishRecovery"": true,
""allowOverwrite"": true
}"
# Putting the JSON request together and outputting the request
$SQLDBJSON = $SQLDBJSONStart + $SQLDBJSONMiddle + $SQLDBJSONEnd
######################
# POST SQL Export
######################
# Creating POST URL
$SQLDBExportURL = $BaseURL+"mssql/db/"+$SQLRefreshSourceDBID+"/export"
# Output to host
"--------------------------------------
Starting Refresh
SQLDB: $SQLTargetDB
SQLServer = $SQLTargetHost"
# POST to API
Try 
{
$SQLDBExportResponse = Invoke-RestMethod -Method Post -Uri $SQLDBExportURL -Body $SQLDBJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBExportResponse.links.href
# Adding to array
$SQLDBRefreshRequest = New-Object PSObject
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "DB" -Value "$SQLTargetDB"
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLTargetHost"
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value "$SQLJobStatusURL"
$SQLDBRefreshRequests += $SQLDBRefreshRequest
##################################
# Getting Job status on a loop
##################################
IF ($SQLDBRefreshRequests -ne $null)
{
# Output to host
"--------------------------------------
Checking Refreshes Complete
"
# Starting for each
ForEach ($SQLDBRefreshRequest in $SQLDBRefreshRequests)
{
# Setting counter
$SQLJobStatusCount = 0
# Setting variables
$SQLDBRefreshRequestDB = $SQLDBRefreshRequest.DB
$SQLDBRefreshRequestServer = $SQLDBRefreshRequest.Server
$SQLDBRefreshRequestURL = $SQLDBRefreshRequest.JobURL
# Looping status check
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLDBRefreshRequestURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
$SQLJobProgress  = $SQLJobStatusResponse.progress
$SQLJobStatusError = $SQLJobStatusResponse.error.message
# Setting progress if success, as API returns null
IF ($SQLJobStatus -eq "SUCCEEDED")
{
$SQLJobProgress = "100"
}
# Output to host
"ExportDB: $SQLDBRefreshRequestDB 
Server: $SQLDBRefreshRequestServer
ExportJob: $SQLJobStatus
Progress: $SQLJobProgress
"
# Showing error if one exists
IF ($SQLJobStatusError -ne $null)
{
"Error: $SQLJobStatusError"
}
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Will run until it succeeds, fails, or hits 1 hour
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "Failed") -OR ($SQLJobStatusCount -eq 10800) -or ($SQLJobStatusError -ne $null))
# End of per job check below
}
# End of per job check above
#
# End of checking backup jobs below
}
# End of checking backup jobs above
##################################
# Perform any next actions you want here
##################################
IF ($SQLJobStatus -eq "SUCCEEDED")
{
# Put your next actions, if needed, here
}
##################################
# End of script
##################################