﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates an on-demand snapshot for the target DB specified, then refreshes it using an export from the source DB
# If the target DB already exists on the server it will use its paths for the export, if it doesn't exist then it will use the paths of the DB with the most backups on the target, if no DBs are protected on the target it will use the source file paths
# If the same Source DB and Server are specified multiple times, it will only take a backup once (so you can export to multiple targets)
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A SQL database ready to protect, it doesn't have to have an existing SLA domain assigned
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
$SQLCSVToProcess = "C:\RubrikSQLAutomationv1\BulkBackupThenRefresh\RubrikSQLBulkExport.csv"
# Max threshold for not exporting (if backup is older than x, don't perform the export)
$MaxBackupThresholdHours = 12
# Average throughput for export
$AverageExportThroughputMBSec = 125
# HTML Color codes used for report rows
$HTMLColorSuccess = "#000000"
$HTMLColorFailure = "#e60000"
# Email settings
$EmailTo = "joshua@lab.local"
$EmailFrom = "rubrik@lab.local"
$EmailServer = "localhost"
# Email subject
$EmailSubject1 = "Rubrik SQLDB Export Progress"
$EmailSubject2 = "Rubrik SQLDB Export Report"
# Logo work
$LogoFile = "C:\Program Files\RubrikPolarisLocalv1\Reports\Images\Logo.png"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing CSV & Sorting
##################################
$SQLCSVImport = Import-Csv $SQLCSVToProcess
# Getting unique DBs to backup
$SQLDBsToBackup = $SQLCSVImport | Select-Object @{Label = "SourceHostAndDB"; Expression = {"$($_.'SourceHost') $($_.'SourceDB')"} } -Unique
# Getting all DBs to export 
$SQLDBsToExport = $SQLCSVImport
# Timestamp
$SystemDateTime = Get-Date
################################################
# Creating the Convert-UNIXTime function
################################################
Function Convert-UNIXTime {
Param ($UNIXTime)

# Example: $PSTime = Convert-UNIXTime -UNIXTime $UNIXTime

# Step 1 - Removing characters and trimming snapshot string for conversion
$PSTimeStep1 = $UNIXTime.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Removing last 4 characters
$PSTimeStep2 = $PSTimeStep1.Substring(0,$PSTimeStep1.Length-4)
# Step 3 - Converting string to PowerShell datetime object
$PSTimeStep3 = ([datetime]::ParseExact($PSTimeStep2,”yyyy-MM-dd HH:mm:ss”,$null))
# Returning Result
Return $PSTimeStep3
}
####################################################################
# SMTP Body - HTML Email style settings
####################################################################
# Start of HTML structure
$HTMLStart = @"
<!DOCTYPE html>
<html>
<head>
<style>

a {
    color: black;
}

a:link {
    text-decoration: none;
}

table.table1 {
  border-collapse: collapse;
  width: 100%;
}
table.table1 th {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #696969;
  font-size:16px
}
table.table1 td {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

table.table2 {
  border-collapse: collapse;
  width: 100%;
}
table.table2 th {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #00B2A9;
  font-size:14px
}
table.table2 td {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

</style>
</head>
<body>

<div style="overflow-x:auto;">

<table style="text-align: right; width: 100%;" border="0"; cellpadding="2"; cellspacing="2">
<tbody>
<tr>
<td style="vertical-align: top;"><img style="width: 350px; height: 95px;" alt="Logo" src="logo.png"><br>
</td>
</tr>
</tbody>
</table>

<br></br>
"@
# End of HTML structure
$HTMLEnd = @"
</div>

</body>
</html>
"@
####################################################################
# Section 1 - Connecting to API and building table of DBs
####################################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainListURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainListJSON = Invoke-RestMethod -Uri $SLADomainListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainList = $SLADomainListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000&is_relic=false"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
##################################
"Updating List of All SQL DBs Prior To Starting Operations....."
# Building array
$SQLInstanceArray = @()
# Starting for Each
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Combining server name and instance
$SQLInstanceServer = $SQLInstanceHostName + "\" + $SQLInstanceName 
# Adding to array
$SQLInstanceArrayLine = New-Object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
$SQLDBCounter = 0
$SQLDBCount = $SQLDBList | Measure | Select -ExpandProperty Count
ForEach ($SQLDB in $SQLDBList)
{
# Incrementing counter
$SQLDBCounter ++
# Output to host
"SQLDB: $SQLDBCounter / $SQLDBCount"
# Setting variables
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
$SQLDBSLADomain = $SQLDB.effectiveSlaDomainName
$SQLDBSLADomainID = $SQLDB.effectiveSlaDomainId
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
$SQLInstanceServer = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Server
}
# Adding to array
$SQLDBArrayLine = New-Object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SQLDBSLADomain"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomainID" -Value "$SQLDBSLADomainID"
$SQLDBArray += $SQLDBArrayLine
}
####################################################################
# Section 2 - Performing backup of target DBs (if they exist)
####################################################################
# Output to host
"--------------------------------------
Starting Backups"
##################################
# Creating array required
##################################
$SQLDBBackupRequests = @()
$SQLDBBackupResults = @()
##################################
# Starting For Each DB To Backup
##################################
ForEach ($SQLDBToBackup in $SQLDBsToBackup)
{
# Setting variable
$SQLDBToSplit = $SQLDBToBackup | Select -ExpandProperty SourceHostAndDB
# Getting Source Host/DB from unique splits
$SQLSourceHost = $SQLDBToSplit.Split(" ")[0]
$SQLSourceDB = $SQLDBToSplit.Split(" ")[1]
##################################
# Selecting the DB ID from the SQLDBArray and SLA domain ID from SLADomainList 
##################################
$SQLBackupDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLSourceHost) -AND ($_.DatabaseName -eq $SQLSourceDB))} | Select -ExpandProperty DatabaseID
# Output to Host
"--------------------------------------
ProcessingBackup:
SourceDB: $SQLSourceDB
SQLDBID: $SQLBackupDBID
SourceHost: $SQLSourceHost"
# Selecting SLA domain ID
$SQLBackupSLADomainID = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupDBID} | Select -ExpandProperty SLADomainID
$SQLBackupSLADomain = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupDBID} | Select -ExpandProperty SLADomain
# If the Source DB exists, but has no SLA, selecting the target DB SLA
IF ($SQLBackupSLADomainID -eq $null)
{
# Selecting source DB ID
$SQLBackupSourceDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.DatabaseName -eq $SQLTargetDB))} | Select -ExpandProperty DatabaseID
# Selecting source SLA
$SQLBackupSLADomainID = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupSourceDBID} | Select -ExpandProperty SLADomainID
}
##################################
# Checking for Required IDs
##################################
IF (($SQLBackupDBID -eq $null) -or ($SQLBackupSLADomainID -eq $null))
{
"
ERROR-Skipping:

SourceDBNotFound: $SQLSourceDB
SourceHost: $SQLSourceHost
"
# Adding to array
$SQLDBBackupResult = New-Object PSObject
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBBackupRequestDB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBBackupRequestHost
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value "DBIDNotFound"
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $Null
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "End" -Value $Null
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $Null
$SQLDBBackupResults += $SQLDBBackupResult
}
ELSE
{
##################################
# IF DB Existing Taking Backup
##################################
# Creating URL
$SQLDBOnDemandSnapURL = $BaseURL+"mssql/db/"+$SQLBackupDBID+"/snapshot"
# Creating JSON body
$SQLDBOnDemandSnapJSON = 
"{
  ""slaId"": ""$SQLBackupSLADomainID""
}"
# Output to host
"StartingBackupWith:
SLADomain: $SQLBackupSLADomain
SLADomainID: $SQLBackupSLADomainID"
# Posting to the API
Try 
{
$SQLDBOnDemandSnapResponse = Invoke-RestMethod -Method Post -Uri $SQLDBOnDemandSnapURL -Body $SQLDBOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBOnDemandSnapResponse.links.href
# Adding to array
$SQLDBBackupRequest = New-Object PSObject
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLSourceDB
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "SourceDBID" -Value $SQLBackupDBID
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLSourceHost
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value $SQLJobStatusURL
$SQLDBBackupRequests += $SQLDBBackupRequest
##################################
# End of bypass for no target DB found below
##################################
}
# End of bypass for no target DB found above
##################################
# End of for each SQL DB below
##################################
}
# End of for each SQL DB above
#
##################################
# Getting Job status on a loop, if requests exist
##################################
IF ($SQLDBBackupRequests -ne $null)
{
# Output to host
"--------------------------------------
Checking Backups Complete
--------------------------------------"
# Starting status check
ForEach ($SQLDBBackupRequest in $SQLDBBackupRequests)
{
# Setting counter
$SQLJobStatusCount = 0
# Setting variables
$SQLDBBackupRequestDB = $SQLDBBackupRequest.SourceDB
$SQLDBBackupRequestDBID = $SQLDBBackupRequest.SourceDBID
$SQLDBBackupRequestHost = $SQLDBBackupRequest.SourceHost
$SQLDBBackupRequestURL = $SQLDBBackupRequest.JobURL
# Getting short ID
$SQLDBBackupRequestDBIDShort = $SQLDBBackupRequestDBID.Replace("MssqlDatabase:::","")
# Looping status check
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLDBBackupRequestURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
$SQLJobProgress  = $SQLJobStatusResponse.progress
$SQLJobStartUNIX = $SQLJobStatusResponse.startTime
$SQLJobEndUNIX = $SQLJobStatusResponse.endTime
# Setting progress if success, as API returns null
IF ($SQLJobStatus -eq "SUCCEEDED")
{
$SQLJobProgress = "100"
}
# Output to host
"BackupDB: $SQLDBBackupRequestDB 
Host: $SQLDBBackupRequestHost
BackupJob: $SQLJobStatus
Progress: $SQLJobProgress
"
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Getting error if FAILED
IF ($SQLJobStatus -eq "FAILED")
{
$SQLJobError = $SQLJobStatusResponse.error.message
}
ELSE
{
$SQLJobError = $null
}
# Will run until it succeeds, fails, or hits 1 hour
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "Failed") -OR ($SQLJobStatusCount -eq 10800))
####################
# Getting event data
####################
$SQLJobInstanceID = $SQLJobStatusResponse.id
# Building URL to get event data
$SQLJobEventURL = $InternalURL + "event?event_type=Backup&object_ids=MssqlDatabase%3A%3A%3A" + $SQLDBBackupRequestDBIDShort + "&show_only_latest=true&filter_only_on_latest=true"
# Getting event
Try 
{
$SQLJobEventJSON = Invoke-RestMethod -Method GET -Uri $SQLJobEventURL -Headers $RubrikSessionHeader
$SQLJobEvent = $SQLJobEventJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# Filtering to ensure matching live mount request
$SQLJobEventFiltered = $SQLJobEvent | Where-Object {$_.JobInstanceID -eq $SQLJobInstanceID}
# Getting event series ID
$SQLJobEventSeriesID = $SQLJobEventFiltered.eventSeriesID
####################
# Getting Event Series
####################
# Building URL to get event series data
$SQLJobEventSeriesURL = $InternalURL + "event_series/" + $SQLJobEventSeriesID
# Getting event series
Try 
{
$SQLJobEventSeries = Invoke-RestMethod -Method GET -Uri $SQLJobEventSeriesURL -Headers $RubrikSessionHeader
$SQLJobEventSeriesDetail = $SQLJobEventSeries.eventDetailList
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# Setting variables
$SQLJobDataStatus = $SQLJobEventSeries.status
# Only getting data if success
IF ($SQLJobDataStatus -eq "Success")
{
# Getting data
$SQLJobDataTransferredBytes = $SQLJobEventSeries.dataTransferred
$SQLJobDataLogicalSizeBytes = $SQLJobEventSeries.logicalSize
# Converting bytes
$SQLJobDataTransferredMB = $SQLJobDataTransferredBytes / 1024 / 1024 
$SQLJobDataTransferredGB = $SQLJobDataTransferredBytes / 1024 / 1024 / 1024
$SQLJobDataLogicalSizeMB = $SQLJobDataLogicalSizeBytes / 1024 / 1024 
$SQLJobDataLogicalSizeGB = $SQLJobDataLogicalSizeBytes / 1024 / 1024 / 1024
# Rounding
$SQLJobDataTransferredGB = [Math]::Round($SQLJobDataTransferredGB,3)
$SQLJobDataLogicalSizeGB = [Math]::Round($SQLJobDataLogicalSizeGB,2)
# Getting time run in seconds to calculate throughput
$SQLJobDataTransferStartUNIX = $SQLJobEventSeries.startTime 
$SQLJobDataTransferEndUNIX = $SQLJobEventSeries.endTime 
# Converting times 
$SQLJobDataTransferStart = Convert-UNIXTime -UNIXTime $SQLJobDataTransferStartUNIX
$SQLJobDataTransferEnd = Convert-UNIXTime -UNIXTime $SQLJobDataTransferEndUNIX
# Getting time in seconds
$SQLJobDataTimeSeconds = New-TimeSpan -Start $SQLJobDataTransferStart -End $SQLJobDataTransferEnd | Select -ExpandProperty TotalSeconds
# Calculating throughput
$SQLJobDataThroughputMB = $SQLJobDataTransferredMB / $SQLJobDataTimeSeconds 
$SQLJobDataThroughputMB = [Math]::Round($SQLJobDataThroughputMB,4)
# Calculating estimated export time
$SQLJobEstimatedExportSeconds = $SQLJobDataLogicalSizeMB / $AverageExportThroughputMBSec
$SQLJobEstimatedExportSeconds = [Math]::Round($SQLJobEstimatedExportSeconds,0)
# Converting to hours
$SQLJobEstimatedExportHours = $SQLJobEstimatedExportSeconds / 60 / 60
$SQLJobEstimatedExportHours = [Math]::Round($SQLJobEstimatedExportHours,2)
}
ELSE
{
# Nulling data fields as job was not successful
$SQLJobDataTransferredGB = $null
$SQLJobDataThroughputMB = $null
$SQLJobDataLogicalSizeGB = $null
}
##################################
# Adding To Array
##################################
# Converting time
$SQLJobStart = Convert-UNIXTime -UNIXTime $SQLJobStartUNIX
# Only converting if end time exists
IF ($SQLJobEndUNIX -ne $null)
{
$SQLJobEnd = Convert-UNIXTime -UNIXTime $SQLJobEndUNIX
# Getting time span
$SQLJobTimeTakenSeconds = New-Timespan -Start $SQLJobStart -End $SQLJobEnd | Select -ExpandProperty TotalSeconds
}
ELSE
{
$SQLJobTimeTakenSeconds = $null
}
# Adding to array
$SQLDBBackupResult = New-Object PSObject
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBBackupRequestDB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBBackupRequestHost
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SizeGB" -Value $SQLJobDataLogicalSizeGB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value $SQLJobStatus
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $SQLJobStart
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "End" -Value $SQLJobEnd
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $SQLJobTimeTakenSeconds
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "TransferredGB" -Value $SQLJobDataTransferredGB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "ThroughputMB" -Value $SQLJobDataThroughputMB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "EstExportSeconds" -Value $SQLJobEstimatedExportSeconds
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "Details" -Value $SQLJobError
$SQLDBBackupResults += $SQLDBBackupResult
# End of per job check below
}
# End of per job check above
#
# End of checking backup jobs below
}
# End of checking backup jobs above
####################################################################
# Section 3 - Performing export to refresh target DBs
####################################################################
# Output to host
"--------------------------------------
Starting Refresh/Exports"
##################################
# Creating array required
##################################
$SQLDBRefreshRequests = @()
$SQLDBRefreshResults = @()
##################################
# Starting For Each DB to Export
##################################
ForEach ($SQLDBToExport in $SQLDBsToExport)
{
# Setting variables
$SQLSourceHost = $SQLDBToExport.SourceHost
$SQLSourceDB = $SQLDBToExport.SourceDB
$SQLTargetHost = $SQLDBToExport.TargetHost
$SQLTargetDB = $SQLDBToExport.TargetDB
# Output to host
"--------------------------------------
ProcessingExport/Refresh
TargetSQLDB: $SQLTargetDB
TargetSQLServer : $SQLTargetHost
SourceSQLDB: $SQLSourceDB
SourceSQLServer : $SQLSourceHost"
######################
# Selecting IDs required
######################
# Selecting the database ID
$SQLRefreshSourceDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLSourceHost) -AND ($_.DatabaseName -eq $SQLSourceDB))} | Select -ExpandProperty DatabaseID
# Selecting the target instance ID
$SQLTargetInstanceID = $SQLInstanceArray | Where-Object {($_.Server -eq $SQLTargetHost)} | Select -ExpandProperty InstanceID
# Selecting ID for refresh of host in Rubrik
$SQLTargetHostID = $SQLInstanceArray | Where-Object {($_.Server -eq $SQLTargetHost)} | Select -ExpandProperty HostID
######################
# Bypass if IDs not found
######################
IF (($SQLRefreshSourceDBID -eq $null) -or ($SQLTargetInstanceID -eq $null))
{
# Adding request to array
$SQLDBRefreshRequest = New-Object PSObject
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value "SourceDBIDorTargetInstanceNotFound"
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceDBID" -Value $SQLRefreshSourceDBID
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLSourceHost
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLTargetDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLTargetHost
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetHostID" -Value $SQLTargetHostID
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetInstanceID" -Value $SQLTargetInstanceID
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "Timestamp" -Value $null
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "AgeInHours" -Value $null
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "ExportAction" -Value $null
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "Status" -Value "FAILED"
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value $SQLJobStatusURL
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $SQLRefreshTargetDBFilePathsFrom
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $SQLRefreshTargetDBFilePathsFromDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $DBPath
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $DBPath
$SQLDBRefreshRequests += $SQLDBRefreshRequest
# Adding result to array
$SQLDBRefreshResult = New-Object PSObject
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLDBRefreshRequestDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLDBRefreshRequestHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetHostID" -Value $SQLDBRefreshRequestHostID
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetInstanceID" -Value $SQLDBRefreshRequestInstanceID
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBRefreshRequestSourceDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceDBID" -Value $SQLDBRefreshRequestSourceDBID
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBRefreshRequestSourceHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value "FAILED"
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "End" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $SQLJobTimeTakenSeconds
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Details" -Value "DBorINSTANCEIDNotFound"
}
ELSE
{
###############################################
# POST to API to Refresh Host before export (prevents errors if DB was only just exported and Rubrik hasn't refreshed yet)
###############################################
$HostRefreshURL = $BaseURL+"host/"+$SQLTargetHostID+"/refresh"
Try 
{
$HostRefreshResponse = Invoke-RestMethod -Uri $HostRefreshURL -Method POST -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Getting Source DB info
######################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLRefreshSourceDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting latest point in time
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Converting latestRecoveryPoint to values needed
$SQLDBLatestRecoveryPointURL= $SQLDBLatestRecoveryPoint.Replace(":","%3A")
$SQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($SQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
# Converting to PS date time format
$SQLDBLatestRecoveryPointTime = Convert-UNIXTime -UNIXTime $SQLDBLatestRecoveryPoint
######################
# Getting DB File info
######################
$UTCDateTime = [System.DateTime]::UtcNow
$SQLDBRecoveryPointTimespan = New-TimeSpan -Start $SQLDBLatestRecoveryPointTime -End $UTCDateTime 
$SQLDBRecoveryPointAgeHours = $SQLDBRecoveryPointTimespan | Select -ExpandProperty TotalHours
# Rounding
$SQLDBRecoveryPointAgeHours = [Math]::Round($SQLDBRecoveryPointAgeHours,2)
# Deciding action based on age of backup
IF ($SQLDBRecoveryPointAgeHours -ge $MaxBackupThresholdHours)
{
$SQLDBRecoveryPerformExport = $FALSE
}
ELSE
{
$SQLDBRecoveryPerformExport = $TRUE
}
# Output to host
"BackupAgeHours: $SQLDBRecoveryPointAgeHours
PerformingExport: $SQLDBRecoveryPerformExport"
######################
# Getting DB File info
######################
$SQLDBFilesURL = $InternalURL+"mssql/db/"+$SQLRefreshSourceDBID+"/restore_files?time="+$SQLDBLatestRecoveryPointURL
Try 
{
$SQLDBFiles = Invoke-RestMethod -Uri $SQLDBFilesURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Using source paths
######################
$SQLRefreshTargetDBFilePathsFrom = "Source"
$SQLRefreshTargetDBFilePathsFromDB = $SQLSourceDB
######################
# Default File Path Selection Section from file structure on Target DB
######################
$DBPath = $SQLDBFiles | Where-Object {$_.originalName -match ".mdf"} | Select -ExpandProperty originalPath -First 1
$LogPath =  $SQLDBFiles | Where-Object {$_.originalName -match ".ldf"} | Select -ExpandProperty originalPath -First 1
# Output to host
"UsingPathFrom: $SQLRefreshTargetDBFilePathsFrom ($SQLRefreshTargetDBFilePathsFromDB)
DBPath: $DBPath 
LogPath: $LogPath"
# Converting
$DBPath = $DBPath.Replace("\","\\")
$LogPath = $LogPath.Replace("\","\\")
######################
# Starting JSON build
######################
$SQLDBJSONStart = 
"{
""recoveryPoint"":
  {
  ""timestampMs"":$SQLDBLatestRecoveryPointMS
  },
""targetInstanceId"": ""$SQLTargetInstanceID"",
""targetDatabaseName"": ""$SQLTargetDB"",
""targetFilePaths"":[
"
######################
# Middle JSON build
######################
$SQLDBJSONMiddle = $null
# Selecting random ID to prevent file conflict
$Range = 1..10000
$RandomID = $Range | Get-Random
# Starting counters from ID
$SQLDBFileCounter = 1
$SQLNDFFileCounter = 1
$SQLDBLogFileCounter = 1
# Creating JSON VM array for all the VMs in the VPG
ForEach ($SQLDBFile in $SQLDBFiles)
{
# Setting the DB file variables
$LogicalName = $SQLDBFile.logicalName
$OriginalPath = $SQLDBFile.originalPath
$OriginalName = $SQLDBFile.originalName
# Converting original path
$OriginalPathConverted = $OriginalPath.Replace("\","\\")
######################
# MDF settings
######################
IF ($OriginalName -match ".mdf")
{
# Setting DB path
$ExportPath = $DBPath
# Setting New DB File Name
$NewFilename = $SQLTargetDB + "_ID" + $RandomID +"_" + $SQLDBFileCounter + ".mdf"
# Incrementing DB counter to prevent duplicates
$SQLDBFileCounter ++
}
######################
# NDF settings
######################
IF ($OriginalName -match ".ndf")
{
# Setting DB path
$ExportPath = $DBPath
# Setting New DB File Name
$NewFilename = $SQLTargetDB + "_ID" + $RandomID +"_" + $SQLNDFFileCounter + ".ndf"
# Incrementing DB counter to prevent duplicates
$SQLNDFFileCounter ++
}
######################
# LDF settings
######################
IF ($OriginalName -match ".ldf")
{
# Setting DB path
$ExportPath = $LogPath
# Setting New DB File Name
$NewFilename = $SQLTargetDB +"_ID" + $RandomID + "_Log" + $SQLDBLogFileCounter + ".ldf"
# Incrementing DB counter to prevent duplicates
$SQLDBLogFileCounter ++
}
######################
# Middle JSON
######################
$SQLDBJSONFile = 
"{
""exportPath"":""$ExportPath"",
""logicalName"":""$logicalName"",
""newFilename"":""$NewFilename""
}"
# Running if statement to check if this is the first file in the array, if not then a comma is added to string
IF ($SQLDBJSONMiddle -ne $null)
{
$SQLDBJSONFile = ",
" + $SQLDBJSONFile
}
# Adding the DB file to the JSON middle
$SQLDBJSONMiddle = $SQLDBJSONMiddle + $SQLDBJSONFile
}
######################
# End JSON & Compiling
######################
# Creating the end of the JSON request
$SQLDBJSONEnd = "],
""finishRecovery"": true,
""allowOverwrite"": true
}"
# Putting the JSON request together and outputting the request
$SQLDBJSON = $SQLDBJSONStart + $SQLDBJSONMiddle + $SQLDBJSONEnd
######################
# POST SQL Export, if backup is within max hours
######################
IF ($SQLDBRecoveryPerformExport -eq $TRUE)
{
# Creating POST URL
$SQLDBExportURL = $BaseURL+"mssql/db/"+$SQLRefreshSourceDBID+"/export"
# POST to API
Try 
{
$SQLDBExportResponse = Invoke-RestMethod -Method Post -Uri $SQLDBExportURL -Body $SQLDBJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBExportRequest = "InProgress"
}
Catch 
{
$SQLDBExportRequest = "FAILED"
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
}
ELSE
{
$SQLDBExportRequest = "NOTRUN"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBExportResponse.links.href
# Adding to array
$SQLDBRefreshRequest = New-Object PSObject
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLSourceDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceDBID" -Value $SQLRefreshSourceDBID
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLSourceHost
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLTargetDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLTargetHost
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetHostID" -Value $SQLTargetHostID
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetInstanceID" -Value $SQLTargetInstanceID
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "Timestamp" -Value $SQLDBLatestRecoveryPoint
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "AgeInHours" -Value $SQLDBRecoveryPointAgeHours
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "ExportAction" -Value $SQLDBRecoveryPerformExport
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "Status" -Value $SQLDBExportRequest
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value $SQLJobStatusURL
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $SQLRefreshTargetDBFilePathsFrom
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $SQLRefreshTargetDBFilePathsFromDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $DBPath
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $LogPath
$SQLDBRefreshRequests += $SQLDBRefreshRequest
######################
# End of bypass if IDs not found below
######################
}
# End of bypass if IDs not found above
##################################
# End of for Each DB to Export below
##################################
}
# End of for Each DB to Export above
####################################################################
# Section 4 - Sending Email 1 (midpoint)
####################################################################
##################################
# Getting totals
##################################
$TotalBackupsRequested = $SQLDBBackupResults | Measure | Select -ExpandProperty Count
$TotalBackupsSuccess = $SQLDBBackupResults | Where-Object {$_.JobStatus -eq "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalBackupsFail = $SQLDBBackupResults | Where-Object {$_.JobStatus -eq "FAILED"} | Measure | Select -ExpandProperty Count
$TotalBackupsNotRun = $SQLDBBackupResults | Where-Object {$_.JobStatus -eq "NOTRUN"} | Measure | Select -ExpandProperty Count
$TotalExportsRequested = $SQLDBRefreshRequests | Measure | Select -ExpandProperty Count
$TotalExportsInProgress = $SQLDBRefreshRequests | Where-Object {$_.Status -eq "InProgress"} | Measure | Select -ExpandProperty Count
$TotalExportsFailed = $SQLDBRefreshRequests | Where-Object {$_.Status -eq "FAILED"} | Measure | Select -ExpandProperty Count
# Backup timing
$BackupStart = $SQLDBBackupResults | Sort-Object Start -Descending | Select -ExpandProperty Start -First 1
$BackupEnd = $SQLDBBackupResults | Sort-Object End -Descending | Select -ExpandProperty End -First 1
$BackupMinutesTaken = New-Timespan -Start $BackupStart -End $BackupEnd | Select -ExpandProperty TotalMinutes
# Rounding
$BackupMinutesTaken = [Math]::Round($BackupMinutesTaken,0)
# Getting longest estimation
$LongestExportSeconds = $SQLDBBackupResults | Sort-Object {$_.EstExportSeconds -as [int]} -Descending | Select -ExpandProperty EstExportSeconds -First 1
$LongestExportHours = $LongestExportSeconds / 60 / 60
$LongestExportHours = [Math]::Round($LongestExportHours,2)
# Getting estimated end time
$Now = Get-Date
$EstimatedEnd = $Now.AddSeconds($LongestExportSeconds)
# Largest DB
$LargestSizeDBSizeGB = $SQLDBBackupResults | Sort-Object {$_.SizeGB -as [int]} -Descending | Select -ExpandProperty SizeGB -First 1
$LargestSizeDBName = $SQLDBBackupResults | Sort-Object {$_.SizeGB -as [int]} -Descending | Select -ExpandProperty SourceDB -First 1
##################################
# Creating HTML Summary table
##################################
$HTML1SummaryTable = @"
<table class="table1">
  <tr>
    <th>Created $SystemDateTime</th>
  </tr>
  <tr>
    <th>SQL Refresh Progress</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>Exports</th>
    <td>$TotalExportsRequested</td>
    <th>Backups</th>
    <td>$TotalBackupsRequested</td>
    <th>Largest</th>
    <td>$LargestSizeDBName</td>
  </tr>
 <tr>
    <th>InProgress</th>
    <td>$TotalExportsInProgress</td>
    <th>Success</th>
    <td>$TotalBackupsSuccess</td>
    <th>SizeGB</th>
    <td>$LargestSizeDBSizeGB</td>
 </tr>
 <tr>
    <th>Failed</th>
    <td>$TotalExportsFailed</td>
    <th>Fail</th>
    <td>$TotalBackupsFail</td>
    <th>EstimatedHours</th>
    <td>$LongestExportHours</td>
 </tr>
  <tr>
    <th>NotRun</th>
    <td>$TotalBackupsNotRun</td>
    <th>Minutes </th>
    <td>$BackupMinutesTaken</td>
    <th>EstimatedEnd</th>
    <td>$EstimatedEnd</td>
 </tr>
 </table>
 <br>
"@
##################################
# Creating Table 1 HTML structure
##################################
$HTML1Table1Start = @"
<table class="table1">
  <tr>
    <th>SQL DB Export Requests</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>SourceHost</th>
    <th>SourceDB</th>
    <th>TargetHost</th>
    <th>TargetDB</th>
    <th>Perform<br>Export</th>
    <th>Export<br>Status</th>
    <th>Size<br>GB</th>
    <th>Estimated<br>Hours</th>
    <th>Timestamp<br>Exported</th>
    <th>BackupAge<br>InHours</th>
    <th>MaxAge<br>Hours</th>
    <th>Backup<br>Status</th>
    <th>Backup<br>Hours</th>
  </tr>
"@
$HTML1Table1End = @"
</table>
<br>
"@
##################################
# Creating Table 1 HTML Rows
##################################
# Building email list by task to put the most important objects for viewing at the top
$Table1Data = $SQLDBRefreshRequests | Sort-Object TargetDB,TargetHost,SourceDB,SourceHost
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTML1ReportTable1Middle = $null
# Creating table row for each line
ForEach ($Row in $Table1Data) 
{
# Setting values
$HTML1SourceHost = $Row.SourceHost
$HTML1SourceDB = $Row.SourceDB
$HTML1TargetHost = $Row.TargetHost
$HTML1TargetDB = $Row.TargetDB
$HTML1ExportTimestamp = $Row.Timestamp
$HTML1ExportAgeInHours = $Row.AgeInHours
$HTML1ExportAction = $Row.ExportAction
$HTML1ExportStatus = $Row.Status
$HTML1DBPath = $Row.PathDBFiles
$HTML1LogPath = $Row.PathLogFiles
# Selecting backup data
$SQLDBBackupResult = $SQLDBBackupResults | Where-Object {(($_.SourceHost -eq $HTML1SourceHost) -AND ($_.SourceDB -eq $HTML1SourceDB))}
$HTML1Backup = $SQLDBBackupResult.JobStatus
$HTML1BackupSeconds = $SQLDBBackupResult.TimeTakenSeconds
$HTML1DBSizeGB = $SQLDBBackupResult.SizeGB
$HTML1EstExportSeconds = $SQLDBBackupResult.EstExportSeconds
# Converting seconds to hours
$HTML1EstExportHours = $HTML1EstExportSeconds / 60 / 60
$HTML1EstExportHours = [Math]::Round($HTML1EstExportHours,2)
$HTML1BackupHours = $HTML1BackupSeconds / 60 / 60
$HTML1BackupHours = [Math]::Round($HTML1BackupHours,2)
# Building HTML table row
$HTMLReportTable1Row = "
<tr>
    <td>$HTML1SourceHost</td>
    <td>$HTML1SourceDB</td>
    <td>$HTML1TargetHost</td>
    <td>$HTML1TargetDB</td>
    <td>$HTML1ExportAction</td>
    <td>$HTML1ExportStatus</td>
    <td>$HTML1DBSizeGB</td>
    <td>$HTML1EstExportHours</td>
    <td>$HTML1ExportTimestamp</td>
    <td>$HTML1ExportAgeInHours</td>
    <td>$MaxBackupThresholdHours</td>
    <td>$HTML1Backup</td>
    <td>$HTML1BackupHours</td>
  </tr>
"
# Adding row to table
$HTML1ReportTable1Middle += $HTMLReportTable1Row
}
##################################
# Putting Table 1 together
##################################
$HTML1Table1 = $HTML1Table1Start + $HTML1ReportTable1Middle + $HTML1Table1End
##################################
# Creating Report
##################################
# Building HTML report:
$HTML1Report = $HTMLStart + $HTML1SummaryTable + $HTML1Table1 + $HTMLEnd
# Replacing strings for easier reading
$HTML1Report = $HTML1Report.Replace("\\","\").TrimEnd()
##################################
# Sending email using function
##################################
# Building attachment
$Attachments = "$LogoFile"
# Sending email
Try
{
Send-MailMessage -To $EmailTo -From $EmailFrom -Body $HTML1Report -Subject $EmailSubject1 -Attachments $Attachments -SmtpServer $EmailServer -BodyAsHtml 
}
Catch
{
$Error[0] | Format-List -Force
}
####################################################################
# Section 5 - Waiting for all Refresh Jobs to Complete
####################################################################
##################################
# Getting Job status on a loop
##################################
IF ($SQLDBRefreshRequests -ne $null)
{
# Output to host
"--------------------------------------
Checking Refreshes Complete
--------------------------------------"
# Starting for each
ForEach ($SQLDBRefreshRequest in $SQLDBRefreshRequests)
{
# Setting counter
$SQLJobStatusCount = 0
# Setting variables
$SQLDBRefreshRequestDB = $SQLDBRefreshRequest.TargetDB
$SQLDBRefreshRequestHost = $SQLDBRefreshRequest.TargetHost
$SQLDBRefreshRequestHostID = $SQLDBRefreshRequest.TargetHostID
$SQLDBRefreshRequestInstanceID = $SQLDBRefreshRequest.TargetInstanceID
$SQLDBRefreshRequestSourceDB = $SQLDBRefreshRequest.SourceDB
$SQLDBRefreshRequestSourceDBID = $SQLDBRefreshRequest.SourceDBID
$SQLDBRefreshRequestSourceHost = $SQLDBRefreshRequest.SourceHost
$SQLDBRefreshRequestURL = $SQLDBRefreshRequest.JobURL
$SQLDBRereshRequestPathSource = $SQLDBRefreshRequest.PathSource
$SQLDBRereshRequestPathDB = $SQLDBRefreshRequest.PathDB
$SQLDBRereshRequestPathDBFiles = $SQLDBRefreshRequest.PathDBFiles
$SQLDBRereshRequestPathLogFiles = $SQLDBRefreshRequest.PathLogFiles
# Getting short ID
$SQLDBRefreshRequestSourceDBIDShort = $SQLDBRefreshRequestSourceDBID.Replace("MssqlDatabase:::","")
##################################
# Looping status check
##################################
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLDBRefreshRequestURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
$SQLJobProgress  = $SQLJobStatusResponse.progress
$SQLJobStatusError = $SQLJobStatusResponse.error.message
$SQLJobStartUNIX = $SQLJobStatusResponse.startTime
$SQLJobEndUNIX = $SQLJobStatusResponse.endTime
# Setting progress if success, as API returns null
IF ($SQLJobStatus -eq "SUCCEEDED")
{
$SQLJobProgress = "100"
}
# Output to host
"TargetDB: $SQLDBRefreshRequestDB 
TargetHost: $SQLDBRefreshRequestHost
ExportJob: $SQLJobStatus
Progress: $SQLJobProgress
"
# Showing error if one exists
IF ($SQLJobStatusError -ne $null)
{
"Error: $SQLJobStatusError"
}
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Getting error message
$SQLJobError = $SQLJobStatusResponse.error.message
# Getting error if FAILED
IF ($SQLJobError -ne $null)
{
$SQLJobStatus = "FAILED"
}
# Will run until it succeeds, fails, or hits 1 hour
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "FAILED") -OR ($SQLJobStatusCount -eq 10800) -or ($SQLJobStatusError -ne $null))
####################
# Getting event data
####################
$SQLJobInstanceID = $SQLJobStatusResponse.id
# Building URL to get event data
$SQLJobEventURL = $InternalURL + "event?event_type=Recovery&object_ids=MssqlDatabase%3A%3A%3A" + $SQLDBRefreshRequestSourceDBIDShort + "&show_only_latest=true&filter_only_on_latest=true"
# Getting event
Try 
{
$SQLJobEventJSON = Invoke-RestMethod -Method GET -Uri $SQLJobEventURL -Headers $RubrikSessionHeader
$SQLJobEvent = $SQLJobEventJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# Filtering to ensure matching live mount request
$SQLJobEventFiltered = $SQLJobEvent | Where-Object {$_.JobInstanceID -eq $SQLJobInstanceID}
# Getting event series ID
$SQLJobEventSeriesID = $SQLJobEventFiltered.eventSeriesID
####################
# Getting Event Series
####################
# Building URL to get event series data
$SQLJobEventSeriesURL = $InternalURL + "event_series/" + $SQLJobEventSeriesID
# Getting event series
Try 
{
$SQLJobEventSeries = Invoke-RestMethod -Method GET -Uri $SQLJobEventSeriesURL -Headers $RubrikSessionHeader
$SQLJobEventSeriesDetail = $SQLJobEventSeries.eventDetailList
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
# Setting variables
$SQLJobDataStatus = $SQLJobEventSeries.status
# Only getting data if success
IF ($SQLJobDataStatus -eq "Success")
{
# Getting job time
$SQLJobDataTransferStartUNIX = $SQLJobEventSeries.startTime 
$SQLJobDataTransferEndUNIX = $SQLJobEventSeries.endTime 
# Converting times 
$SQLJobDataTransferStart = Convert-UNIXTime -UNIXTime $SQLJobDataTransferStartUNIX
$SQLJobDataTransferEnd = Convert-UNIXTime -UNIXTime $SQLJobDataTransferEndUNIX
# Getting time in seconds
$SQLJobDataTimeSeconds = New-TimeSpan -Start $SQLJobDataTransferStart -End $SQLJobDataTransferEnd | Select -ExpandProperty TotalSeconds
$SQLJobDataTimeSeconds = [Math]::Round($SQLJobDataTimeSeconds,0)
# Getting data
$SQLJobDataTransferredBytes = $SQLJobEventSeries.dataTransferred
# $SQLJobDataThroughputBytes = $SQLJobEventSeries.effectiveThroughput
# Converting bytes
$SQLJobDataTransferredMB = $SQLJobDataTransferredBytes / 1024 / 1024
$SQLJobDataTransferredGB = $SQLJobDataTransferredBytes / 1024 / 1024 / 1024
$SQLJobDataTransferredGB = [Math]::Round($SQLJobDataTransferredGB,2)
# Calculating throughput 
$SQLJobDataThroughputMB = $SQLJobDataTransferredMB / $SQLJobDataTimeSeconds
$SQLJobDataThroughputMB = [Math]::Round($SQLJobDataThroughputMB,2)
}
ELSE
{
# Nulling data fields as job was not successful
$SQLJobDataTransferredGB = $null
$SQLJobDataThroughputMB = $null
$SQLJobDataTimeSeconds = $null
}
##################################
# Adding To Array
##################################
# Adding to array
$SQLDBRefreshResult = New-Object PSObject
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLDBRefreshRequestDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLDBRefreshRequestHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetHostID" -Value $SQLDBRefreshRequestHostID
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetInstanceID" -Value $SQLDBRefreshRequestInstanceID
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBRefreshRequestSourceDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceDBID" -Value $SQLDBRefreshRequestSourceDBID
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBRefreshRequestSourceHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value $SQLJobStatus
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $SQLJobStart
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "End" -Value $SQLJobEnd
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $SQLJobDataTimeSeconds
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TransferredGB" -Value $SQLJobDataTransferredGB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "ThroughputMBSec" -Value $SQLJobDataThroughputMB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $SQLDBRereshRequestPathSource
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $SQLDBRereshRequestPathDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $SQLDBRereshRequestPathDBFiles
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $SQLDBRereshRequestPathLogFiles
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Details" -Value $SQLJobError
$SQLDBRefreshResults += $SQLDBRefreshResult
# End of per job check below
}
# End of per job check above
#
# End of checking backup jobs below
}
# End of checking backup jobs above
####################################################################
# Section 6 - Totalling results
####################################################################
##################################
# Getting totals
##################################
$TotalBackupsRequested = $SQLDBsToBackup | Measure | Select -ExpandProperty Count
$TotalBackupsSuccess = $SQLDBBackupResults | Where-Object {$_.JobStatus -eq "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalBackupsFail = $SQLDBBackupResults | Where-Object {$_.JobStatus -ne "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalExportsRequested = $SQLDBsToExport | Measure | Select -ExpandProperty Count
$TotalExportsSuccess = $SQLDBRefreshResults | Where-Object {$_.JobStatus -eq "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalExportsFail = $SQLDBRefreshResults | Where-Object {$_.JobStatus -ne "SUCCEEDED"} | Measure | Select -ExpandProperty Count
# Backup timing
$BackupStart = $SQLDBBackupResults | Sort-Object Start -Descending | Select -ExpandProperty Start -First 1
$BackupEnd = $SQLDBBackupResults | Sort-Object End -Descending | Select -ExpandProperty End -First 1
$BackupHoursTaken = New-Timespan -Start $BackupStart -End $BackupEnd | Select -ExpandProperty TotalHours
# Export timing
$ExportStart = $SQLDBRefreshResults | Sort-Object Start -Descending | Select -ExpandProperty Start -First 1
$ExportEnd = $SQLDBRefreshResults | Sort-Object End -Descending | Select -ExpandProperty End -First 1
$ExportHoursTaken = New-Timespan -Start $ExportStart -End $ExportEnd | Select -ExpandProperty TotalHours
# Total timing
$TotalHoursTaken = New-Timespan -Start $BackupStart -End $ExportEnd | Select -ExpandProperty TotalHours
# Averages
$AverageThroughputMB = $SQLDBRefreshResults | Select -ExpandProperty ThroughputMBSec | Measure -Average | Select -ExpandProperty Average
# Rounding
$BackupHoursTaken = [Math]::Round($BackupHoursTaken,2)
$ExportHoursTaken = [Math]::Round($ExportHoursTaken,2)
$TotalHoursTaken = [Math]::Round($TotalHoursTaken,2)
$AverageThroughputMB = [Math]::Round($AverageThroughputMB,2)
####################################################################
# Section 7 - Sending Email 2 (Summary)
####################################################################
##################################
# Creating HTML Summary table
##################################
$HTMLSummaryTable = @"
<table class="table1">
  <tr>
    <th>Created $SystemDateTime</th>
  </tr>
  <tr>
    <th>SQL Refresh Report</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>Backups</th>
    <td>$TotalBackupsRequested</td>
    <th>Exports</th>
    <td>$TotalExportsRequested</td>
    <th>Start</th>
    <td>$BackupStart</td>
    <th>Exports</th>
    <td>$ExportStart</td>
    <th>Backups</th>
    <td>$BackupStart</td>
  </tr>
 <tr>
    <th>Success</th>
    <td>$TotalBackupsSuccess</td>
    <th>Success</th>
    <td>$TotalExportsSuccess</td>
    <th>End</th>
    <td>$ExportEnd</td>
    <th>End</th>
    <td>$ExportEnd</td>
    <th>End</th>
    <td>$BackupEnd</td>
 </tr>
 <tr>
    <th>Fail</th>
    <td>$TotalBackupsFail</td>
    <th>Fail</th>
    <td>$TotalExportsFail</td>
    <th>TotalHours</th>
    <td>$TotalHoursTaken</td>
    <th>Hours</th>
    <td>$ExportHoursTaken</td>
    <th>Hours</th>
    <td>$BackupHoursTaken</td>
 </tr>
 </table>
 <br>
"@
##################################
# Creating Table 1 HTML structure
##################################
$HTMLTable1Start = @"
<table class="table1">
  <tr>
    <th>SQL DB Export Requests</th>
  </tr>
  <tr>
    <th>Average Throughput: $AverageThroughputMB MB/Sec</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>SourceHost</th>
    <th>SourceDB</th>
    <th>TargetHost</th>
    <th>TargetDB</th>
    <th>Data<br>GB</th>
    <th>Export<br>Status</th>
    <th>Export<br>Hours</th>
    <th>Throughput<br>MB/Sec</th>
    <th>Backup<br>Status</th>
    <th>Backup<br>Hours</th>
    <th>Details</th>
  </tr>
"@
$HTMLTable1End = @"
</table>
<br>
"@
##################################
# Creating Table 1 HTML Rows
##################################
# Building email list by task to put the most important objects for viewing at the top
$Table1Data = $SQLDBRefreshResults | Sort-Object TargetDB,TargetHost,SourceDB,SourceHost
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTable1Middle = $null
# Creating table row for each line
ForEach ($Row in $Table1Data) 
{
# Setting values
$HTML1SourceHost = $Row.SourceHost
$HTML1SourceDB = $Row.SourceDB
$HTML1TargetHost = $Row.TargetHost
$HTML1TargetDB = $Row.TargetDB
$HTML1Export = $Row.JobStatus
$HTML1ExportSeconds = $Row.TimeTakenSeconds
$HTML1ExportTransferredGB = $Row.TransferredGB
$HTML1ExportThroughputMBSec = $Row.ThroughputMBSec
$HTML1DBPath = $Row.PathDBFiles
$HTML1LogPath = $Row.PathLogFiles
$HTML1Details = $Row.Details
# Converting seconds to hours
$HTML1ExportHours = $HTML1ExportSeconds / 60 / 60
$HTML1ExportHours = [Math]::Round($HTML1ExportHours,2)
# Selecting backup data
$SQLDBBackupResult = $SQLDBBackupResults | Where-Object {(($_.SourceHost -eq $HTML1SourceHost) -AND ($_.SourceDB -eq $HTML1SourceDB))}
$HTML1Backup = $SQLDBBackupResult.JobStatus
$HTML1BackupSeconds = $SQLDBBackupResult.TimeTakenSeconds
# Converting seconds to hours
$HTML1BackupHours = $HTML1BackupSeconds / 60 / 60
$HTML1BackupHours = [Math]::Round($HTML1BackupHours,2)
# Building HTML table row
$HTMLReportTable1Row = "
<tr>
    <td>$HTML1SourceHost</td>
    <td>$HTML1SourceDB</td>
    <td>$HTML1TargetHost</td>
    <td>$HTML1TargetDB</td>
    <td>$HTML1ExportTransferredGB</td>
    <td>$HTML1Export</td>
    <td>$HTML1ExportHours</td>
    <td>$HTML1ExportThroughputMBSec</td>
    <td>$HTML1Backup</td>
    <td>$HTML1BackupHours</td>
    <td>$HTML1Details</td>
  </tr>
"
# Adding row to table
$HTMLReportTable1Middle += $HTMLReportTable1Row
}
##################################
# Putting Table 1 together
##################################
$HTMLTable1 = $HTMLTable1Start + $HTMLReportTable1Middle + $HTMLTable1End
##################################
# Creating Report
##################################
# Building HTML report:
$HTMLReport = $HTMLStart + $HTMLSummaryTable + $HTMLTable1 + $HTMLEnd
# Replacing strings for easier reading
$HTMLReport = $HTMLReport.Replace("\\","\").TrimEnd()
##################################
# Creating CSVs
##################################
# Creating the file names
# $SQLDBComplianceReportCSVFile = $CSVOutputDirectory + "\SQLDBReport-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
# Exporting to CSV
# $SQLDBBackupResults | Export-Csv -Path $SQLDBComplianceReportCSVFile -NoTypeInformation -Force
# $SQLDBRefreshResults | Export-Csv -Path $SQLDBComplianceReportCSVFile -NoTypeInformation -Force
##################################
# Sending email using function
##################################
# Building attachment
$Attachments = "$LogoFile"
# Sending email
Try
{
Send-MailMessage -To $EmailTo -From $EmailFrom -Body $HTMLReport -Subject $EmailSubject2 -Attachments $Attachments -SmtpServer $EmailServer -BodyAsHtml 
}
Catch
{
$Error[0] | Format-List -Force
}
##################################
# End of script
##################################