﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates an on-demand snapshot for the target DB specified, then refreshes it using an export from the source DB
# If the target DB already exists on the server it will use its paths for the export, if it doesn't exist then it will use the paths of the DB with the most backups on the target, if no DBs are protected on the target it will use the source file paths
# If the same Source DB and Server are specified multiple times, it will only take a backup once (so you can export to multiple targets)
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A SQL database ready to protect, it doesn't have to have an existing SLA domain assigned
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
$SQLCSVToProcess = "C:\RubrikSQLAutomationv1\BulkBackupThenRefresh\RubrikSQLBulkExport.csv"
# HTML Color codes used for report rows
$HTMLColorSuccess = "#000000"
$HTMLColorFailure = "#e60000"
# Email settings
$EmailTo = "joshua@lab.local"
$EmailFrom = "rubrik@lab.local"
$EmailServer = "localhost"
# Email subject
$EmailSubject = "Rubrik SQLDB Export Report"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing CSV & Sorting
##################################
$SQLCSVImport = Import-Csv $SQLCSVToProcess
# Getting unique DBs to backup
$SQLDBsToBackup = $SQLCSVImport | Select-Object @{Label = "SourceHostAndDB"; Expression = {"$($_.'SourceHost') $($_.'SourceDB')"} } -Unique
# Getting all DBs to export 
$SQLDBsToExport = $SQLCSVImport
# Logo work
$LogoFile = "C:\Program Files\RubrikPolarisLocalv1\Reports\Images\Logo.png"
# Timestamp
$SystemDateTime = Get-Date
################################################
# Creating the Convert-UNIXTime function
################################################
Function Convert-UNIXTime {
Param ($UNIXTime)

# Example: $PSTime = Convert-UNIXTime -UNIXTime $UNIXTime

# Step 1 - Removing characters and trimming snapshot string for conversion
$PSTimeStep1 = $UNIXTime.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Removing last 4 characters
$PSTimeStep2 = $PSTimeStep1.Substring(0,$PSTimeStep1.Length-4)
# Step 3 - Converting string to PowerShell datetime object
$PSTimeStep3 = ([datetime]::ParseExact($PSTimeStep2,”yyyy-MM-dd HH:mm:ss”,$null))
# Returning Result
Return $PSTimeStep3
}
####################################################################
# Section 1 - Connecting to API and building table of DBs
####################################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainListURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainListJSON = Invoke-RestMethod -Uri $SLADomainListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainList = $SLADomainListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000&is_relic=false"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
##################################
"Updating List of All SQL DBs Prior To Starting Operations....."
# Building array
$SQLInstanceArray = @()
# Starting for Each
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Combining server name and instance
$SQLInstanceServer = $SQLInstanceHostName + "\" + $SQLInstanceName 
# Adding to array
$SQLInstanceArrayLine = New-Object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
$SQLDBCounter = 0
$SQLDBCount = $SQLDBList | Measure | Select -ExpandProperty Count
ForEach ($SQLDB in $SQLDBList)
{
# Incrementing counter
$SQLDBCounter ++
# Output to host
"SQLDB: $SQLDBCounter / $SQLDBCount"
# Setting variables
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
$SQLDBSLADomain = $SQLDB.effectiveSlaDomainName
$SQLDBSLADomainID = $SQLDB.effectiveSlaDomainId
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
$SQLInstanceServer = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Server
}
# Getting DB backup count for DBs if target host matches for file selection later
IF ($SQLInstanceServer -eq $SQLTargetHost)
{
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting variables
$SQLDBSnapshots = $SQLDBInfo.snapshotCount
}
ELSE
{
$SQLDBSnapshots = $null
}
# Adding to array
$SQLDBArrayLine = New-Object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "Server" -Value "$SQLInstanceServer"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SQLDBSLADomain"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomainID" -Value "$SQLDBSLADomainID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "Snapshots" -Value "$SQLDBSnapshots"
$SQLDBArray += $SQLDBArrayLine
}
####################################################################
# Section 2 - Performing backup of target DBs (if they exist)
####################################################################
# Output to host
"--------------------------------------
Starting Backups"
##################################
# Creating array required
##################################
$SQLDBBackupRequests = @()
$SQLDBBackupResults = @()
##################################
# Starting For Each DB To Backup
##################################
ForEach ($SQLDBToBackup in $SQLDBsToBackup)
{
# Setting variable
$SQLDBToSplit = $SQLDBToBackup | Select -ExpandProperty SourceHostAndDB
# Getting Source Host/DB from unique splits
$SQLSourceHost = $SQLDBToSplit.Split(" ")[0]
$SQLSourceDB = $SQLDBToSplit.Split(" ")[1]
# Output to Host
"--------------------------------------
ProcessingBackup:
SourceDB: $SQLSourceDB
SQLDBID: $SQLBackupDBID
SourceHost: $SQLSourceHost"
##################################
# Selecting the DB ID from the SQLDBArray and SLA domain ID from SLADomainList 
##################################
$SQLBackupDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLSourceHost) -AND ($_.DatabaseName -eq $SQLSourceDB))} | Select -ExpandProperty DatabaseID
# Selecting SLA domain ID
$SQLBackupSLADomainID = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupDBID} | Select -ExpandProperty SLADomainID
$SQLBackupSLADomain = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupDBID} | Select -ExpandProperty SLADomain
# If the Source DB exists, but has no SLA, selecting the target DB SLA
IF ($SQLBackupSLADomainID -eq $null)
{
# Selecting source DB ID
$SQLBackupSourceDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.DatabaseName -eq $SQLTargetDB))} | Select -ExpandProperty DatabaseID
# Selecting source SLA
$SQLBackupSLADomainID = $SQLDBArray | Where-Object {$_.DatabaseID -eq $SQLBackupSourceDBID} | Select -ExpandProperty SLADomainID
}
##################################
# Checking for Required IDs
##################################
IF (($SQLBackupDBID -eq $null) -or ($SQLBackupSLADomainID -eq $null))
{
"
ERROR-Skipping:

SourceDBNotFound: $SQLSourceDB
SourceHost: $SQLSourceHost
"

# Adding to array
$SQLDBBackupResult = New-Object PSObject
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBBackupRequestDB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBBackupRequestHost
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value "DBIDNotFound"
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $Null
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "End" -Value $Null
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $Null
$SQLDBBackupResults += $SQLDBBackupResult


}
ELSE
{
##################################
# IF DB Existing Taking Backup
##################################
# Creating URL
$SQLDBOnDemandSnapURL = $BaseURL+"mssql/db/"+$SQLBackupDBID+"/snapshot"
# Creating JSON body
$SQLDBOnDemandSnapJSON = 
"{
  ""slaId"": ""$SQLBackupSLADomainID""
}"
# Output to host
"StartingBackupWith:
SLADomain: $SQLBackupSLADomain
SLADomainID: $SQLBackupSLADomainID"
# Posting to the API
Try 
{
$SQLDBOnDemandSnapResponse = Invoke-RestMethod -Method Post -Uri $SQLDBOnDemandSnapURL -Body $SQLDBOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBOnDemandSnapResponse.links.href
# Adding to array
$SQLDBBackupRequest = New-Object PSObject
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLSourceDB
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLSourceHost
$SQLDBBackupRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value $SQLJobStatusURL
$SQLDBBackupRequests += $SQLDBBackupRequest
##################################
# End of bypass for no target DB found below
##################################
}
# End of bypass for no target DB found above

##################################
# End of for each SQL DB below
##################################
}
# End of for each SQL DB above
#
##################################
# Getting Job status on a loop, if requests exist
##################################
IF ($SQLDBBackupRequests -ne $null)
{
# Output to host
"--------------------------------------
Checking Backups Complete
--------------------------------------"
# Starting status check
ForEach ($SQLDBBackupRequest in $SQLDBBackupRequests)
{
# Setting counter
$SQLJobStatusCount = 0
# Setting variables
$SQLDBBackupRequestDB = $SQLDBBackupRequest.SourceDB
$SQLDBBackupRequestHost = $SQLDBBackupRequest.SourceHost
$SQLDBBackupRequestURL = $SQLDBBackupRequest.JobURL
# Looping status check
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLDBBackupRequestURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
$SQLJobProgress  = $SQLJobStatusResponse.progress
$SQLJobStartUNIX = $SQLJobStatusResponse.startTime
$SQLJobEndUNIX = $SQLJobStatusResponse.endTime
# Setting progress if success, as API returns null
IF ($SQLJobStatus -eq "SUCCEEDED")
{
$SQLJobProgress = "100"
}
# Output to host
"BackupDB: $SQLDBBackupRequestDB 
Host: $SQLDBBackupRequestHost
BackupJob: $SQLJobStatus
Progress: $SQLJobProgress
"
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Getting error if FAILED
IF ($SQLJobStatus -eq "FAILED")
{
$SQLJobError = $SQLJobStatusResponse.error
}
ELSE
{
$SQLJobError = $null
}
# Will run until it succeeds, fails, or hits 1 hour
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "Failed") -OR ($SQLJobStatusCount -eq 10800))
# Converting time
$SQLJobStart = Convert-UNIXTime -UNIXTime $SQLJobStartUNIX
# Only converting if end time exists
IF ($SQLJobEndUNIX -ne $null)
{
$SQLJobEnd = Convert-UNIXTime -UNIXTime $SQLJobEndUNIX
# Getting time span
$SQLJobTimeTakenSeconds = New-Timespan -Start $SQLJobStart -End $SQLJobEnd | Select -ExpandProperty TotalSeconds
}
ELSE
{
$SQLJobTimeTakenSeconds = $null
}
# Adding to array
$SQLDBBackupResult = New-Object PSObject
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBBackupRequestDB
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBBackupRequestHost
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value $SQLJobStatus
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $SQLJobStart
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "End" -Value $SQLJobEnd
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $SQLJobTimeTakenSeconds
$SQLDBBackupResult | Add-Member -MemberType NoteProperty -Name "Details" -Value $SQLJobError
$SQLDBBackupResults += $SQLDBBackupResult
# End of per job check below
}
# End of per job check above
#
# End of checking backup jobs below
}
# End of checking backup jobs above
####################################################################
# Section 3 - Performing export to refresh target DBs
####################################################################
# Output to host
"--------------------------------------
Starting Refresh/Exports"
##################################
# Creating array required
##################################
$SQLDBRefreshRequests = @()
$SQLDBRefreshResults = @()
##################################
# Starting For Each DB to Export
##################################
ForEach ($SQLDBToExport in $SQLDBsToExport)
{
# Setting variables
$SQLSourceHost = $SQLDBToExport.SourceHost
$SQLSourceDB = $SQLDBToExport.SourceDB
$SQLTargetHost = $SQLDBToExport.TargetHost
$SQLTargetDB = $SQLDBToExport.TargetDB
# Output to host
"--------------------------------------
ProcessingExport/Refresh
TargetSQLDB: $SQLTargetDB
TargetSQLServer : $SQLTargetHost
SourceSQLDB: $SQLSourceDB
SourceSQLServer : $SQLSourceHost"
######################
# Selecting IDs required
######################
# Selecting the database ID
$SQLRefreshSourceDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLSourceHost) -AND ($_.DatabaseName -eq $SQLSourceDB))} | Select -ExpandProperty DatabaseID
# Selecting the target instance ID
$SQLTargetInstanceID = $SQLInstanceArray | Where-Object {($_.Server -eq $SQLTargetHost)} | Select -ExpandProperty InstanceID
# Selecting ID for refresh of host in Rubrik
$SQLTargetHostID = $SQLInstanceArray | Where-Object {($_.Server -eq $SQLTargetHost)} | Select -ExpandProperty HostID
######################
# Bypass if IDs not found
######################
IF (($SQLRefreshSourceDBID -eq $null) -or ($SQLTargetInstanceID -eq $null))
{
# Adding to array
$SQLDBRefreshResult = New-Object PSObject
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLDBRefreshRequestDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLDBRefreshRequestHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBRefreshRequestSourceDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBRefreshRequestSourceHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value "FAILED"
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "End" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $null
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Details" -Value "DBorINSTANCEIDNotFound"
$SQLDBRefreshResults += $SQLDBRefreshResult
}
ELSE
{
######################
# Selecting Target DB IDs
######################
# Selecting target DB ID
$SQLRefreshTargetDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.DatabaseName -eq $SQLTargetDB) -AND ($_.SLADomain -ne "UNPROTECTED") -AND ($_.Snapshots -gt 0))} | Select -ExpandProperty DatabaseID -First 1
$SQLRefreshTargetDBFilePathsFromDB = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.DatabaseName -eq $SQLTargetDB) -AND ($_.SLADomain -ne "UNPROTECTED") -AND ($_.Snapshots -gt 0))} | Select -ExpandProperty DatabaseName -First 1
$SQLRefreshTargetDBFilePathsFrom = "Target"
# If target DB doe not exist, getting any DB (for file locations)
IF ($SQLRefreshTargetDBID -eq $null)
{
# If target DB doesn't already exist then selecting the DB with the most backups to copy its file structure
$SQLRefreshTargetDBID = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.SLADomain -ne "UNPROTECTED") -AND ($_.Snapshots -gt 0))} | Sort-Object Snapshots -Descending | Select -ExpandProperty DatabaseID -First 1
$SQLRefreshTargetDBFilePathsFromDB = $SQLDBArray | Where-Object {(($_.Server -eq $SQLTargetHost) -AND ($_.SLADomain -ne "UNPROTECTED") -AND ($_.Snapshots -gt 0))} | Sort-Object Snapshots -Descending | Select -ExpandProperty DatabaseName -First 1
}
######################
# Getting Target DB info if any backups exist
######################
IF ($SQLRefreshTargetDBID -ne $null)
{
$TargetSQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLRefreshTargetDBID
Try 
{
$TargetSQLDBInfo = Invoke-RestMethod -Uri $TargetSQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting latest point in time
$TargetSQLDBLatestRecoveryPoint = $TargetSQLDBInfo.latestRecoveryPoint
# Converting latestRecoveryPoint to values needed
$TargetSQLDBLatestRecoveryPointURL= $TargetSQLDBLatestRecoveryPoint.Replace(":","%3A")
$TargetSQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($TargetSQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
######################
# Getting Target DB File info
######################
$TargetSQLDBFilesURL = $InternalURL+"mssql/db/"+$SQLRefreshTargetDBID+"/restore_files?time="+$TargetSQLDBLatestRecoveryPointURL
Try 
{
$TargetSQLDBFiles = Invoke-RestMethod -Uri $TargetSQLDBFilesURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Default File Path Selection Section from file structure on Target DB
######################
$DBPath = $TargetSQLDBFiles | Where-Object {$_.originalName -match ".mdf"} | Select -ExpandProperty originalPath -First 1
$LogPath =  $TargetSQLDBFiles | Where-Object {$_.originalName -match ".ldf"} | Select -ExpandProperty originalPath -First 1
# Output to host
"UsingPathFrom: $SQLRefreshTargetDBFilePathsFrom ($SQLRefreshTargetDBFilePathsFromDB)
DBPath: $DBPath 
LogPath: $LogPath"
# Converting
$DBPath = $DBPath.Replace("\","\\")
$LogPath = $LogPath.Replace("\","\\")
# End of if any target DBs have been backed up below
}
# End of if any target DBs have been backed up above
###############################################
# POST to API to Refresh Host before export (prevents errors if DB was only just exported and Rubrik hasn't refreshed yet)
###############################################
$HostRefreshURL = $BaseURL+"host/"+$SQLTargetHostID+"/refresh"
Try 
{
$HostRefreshResponse = Invoke-RestMethod -Uri $HostRefreshURL -Method POST -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Getting Source DB info
######################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLRefreshSourceDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting latest point in time
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Converting latestRecoveryPoint to values needed
$SQLDBLatestRecoveryPointURL= $SQLDBLatestRecoveryPoint.Replace(":","%3A")
$SQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($SQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
######################
# Getting DB File info
######################
$SQLDBFilesURL = $InternalURL+"mssql/db/"+$SQLRefreshSourceDBID+"/restore_files?time="+$SQLDBLatestRecoveryPointURL
Try 
{
$SQLDBFiles = Invoke-RestMethod -Uri $SQLDBFilesURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# If no target DBs exist to file path info, using source
######################
IF (($DBPath -eq $null) -and ($LogPath -eq $null))
{
$SQLRefreshTargetDBFilePathsFrom = "Source"
$SQLRefreshTargetDBFilePathsFromDB = $SQLDBSource
######################
# Default File Path Selection Section from file structure on Target DB
######################
$DBPath = $SQLDBFiles | Where-Object {$_.originalName -match ".mdf"} | Select -ExpandProperty originalPath -First 1
$LogPath =  $SQLDBFiles | Where-Object {$_.originalName -match ".ldf"} | Select -ExpandProperty originalPath -First 1
# Output to host
"UsingPathFrom: $SQLRefreshTargetDBFilePathsFrom ($SQLRefreshTargetDBFilePathsFromDB)
DBPath: $DBPath 
LogPath: $LogPath"
# Converting
$DBPath = $DBPath.Replace("\","\\")
$LogPath = $LogPath.Replace("\","\\")
}
######################
# Starting JSON build
######################
$SQLDBJSONStart = 
"{
""recoveryPoint"":
  {
  ""timestampMs"":$SQLDBLatestRecoveryPointMS
  },
""targetInstanceId"": ""$SQLTargetInstanceID"",
""targetDatabaseName"": ""$SQLTargetDB"",
""targetFilePaths"":[
"
######################
# Middle JSON build
######################
$SQLDBJSONMiddle = $null
# Selecting random ID to prevent file conflict
$Range = 1..10000
$RandomID = $Range | Get-Random
# Starting counters from ID
$SQLDBFileCounter = 1
$SQLNDFFileCounter = 1
$SQLDBLogFileCounter = 1
# Creating JSON VM array for all the VMs in the VPG
ForEach ($SQLDBFile in $SQLDBFiles)
{
# Setting the DB file variables
$LogicalName = $SQLDBFile.logicalName
$OriginalPath = $SQLDBFile.originalPath
$OriginalName = $SQLDBFile.originalName
# Converting original path
$OriginalPathConverted = $OriginalPath.Replace("\","\\")
######################
# MDF settings
######################
IF ($OriginalName -match ".mdf")
{
# Setting DB path
$ExportPath = $DBPath
# Setting New DB File Name
$NewFilename = $SQLTargetDB + "_ID" + $RandomID +"_" + $SQLDBFileCounter + ".mdf"
# Incrementing DB counter to prevent duplicates
$SQLDBFileCounter ++
}
######################
# NDF settings
######################
IF ($OriginalName -match ".ndf")
{
# Setting DB path
$ExportPath = $DBPath
# Setting New DB File Name
$NewFilename = $SQLTargetDB + "_ID" + $RandomID +"_" + $SQLNDFFileCounter + ".ndf"
# Incrementing DB counter to prevent duplicates
$SQLNDFFileCounter ++
}
######################
# LDF settings
######################
IF ($OriginalName -match ".ldf")
{
# Setting DB path
$ExportPath = $LogPath
# Setting New DB File Name
$NewFilename = $SQLTargetDB +"_ID" + $RandomID + "_Log" + $SQLDBLogFileCounter + ".ldf"
# Incrementing DB counter to prevent duplicates
$SQLDBLogFileCounter ++
}
######################
# Middle JSON
######################
$SQLDBJSONFile = 
"{
""exportPath"":""$ExportPath"",
""logicalName"":""$logicalName"",
""newFilename"":""$NewFilename""
}"
# Running if statement to check if this is the first file in the array, if not then a comma is added to string
IF ($SQLDBJSONMiddle -ne $null)
{
$SQLDBJSONFile = ",
" + $SQLDBJSONFile
}
# Adding the DB file to the JSON middle
$SQLDBJSONMiddle = $SQLDBJSONMiddle + $SQLDBJSONFile
}
######################
# End JSON & Compiling
######################
# Creating the end of the JSON request
$SQLDBJSONEnd = "],
""finishRecovery"": true,
""allowOverwrite"": true
}"
# Putting the JSON request together and outputting the request
$SQLDBJSON = $SQLDBJSONStart + $SQLDBJSONMiddle + $SQLDBJSONEnd
######################
# POST SQL Export
######################
# Creating POST URL
$SQLDBExportURL = $BaseURL+"mssql/db/"+$SQLRefreshSourceDBID+"/export"
# POST to API
Try 
{
$SQLDBExportResponse = Invoke-RestMethod -Method Post -Uri $SQLDBExportURL -Body $SQLDBJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBExportResponse.links.href
# Adding to array
$SQLDBRefreshRequest = New-Object PSObject
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLTargetDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLTargetHost
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLSourceDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLSourceHost
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "JobURL" -Value $SQLJobStatusURL
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $SQLRefreshTargetDBFilePathsFrom
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $SQLRefreshTargetDBFilePathsFromDB
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $DBPath
$SQLDBRefreshRequest | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $DBPath
$SQLDBRefreshRequests += $SQLDBRefreshRequest
######################
# End of bypass if IDs not found below
######################
}
# End of bypass if IDs not found above
##################################
# End of for Each DB to Export below
##################################
}
# End of for Each DB to Export above
##################################
# Getting Job status on a loop
##################################
IF ($SQLDBRefreshRequests -ne $null)
{
# Output to host
"--------------------------------------
Checking Refreshes Complete
--------------------------------------"
# Starting for each
ForEach ($SQLDBRefreshRequest in $SQLDBRefreshRequests)
{
# Setting counter
$SQLJobStatusCount = 0
# Setting variables
$SQLDBRefreshRequestDB = $SQLDBRefreshRequest.TargetDB
$SQLDBRefreshRequestHost = $SQLDBRefreshRequest.TargetHost
$SQLDBRefreshRequestSourceDB = $SQLDBRefreshRequest.SourceDB
$SQLDBRefreshRequestSourceHost = $SQLDBRefreshRequest.SourceHost
$SQLDBRefreshRequestURL = $SQLDBRefreshRequest.JobURL
$SQLDBRereshRequestPathSource = $SQLDBRefreshRequest.PathSource
$SQLDBRereshRequestPathDB = $SQLDBRefreshRequest.PathDB
$SQLDBRereshRequestPathDBFiles = $SQLDBRefreshRequest.PathDBFiles
$SQLDBRereshRequestPathLogFiles = $SQLDBRefreshRequest.PathLogFiles
# Looping status check
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLDBRefreshRequestURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
$SQLJobProgress  = $SQLJobStatusResponse.progress
$SQLJobStatusError = $SQLJobStatusResponse.error.message
$SQLJobStartUNIX = $SQLJobStatusResponse.startTime
$SQLJobEndUNIX = $SQLJobStatusResponse.endTime
# Setting progress if success, as API returns null
IF ($SQLJobStatus -eq "SUCCEEDED")
{
$SQLJobProgress = "100"
}
# Output to host
"TargetDB: $SQLDBRefreshRequestDB 
TargetHost: $SQLDBRefreshRequestHost
ExportJob: $SQLJobStatus
Progress: $SQLJobProgress
"
# Showing error if one exists
IF ($SQLJobStatusError -ne $null)
{
"Error: $SQLJobStatusError"
}
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Getting error message
$SQLJobError = $SQLJobStatusResponse.error.message
# Getting error if FAILED
IF ($SQLJobError -ne $null)
{
$SQLJobStatus = "FAILED"
}
# Will run until it succeeds, fails, or hits 1 hour
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "FAILED") -OR ($SQLJobStatusCount -eq 10800) -or ($SQLJobStatusError -ne $null))
# Converting time
$SQLJobStart = Convert-UNIXTime -UNIXTime $SQLJobStartUNIX
# Only converting end time and getting time run if the job returned an end time, otherwise it errors
IF ($SQLJobEndUNIX -ne $null)
{
$SQLJobEnd = Convert-UNIXTime -UNIXTime $SQLJobEndUNIX
# Getting time span
$SQLJobTimeTakenSeconds = New-Timespan -Start $SQLJobStart -End $SQLJobEnd | Select -ExpandProperty TotalSeconds
}
ELSE
{
$SQLJobTimeTakenSeconds = $null
}
# Adding to array
$SQLDBRefreshResult = New-Object PSObject
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetDB" -Value $SQLDBRefreshRequestDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TargetHost" -Value $SQLDBRefreshRequestHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceDB" -Value $SQLDBRefreshRequestSourceDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $SQLDBRefreshRequestSourceHost
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "JobStatus" -Value $SQLJobStatus
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Start" -Value $SQLJobStart
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "End" -Value $SQLJobEnd
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "TimeTakenSeconds" -Value $SQLJobTimeTakenSeconds
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathSource" -Value $SQLDBRereshRequestPathSource
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDB" -Value $SQLDBRereshRequestPathDB
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathDBFiles" -Value $SQLDBRereshRequestPathDBFiles
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "PathLogFiles" -Value $SQLDBRereshRequestPathLogFiles
$SQLDBRefreshResult | Add-Member -MemberType NoteProperty -Name "Details" -Value $SQLJobError
$SQLDBRefreshResults += $SQLDBRefreshResult
# End of per job check below
}
# End of per job check above
#
# End of checking backup jobs below
}
# End of checking backup jobs above
##################################
# Getting totals
##################################
$TotalBackupsRequested = $SQLDBsToBackup | Measure | Select -ExpandProperty Count
$TotalBackupsSuccess = $SQLDBBackupResults | Where-Object {$_.JobStatus -eq "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalBackupsFail = $SQLDBBackupResults | Where-Object {$_.JobStatus -ne "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalExportsRequested = $SQLDBsToExport | Measure | Select -ExpandProperty Count
$TotalExportsSuccess = $SQLDBRefreshResults | Where-Object {$_.JobStatus -eq "SUCCEEDED"} | Measure | Select -ExpandProperty Count
$TotalExportsFail = $SQLDBRefreshResults | Where-Object {$_.JobStatus -ne "SUCCEEDED"} | Measure | Select -ExpandProperty Count
# Backup timing
$BackupStart = $SQLDBBackupResults | Sort-Object Start -Descending | Select -ExpandProperty Start -First 1
$BackupEnd = $SQLDBBackupResults | Sort-Object End -Descending | Select -ExpandProperty End -First 1
$BackupMinutesTaken = New-Timespan -Start $BackupStart -End $BackupEnd | Select -ExpandProperty TotalMinutes
# Export timing
$ExportStart = $SQLDBRefreshResults | Sort-Object Start -Descending | Select -ExpandProperty Start -First 1
$ExportEnd = $SQLDBRefreshResults | Sort-Object End -Descending | Select -ExpandProperty End -First 1
$ExportMinutesTaken = New-Timespan -Start $ExportStart -End $ExportEnd | Select -ExpandProperty TotalMinutes
# Total timing
$TotalMinutesTaken = New-Timespan -Start $BackupStart -End $ExportEnd | Select -ExpandProperty TotalMinutes
# Rounding
$BackupMinutesTaken = [Math]::Round($BackupMinutesTaken,2)
$ExportMinutesTaken = [Math]::Round($ExportMinutesTaken,2)
$TotalMinutesTaken = [Math]::Round($TotalMinutesTaken,2)
# Output to host
"TotalBackupsRequested: $TotalBackupsRequested
TotalBackupsSuccess: $TotalBackupsSuccess
TotalBackupsFail: $TotalBackupsFail

TotalExportsRequested: $TotalExportsRequested 
TotalExportsSuccess: $TotalExportsSuccess
TotalExportsFail: $TotalExportsFail

BackupStart: $BackupStart
BackupEnd: $BackupEnd
Minutes: $BackupMinutesTaken 

ExportStart: $ExportStart
ExportEnd: $ExportEnd
Minutes: $ExportMinutesTaken

TotalMinutesTaken: $TotalMinutesTaken"
##################################
# End of script
##################################