﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates multiple SQL exports from the CSV specified
################################################ 
# Requirements:
# - Run the RubrikSQLExportv1-Auth.ps1 to secure your Rubrik credentials, this script won't work without this being run first
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - At least 1 sql database protected in Rubrik and therefore 1 windows host
# - A CSV with the following fields: SourceSQLHostName,SourceInstanceName,SourceDatabaseName,TargetSQLHostName,TargetInstanceName,TargetDatabaseName,DBPath,LogPath
# - Example CSV Line 1 = SQL16-VM01.lab.local,MSSQLSERVER,SQL16-VM01-DemoDB01,SQL16-VM02.lab.local,MSSQLSERVER,DBScriptNewExport02,Default,Default
# - Example CSV Line 2 = SQL16-VM01.lab.local,MSSQLSERVER,SQL16-VM01-DemoDB01,SQL16-VM02.lab.local,MSSQLSERVER,DBScriptNewExport03,H:\Exports,H:\Exports
# - Example CSV Line 3 = AvailabilityGroup,SQLAG1,vSphereChangeControlv1,SQL16-VM02.lab.local,MSSQLSERVER,DBScriptNewExport05,Default,Default
# - For exports of a DB in an Availabiltiy Group just put the SourceSQLHostName
# - If you are unsure on the valid names for each field run the included 
# - This script always exports the latest snapshot available
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, 
# even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
$SQLDBExportCSV = "C:\RubrikSQLAutomationv1\BulkExport\RubrikSQLBulkExport.csv"
# Overwrite target DB if already exists, true or false, has to be one or the other
$OverwriteTargetDB = "true"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
###############################################
# Importing DBs to Export
###############################################
$SQLDBExportList = Import-Csv -Path $SQLDBExportCSV
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
###############################################
# Getting list of SQL Databases
###############################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
###############################################
# Getting list of SQL Instances
###############################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
###############################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
###############################################
$SQLInstanceArray = @()
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Adding to array
$SQLInstanceArrayLine = new-object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostNameID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
ForEach ($SQLDB in $SQLDBList)
{
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
}
# Adding to array
$SQLDBArrayLine = new-object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "SQLHostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArray += $SQLDBArrayLine
}
###############################################
# Starting For Each DB to Export
###############################################
ForEach ($SQLDBExport in $SQLDBExportList)
{
######################
# Setting variables
######################
$SourceSQLHostName = $SQLDBExport.SourceSQLHostName
$SourceInstanceName = $SQLDBExport.SourceInstanceName
$SourceDatabaseName = $SQLDBExport.SourceDatabaseName
$TargetSQLHostName = $SQLDBExport.TargetSQLHostName
$TargetInstanceName = $SQLDBExport.TargetInstanceName
$TargetDatabaseName = $SQLDBExport.TargetDatabaseName
$DBPath = $SQLDBExport.DBPath
$LogPath = $SQLDBExport.LogPath
######################
# Converting the paths to usable format for JSON, if not default
######################
IF ($DBPath -ne "Default")
{
$DBPath = $DBPath.Replace("\","\\")
}
IF ($LogPath -ne "Default")
{
$LogPath = $LogPath.Replace("\","\\")
}
######################
# Selecting IDs required
######################
# Selecting the database ID
$SQLDBID = $SQLDBArray | Where-Object {($_.SQLHostName -eq $SourceSQLHostName) -and ($_.InstanceName -eq $SourceInstanceName) -and ($_.DatabaseName -eq $SourceDatabaseName)} | Select -ExpandProperty DatabaseID
# Selecting the target instance ID
$SQLTargetInstanceID = $SQLInstanceArray | Where-Object {($_.InstanceName -eq $TargetInstanceName) -and ($_.Hostname -eq $TargetSQLHostName)} | Select -ExpandProperty InstanceID
######################
# Getting DB info
######################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Selecting latest point in time
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Converting latestRecoveryPoint to values needed
$SQLDBLatestRecoveryPointURL= $SQLDBLatestRecoveryPoint.Replace(":","%3A")
$SQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($SQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
######################
# Getting DB File info
######################
$SQLDBFilesURL = $InternalURL+"mssql/db/"+$SQLDBID+"/restore_files?time="+$SQLDBLatestRecoveryPointURL
Try 
{
$SQLDBFiles = Invoke-RestMethod -Uri $SQLDBFilesURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
######################
# Starting JSON build
######################
$SQLDBJSONStart = 
"{
""recoveryPoint"":
  {
  ""timestampMs"":$SQLDBLatestRecoveryPointMS
  },
""targetInstanceId"": ""$SQLTargetInstanceID"",
""targetDatabaseName"": ""$TargetDatabaseName"",
""targetFilePaths"":[
"
######################
# Middle JSON build
######################
$SQLDBJSONMiddle = $null
# Selecting random ID to prevent file conflict
$Range = 1..10000
$RandomID = $Range | Get-Random
# Starting counters from ID
$SQLDBFileCounter = 1
$SQLNDFFileCounter = 1
$SQLDBLogFileCounter = 1
# Creating JSON VM array for all the VMs in the VPG
foreach ($SQLDBFile in $SQLDBFiles)
{
# Setting the DB file variables
$logicalName = $SQLDBFile.logicalName
$originalPath = $SQLDBFile.originalPath
$originalName = $SQLDBFile.originalName
# Converting original path
$originalPathConverted = $originalPath.Replace("\","\\")
######################
# MDF settings
######################
IF ($originalName -match ".mdf")
{
# Setting DB path
If ($DBPath -eq "Default")
{
$ExportPath = $originalPathConverted
}
ELSE
{
$ExportPath = $DBPath
}
# Setting New DB File Name
$NewFilename = $TargetDatabaseName + "_ID" + $RandomID +"_" +$SQLDBFileCounter + ".mdf"
# Incrementing DB counter to prevent duplicates
$SQLDBFileCounter ++
}
######################
# NDF settings
######################
IF ($originalName -match ".ndf")
{
# Setting DB path
If ($DBPath -eq "Default")
{
$ExportPath = $originalPathConverted
}
ELSE
{
$ExportPath = $DBPath
}
# Setting New DB File Name
$NewFilename = $TargetDatabaseName + "_ID" + $RandomID +"_" + $SQLNDFFileCounter + ".ndf"
# Incrementing DB counter to prevent duplicates
$SQLNDFFileCounter ++
}
######################
# LDF settings
######################
IF ($originalName -match ".ldf")
{
# Setting DB path
If ($LogPath -eq "Default")
{
$ExportPath = $originalPathConverted
}
ELSE
{
$ExportPath = $LogPath
}
# Setting New DB File Name
$NewFilename = $TargetDatabaseName + "_ID" + $RandomID +"_Log" + $SQLDBLogFileCounter + ".ldf"
# Incrementing DB counter to prevent duplicates
$SQLDBLogFileCounter ++
}
######################
# Middle JSON
######################
$SQLDBJSONFile = 
"{
""exportPath"":""$ExportPath"",
""logicalName"":""$logicalName"",
""newFilename"":""$NewFilename""
}"
# Running if statement to check if this is the first file in the array, if not then a comma is added to string
if ($SQLDBJSONMiddle -ne $null)
{
$SQLDBJSONFile = ",
" + $SQLDBJSONFile
}
# Adding the DB file to the JSON middle
$SQLDBJSONMiddle = $SQLDBJSONMiddle + $SQLDBJSONFile
}
######################
# End JSON & Compiling
######################
# Creating the end of the JSON request
$SQLDBJSONEnd = "],
""finishRecovery"": true,
""allowOverwrite"": $OverwriteTargetDB
}"
# Putting the JSON request together and outputting the request
$SQLDBJSON = $SQLDBJSONStart + $SQLDBJSONMiddle + $SQLDBJSONEnd
######################
# POST SQL Export
######################
# Creating POST URL
$SQLDBExportURL = $BaseURL+"mssql/db/"+$SQLDBID+"/export"
# POST to API
Try 
{
$SQLDBExportPOST = Invoke-RestMethod -Method Post -Uri $SQLDBExportURL -Body $SQLDBJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
# Output of task request
"DB Export Task Result:"
$SQLDBExportPOST
# Waiting 5 seconds before next DB
sleep 5
# End of Export action per DB below
} 
# End of Export action per DB above
#
###############################################
# End of script
###############################################