﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates an on-demand snapshots for all the SQL Dabatases listed in the CSV along with the SLA domain specified
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A CSV with the following columns: VMName,SnapshotSLADomain
# - Example CSV line: SQLDB01,GlacierArchive
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
$DBSnapshotCSV = "C:\RubrikSQLAutomationv1\RubrikSQLDBList.csv"
# Time delay between snapshot requests (in seconds, if you want to spread them out)
$TimeDelay = 0
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
##################################
# Importing the CSV
##################################
$DBsToSnapshot = Import-CSV $DBSnapshotCSV 
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000&is_relic=false"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
##################################
$SQLInstanceArray = @()
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Adding to array
$SQLInstanceArrayLine = New-Object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostNameID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
ForEach ($SQLDB in $SQLDBList)
{
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
$SQLDBInstanceID = $SQLAvailabilityGroupID
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
}
# Adding to array
$SQLDBArrayLine = New-Object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArray += $SQLDBArrayLine
}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainListURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainListJSON = Invoke-RestMethod -Uri $SLADomainListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainList = $SLADomainListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Performing For Each DB On Demand Snapshot Action
##################################
ForEach ($DB in $DBsToSnapshot)
{
# Assigning variables from the CSV
$SQLDBHostName = $DB.SQLHostName
$SQLDBInstanceName = $DB.InstanceName
$SQLDBName = $DB.DatabaseName
$SQLDBSLADomain = $DB.SLADomain
# Troubleshooting output
"--------------------------"
"Processing On-Demand Snapshot For:"
"Host: $SQLDBHostName"
"Instance: $SQLDBInstanceName"
"DB: $SQLDBName"
"SLADomain: $SQLDBSLADomain"
# Selecting the Instance ID
$SQLDBInstanceID = $SQLDBArray | Where-Object {(($_.HostName -eq $SQLDBHostName) -And ($_.InstanceName -eq $SQLDBInstanceName))} | Select -ExpandProperty InstanceID -First 1
IF (!$SQLDBInstanceID){$SQLDBInstanceID = "NotFound"}
# Selecting Database ID
$SQLDBID = $SQLDBArray | Where-Object {(($_.DatabaseName -eq $SQLDBName) -And ($_.InstanceID -eq $SQLDBInstanceID))} | Select -ExpandProperty DatabaseID -First 1
IF (!$SQLDBID){$SQLDBID = "NotFound"}
# Selecting SLA domain ID
$SQLDBSLADomainID = $SLADomainList | Where-Object {$_.Name -eq $SQLDBSLADomain} | Select -ExpandProperty id
IF (!$SQLDBSLADomainID){$SQLDBSLADomainID = "NotFound"}
##################################
# Posting To Snapshot URL
##################################
# Creating URL
$DBOnDemandSnapURL = $baseURL+"mssql/db/"+$SQLDBID+"/snapshot"
# Creating JSON body
$DBOnDemandSnapJSON = 
"{
  ""slaId"": ""$SQLDBSLADomainID""
}"
# Output to host
"Starting On-Demand Snapshot for:
SQLDB: $SQLDBName
SQLDBID: $SQLDBID
SLADomain: $SQLDBSLADomain
SLADomainID: $SQLDBSLADomainID"
# Posting to the API
Try 
{
Invoke-RestMethod -Method Post -Uri $DBOnDemandSnapURL -Body $DBOnDemandSnapJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Sleeping wait time specified
"Waiting $TimeDelay seconds before processing next DB"
sleep $TimeDelay
# End of per DB action below
}
# End of per DB action above
##################################
# End of script
##################################