﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates a report on the protection status of SQL DBs in Rubrik and applies a business SLA to determine status
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Windows hosts with SQL added to Rubrik with SQL DBs!
# - Configure the variables below
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse and is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below
################################################
$RubrikCluster = "rubrik.lab.local"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
# Configure your Business SLA in hours here. If any Host/Fileset doesn't have a snapshot within the last x hours specified, it will be indicated as NotMeetingSLA.
$BusinessSLAInHours = 24
# Specify the SLA Domains you DON'T want to include in the report
$SLADomainsToExclude = "Tier2Demo-Silver","Tier3Demo-Bronze"
# Specify the DB names NOT to include in the report
$DBsToExclude = "master","model","msdb"
# Files needed
$CSVOutputDirectory = "C:\RubrikSQLAutomationv1\BusinessSLAReport\"
$LogoFile = "C:\RubrikSQLAutomationv1\BusinessSLAReport\logo.png"
# HTML Color codes used for report rows
$HTMLColorSuccess = "#000000"
$HTMLColorFailure = "#e60000"
# Email settings
$EmailTo = "joshua@lab.local"
$EmailFrom = "rubrik@lab.local"
$EmailServer = "localhost"
# Email subject
$EmailSubject = "Rubrik SQLDB 24-Hour SLA Report"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception and TLS 1.2 to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Calculating timezone of machine running script
##################################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##################################
# Getting list of SLA Domains
##################################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of MS SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceListJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Building list of SQL DBs
##################################
# $SQLDBs = $SQLDBList | Where-Object {(($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($DBsToExclude -notcontains $_.name) -and ($_.state -eq "ONLINE") -and ($_.isRelic -ne "True"))} | select *
# Comment out the above, and uncomment the below if you only want to see protected DBs
$SQLDBs = ($SQLDBList | Where-Object {($_.effectiveSlaDomainId -ne "UNPROTECTED") -and ($DBsToExclude -notcontains $_.name) -and ($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.state -eq "ONLINE") -and ($_.isRelic -ne "True")} | select *)
##################################
# Building Report per SQL DB
##################################
# Creating array to store report info
$SQLDBComplianceReport=@()
# Performing per Fileset action
ForEach ($SQLDB in $SQLDBs)
{
# Setting variables required
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SLADomainName = $SQLDB.effectiveSlaDomainName
$SQLInAG = $SQLDB.isInAvailabilityGroup
$SQLInLM = $SQLDB.isLiveMount
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBHostName = $SQLDB.replicas.rootProperties.rootName
$SQLDBHostID = $SQLDB.replicas.rootProperties.rootId
$SQLDBState = $SQLDB.state
$SQLDBSLADomainID = $SQLDBInfo.effectiveSlaDomainId
##################################
# Creating URLs for click throughs
##################################
# DB
$SQLDBMgmtURL = "https://" + $RubrikCluster + "/web/bin/index.html#/object_details/mssql/" + $SQLDBID
# Host
$SQLDBHostIDURL = $SQLDBHostID.Replace("Host:::","").TrimEnd()
$SQLDBHostMgmtURL = "https://" + $RubrikCluster + "/web/bin/index.html#/protection/mssql?tab=HostsInstances&path=AllWindowsServers%252FHost%253A%253A%253A" + $SQLDBHostIDURL
# Instance
$SQLDBInstanceIDURL = $SQLDBInstanceID.Replace("MssqlInstance:::","").TrimEnd()
$SQLDBInstanceMgmtURL = $SQLDBHostMgmtURL + "%252FMssqlInstance%253A%253A%253A" + $SQLDBInstanceIDURL
##################################
# Getting SQL DB detail
##################################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$SQLDBSnapshotCount = $SQLDBInfo.snapshotCount
$SQLDBStorageUsed = $SQLDBInfo.localStorage
$SQLDBStorageUsedGB = $SQLDBStorageUsed / 1000 / 1000 / 1000
# Rounding differently depending on size
IF ($SQLDBStorageUsedGB -lt 1)
{
$SQLDBStorageUsedGB = [Math]::Round($SQLDBStorageUsedGB, 3)
}
ELSE
{
$SQLDBStorageUsedGB = [Math]::Round($SQLDBStorageUsedGB, 0)
} 
# The below is the latest recoverable point in time including logs (if logs are being protected)
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Getting the recovery model
$SQLDBRecoveryModel = $SQLDBInfo.replicas.recoveryModel
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($SQLDBSnapshotCount -eq 0)
{
# If DB is unprotected setting SLA compliance to "UNPROTECTED"
IF ($SQLDBSLADomainID -eq "UNPROTECTED")
{
$SLACompliance = "Unprotected"
}
ELSE
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
}
# Resetting values to ensure data isn't carried over between rows
$SQLDBLatestSnapshotAdjusted = $null
$SnapshotAdjustedGapInHours = $null
$SQLDBLatestRecoveryPointAdjusted = $null
$LogAdjustedGapInMinutes = $null
}
ELSE
{
# Snapshot exists, so performing actions
##################################
# Getting SQL DB snapshot detail
##################################
$SQLDBSnapshotsURL = $BaseURL+"mssql/db/"+$SQLDBID+"/snapshot"
Try 
{
$SQLDBSnapshotsJSON = Invoke-RestMethod -Uri $SQLDBSnapshotsURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBSnapshots = $SQLDBSnapshotsJSON.data
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
##################################
# Selecting snapshots
##################################
$SQLDBLatestSnapshot1 = $SQLDBSnapshots | Sort-Object -Descending date | select -ExpandProperty date -First 1
$SQLDBLatestSnapshotID = $SQLDBSnapshots | Sort-Object -Descending date | select -ExpandProperty id -First 1
$SQLDBOldestSnapshot1 = $SQLDBSnapshots | Sort-Object date | select -ExpandProperty date -First 1
##################################
# Converting Latest Snapshot
##################################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBLatestSnapshot2 = $SQLDBLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBLatestCharCount = $SQLDBLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBLatestCharSubtract = $SQLDBLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBLatestSnapshot3 = $SQLDBLatestSnapshot2.Substring(0,$SQLDBLatestSnapshot2.Length-$SQLDBLatestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBLatestSnapshot = ([datetime]::ParseExact($SQLDBLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBOldestSnapshot2 = $SQLDBOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBOldestCharCount = $SQLDBOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBOldestCharSubtract = $SQLDBOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBOldestSnapshot3 = $SQLDBOldestSnapshot2.Substring(0,$SQLDBOldestSnapshot2.Length-$SQLDBOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBOldestSnapshot = ([datetime]::ParseExact($SQLDBOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Latest Log Backup
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBLatestRecoveryPoint1 = $SQLDBLatestRecoveryPoint.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBLatestRPCharCount = $SQLDBLatestRecoveryPoint1 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBLatestRPCharSubtract = $SQLDBLatestRPCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBLatestRecoveryPoint2 = $SQLDBLatestRecoveryPoint1.Substring(0,$SQLDBLatestRecoveryPoint1.Length-$SQLDBLatestRPCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBLatestRecoveryPoint = ([datetime]::ParseExact($SQLDBLatestRecoveryPoint2,”yyyy-MM-dd HH:mm:ss”,$null))
##################################
# Calculating SLA compliance
##################################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $SQLDBLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$SQLDBLatestSnapshotAdjusted = $SQLDBLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $SQLDBLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
# Adjusting Log backup
$SQLDBLatestRecoveryPointAdjusted = $SQLDBLatestRecoveryPoint.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$LogAdjustedGap = NEW-TIMESPAN –Start $SQLDBLatestRecoveryPointAdjusted –End $SystemDateTime
$LogAdjustedGapInMinutes = $LogAdjustedGap.TotalMinutes
$LogAdjustedGapInMinutes = [Math]::Round($LogAdjustedGapInMinutes, 0)
##################################
# Setting log values to NULL if DB is in SIMPLE $SQLDBRecoveryModel
##################################
IF ($SQLDBRecoveryModel -eq "SIMPLE")
{
$SQLDBLatestRecoveryPointAdjusted = $null
$LogAdjustedGapInMinutes= $null
}
# End of ELSE action to IF snapshot count equals 0 above
}
# End of ELSE action to IF snapshot count equals 0 above
##################################
# Summarizing info into report
##################################
$SQLDBComplianceReportLine = New-Object PSObject
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SLADomainName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Host" -Value "$SQLDBHostName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HostID" -Value "$SQLDBHostID"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLDBInstanceName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "DB" -Value "$SQLDBName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "DBID" -Value "$SQLDBID"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastFullBackup" -Value "$SQLDBLatestSnapshotAdjusted"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "RecoveryModel" -Value "$SQLDBRecoveryModel"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastLogBackup" -Value "$SQLDBLatestRecoveryPointAdjusted"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "MinutesSince" -Value "$LogAdjustedGapInMinutes"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "TotalBackups" -Value "$SQLDBSnapshotCount"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "StorageUsedGB" -Value "$SQLDBStorageUsedGB"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LiveMounted" -Value "$SQLInLM"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "InAvGroup" -Value "$SQLInAG"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "DBMgmtURL" -Value "$SQLDBMgmtURL"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "InstanceMgmtURL" -Value "$SQLDBInstanceMgmtURL"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HostMgmtURL" -Value "$SQLDBHostMgmtURL"
# Adding row to array
$SQLDBComplianceReport += $SQLDBComplianceReportLine
# End of For Each SQL DB below
}
# End of For Each SQL DB above
##################################
# Getting totals
##################################
$TotalDBs = $SQLDBComplianceReport | Measure | Select -ExpandProperty Count
$TotalInstances = $SQLDBComplianceReport | Select InstanceID -Unique | Measure | Select -ExpandProperty Count
$TotalServers = $SQLDBComplianceReport | Select HostID -Unique | Measure | Select -ExpandProperty Count
$TotalFullRecoveryModel = $SQLDBComplianceReport | Where-Object {$_.RecoveryModel -eq "FULL"} | Measure | Select -ExpandProperty Count
$TotalSimpleRecoveryModel = $SQLDBComplianceReport | Where-Object {$_.RecoveryModel -eq "SIMPLE"} | Measure | Select -ExpandProperty Count
$TotalDBsMeetingSLA = $SQLDBComplianceReport | Where-Object {$_.SLACompliance -eq "MeetingSLA"} | Measure | Select -ExpandProperty Count
$TotalDBsNotMeetingSLA = $SQLDBComplianceReport | Where-Object {$_.SLACompliance -eq "NotMeetingSLA"} | Measure | Select -ExpandProperty Count
$TotalLiveMounts = $SQLDBComplianceReport | Where-Object {$_.LiveMounted -eq "true"} | Measure | Select -ExpandProperty Count
# Getting success rate
$TotalDBSuccessRate = ($TotalDBsMeetingSLA / $TotalDBs).ToString("P")
####################################################################
# SMTP Body - HTML Email style settings
####################################################################
# Start of HTML structure
$HTMLStart = @"
<!DOCTYPE html>
<html>
<head>
<style>

a {
    color: black;
}

a:link {
    text-decoration: none;
}

table.table1 {
  border-collapse: collapse;
  width: 100%;
}
table.table1 th {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #696969;
  font-size:16px
}
table.table1 td {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

table.table2 {
  border-collapse: collapse;
  width: 100%;
}
table.table2 th {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #00B2A9;
  font-size:14px
}
table.table2 td {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

</style>
</head>
<body>

<div style="overflow-x:auto;">

<table style="text-align: right; width: 100%;" border="0"; cellpadding="2"; cellspacing="2">
<tbody>
<tr>
<td style="vertical-align: top;"><img style="width: 350px; height: 95px;" alt="Rubrik" src="logo.png"><br>
</td>
</tr>
</tbody>
</table>

<br></br>
"@
# End of HTML structure
$HTMLEnd = @"
</div>

</body>
</html>
"@
##################################
# Creating HTML Summary table
##################################
$HTMLSummaryTable = @"
<table class="table1">
  <tr>
    <th>SQL Backup Report</th>
  </tr>
  <tr>
    <th>Created $SystemDateTime</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>DBs</th>
    <td>$TotalDBs</td>
    <th>SucccessRate</th>
    <td>$TotalDBSuccessRate</td>
    <th>FullRecoveryModel</th>
    <td>$TotalFullRecoveryModel</td>
  </tr>
 <tr>
    <th>Instances</th>
    <td>$TotalInstances</td>
    <th>MeetingSLA</th>
    <td>$TotalDBsMeetingSLA</td>
    <th>SimpleRecoveryModel</th>
    <td>$TotalSimpleRecoveryModel</td>

 </tr>
 <tr>
    <th>Servers</th>
    <td>$TotalServers</td>
    <th>NotMeetingSLA</th>
    <td>$TotalDBsNotMeetingSLA</td>
    <th>LiveMounts</th>
    <td>$TotalLiveMounts</td>
 </tr>
 </table>
 <br>
"@
##################################
# Creating Table 1 HTML structure
##################################
$HTMLTable1Start = @"
<table class="table1">
  <tr>
    <th>DBs Not Meeting SLA By Age</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>Server</th>
    <th>Instance</th>
    <th>DB</th>
    <th>SLADomain</th>
    <th>Compliance</th>
    <th>LastFullBackup</th>
    <th>HoursSince</th>
    <th>RecoveryModel</th>
    <th>LastLogBackup</th>
    <th>MinutesSince</th>
    <th>TotalBackups</th>
    <th>StorageUsedGB</th>
    <th>LiveMounted</th>
  </tr>
"@
$HTMLTable1End = @"
</table>
<br>
"@
##################################
# Creating Table 1 HTML Rows
##################################
# Building email list by task to put the most important objects for viewing at the top
$Table1Data = $SQLDBComplianceReport | Where-Object {$_.SLACompliance -eq "NotMeetingSLA"} | Sort-Object {[int]$_.HoursSince} -Descending
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTable1Middle = $null
# Creating table row for each line
ForEach ($Row in $Table1Data) 
{
# Setting values
$HTML1Server = $Row.Host
$HTML1Instance = $Row.Instance
$HTML1DB = $Row.DB
$HTML1SLADomain = $Row.SLADomain
$HTML1Compliance = $Row.SLACompliance
$HTML1LastFullBackup = $Row.LastFullBackup
$HTML1HoursSince = $Row.HoursSince
$HTML1RecoveryModel = $Row.RecoveryModel
$HTML1LastLogBackup = $Row.LastLogBackup
$HTML1MinutesSince = $Row.MinutesSince
$HTML1TotalBackups = $Row.TotalBackups
$HTML1StorageUsedGB = $Row.StorageUsedGB
$HTML1LiveMounted = $Row.LiveMounted
$HTML1DBMgmtURL = $Row.DBMgmtURL
$HTML1InstanceMgmtURL = $Row.InstanceMgmtURL
$HTML1HostMgmtURL = $Row.HostMgmtURL
# Setting colors
IF ($HTML1Compliance -eq "MeetingSLA"){$HTMLStatusColor =  $HTMLColorSuccess}
IF ($HTML1Compliance -eq "NotMeetingSLA"){$HTMLStatusColor =  $HTMLColorFailure}
# Building HTML table row
$HTMLReportTable1Row = "
<tr>
	<td><a href=$HTML1HostMgmtURL>$HTML1Server</a></td>
    <td><a href=$HTML1InstanceMgmtURL>$HTML1Instance</a></td>
    <td><a href=$HTML1DBMgmtURL>$HTML1DB</a></td>
    <td>$HTML1SLADomain</td>
    <td><font color=$HTMLStatusColor>$HTML1Compliance</font></td>
    <td>$HTML1LastFullBackup</td>
    <td>$HTML1HoursSince</td>
    <td>$HTML1RecoveryModel</td>
    <td>$HTML1LastLogBackup</td>
    <td>$HTML1MinutesSince</td>
    <td>$HTML1TotalBackups</td>
    <td>$HTML1StorageUsedGB</td>
    <td>$HTML1LiveMounted</td>
  </tr>
"
# Adding row to table
$HTMLReportTable1Middle += $HTMLReportTable1Row
}
##################################
# Putting Table 1 together
##################################
$HTMLTable1 = $HTMLTable1Start + $HTMLReportTable1Middle + $HTMLTable1End
##################################
# Creating Table 2 HTML structure
##################################
$HTMLTable2Start = @"
<table class="table1">
  <tr>
    <th>All DBs by Server, Instance & Name</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>Server</th>
    <th>Instance</th>
    <th>DB</th>
    <th>SLADomain</th>
    <th>Compliance</th>
    <th>LastFullBackup</th>
    <th>HoursSince</th>
    <th>RecoveryModel</th>
    <th>LastLogBackup</th>
    <th>MinutesSince</th>
    <th>TotalBackups</th>
    <th>StorageUsedGB</th>
    <th>LiveMounted</th>
  </tr>
"@
$HTMLTable2End = @"
</table>
<br>
"@
##################################
# Creating Table 2 HTML Rows
##################################
# Building email list by task to put the most important objects for viewing at the top
$Table2Data = $SQLDBComplianceReport | Sort-Object Server, Instance, DB
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTable2Middle = $null
# Creating table row for each line
ForEach ($Row in $Table2Data) 
{
# Setting values
$HTML2Server = $Row.Host
$HTML2Instance = $Row.Instance
$HTML2DB = $Row.DB
$HTML2SLADomain = $Row.SLADomain
$HTML2Compliance = $Row.SLACompliance
$HTML2LastFullBackup = $Row.LastFullBackup
$HTML2HoursSince = $Row.HoursSince
$HTML2RecoveryModel = $Row.RecoveryModel
$HTML2LastLogBackup = $Row.LastLogBackup
$HTML2MinutesSince = $Row.MinutesSince
$HTML2TotalBackups = $Row.TotalBackups
$HTML2StorageUsedGB = $Row.StorageUsedGB
$HTML2LiveMounted = $Row.LiveMounted
$HTML2DBMgmtURL = $Row.DBMgmtURL
$HTML2InstanceMgmtURL = $Row.InstanceMgmtURL
$HTML2HostMgmtURL = $Row.HostMgmtURL
# Setting colors
IF ($HTML2Compliance -eq "MeetingSLA"){$HTMLStatusColor =  $HTMLColorSuccess}
IF ($HTML2Compliance -eq "NotMeetingSLA"){$HTMLStatusColor =  $HTMLColorFailure}
# Building HTML table row
$HTMLReportTable2Row = "
<tr>
	<td><a href=$HTML2HostMgmtURL>$HTML2Server</a></td>
    <td><a href=$HTML2InstanceMgmtURL>$HTML2Instance</a></td>
    <td><a href=$HTML2DBMgmtURL>$HTML2DB</a></td>
    <td>$HTML2SLADomain</td>
    <td><font color=$HTMLStatusColor>$HTML2Compliance</font></td>
    <td>$HTML2LastFullBackup</td>
    <td>$HTML2HoursSince</td>
    <td>$HTML2RecoveryModel</td>
    <td>$HTML2LastLogBackup</td>
    <td>$HTML2MinutesSince</td>
    <td>$HTML2TotalBackups</td>
    <td>$HTML2StorageUsedGB</td>
    <td>$HTML2LiveMounted</td>
  </tr>
"
# Adding row to table
$HTMLReportTable2Middle += $HTMLReportTable2Row
}
##################################
# Putting Table 2 together
##################################
$HTMLTable2 = $HTMLTable2Start + $HTMLReportTable2Middle + $HTMLTable2End
##################################
# Creating Report
##################################
# Building HTML report:
$HTMLReport = $HTMLStart + $HTMLSummaryTable + $HTMLTable1 +  $HTMLTable2 + $HTMLEnd
# Replacing any 100.00% strings with 100% for easier reading
$HTMLReport = $HTMLReport.Replace("100.00%","100%").TrimEnd()
##################################
# Creating CSVs
##################################
# Creating the file names
$SQLDBComplianceReportCSVFile = $CSVOutputDirectory + "SQLDBReport-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
# Exporting to CSV
$SQLDBComplianceReport | Export-Csv -Path $SQLDBComplianceReportCSVFile -NoTypeInformation -Force
# Putting all 3 files together for the email attachement
$Attachments = "$SQLDBComplianceReportCSVFile","$LogoFile"
##################################
# Sending email using function
##################################
Try
{
Send-MailMessage -To $EmailTo -From $EmailFrom -Body $HTMLReport -Subject $EmailSubject -Attachments $Attachments -SmtpServer $EmailServer -BodyAsHtml 
}
Catch
{
$Error[0] | Format-List -Force
}
###############################################
# End of script
###############################################