﻿########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates a report on the protection status of SQL DBs in Rubrik and applies a business SLA to determine status
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - Windows hosts with SQL added to Rubrik with SQL DBs!
# - Configure the variables below
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse and is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
# Configure your Business SLA in hours here. If any Host/Fileset doesn't have a snapshot within the last x hours specified, it will be indicated as NotMeetingSLA.
$BusinessSLAInHours = 24
# Specify the SLA Domains you DON'T want to include in the report
$SLADomainsToExclude = "Tier2Demo-Silver","Tier3Demo-Bronze"
# Specify the DB names NOT to include in the report
$DBsToExclude = "master","model","msdb"
# Config CSV output
$EnableCSVOutput = $TRUE
$CSVOutputDirectory = "C:\RubrikSQLAutomationv1\BusinessSLAReportv1\"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception and TLS 1.2 to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Calculating timezone of machine running script
##################################
$SystemDateTime = Get-Date
sleep 1
$UTCDateTime = [System.DateTime]::UtcNow
# Caculating difference
$SystemTimeGapToUTC = NEW-TIMESPAN –Start $UTCDateTime –End $SystemDateTime
$SystemTimeGapToUTCInHours = $SystemTimeGapToUTC.TotalHours
$SystemTimeGapToUTCInHours = [Math]::Round($SystemTimeGapToUTCInHours, 1)
##################################
# Getting list of SLA Domains
##################################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of MS SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceListJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
##################################
# Building list of SQL DBs
##################################
# $SQLDBs = $SQLDBList | Where-Object {(($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($DBsToExclude -notcontains $_.name) -and ($_.state -eq "ONLINE") -and ($_.isRelic -ne "True"))} | select *
# Comment out the above, and uncomment the below if you only want to see protected DBs
$SQLDBs = ($SQLDBList | Where-Object {($_.effectiveSlaDomainId -ne "UNPROTECTED") -and ($DBsToExclude -notcontains $_.name) -and ($SLADomainsToExclude -notcontains $_.effectiveSlaDomainName) -and ($_.state -eq "ONLINE") -and ($_.isRelic -ne "True")} | select *)
##################################
# Building Report per SQL DB
##################################
# Creating array to store report info
$SQLDBComplianceReport=@()
# Performing per Fileset action
ForEach ($SQLDB in $SQLDBs)
{
# Setting variables required
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SLADomainName = $SQLDB.effectiveSlaDomainName
$SQLInAG = $SQLDB.isInAvailabilityGroup
$SQLInLM = $SQLDB.isLiveMount
$SQLDBInstanceName = $SQLDB.replicas.instanceName
$SQLDBHostName = $SQLDB.replicas.rootProperties.rootName
$SQLDBState = $SQLDB.state
$SQLDBEffectiveSLADomainID = $SQLDB.effectiveSlaDomainId
##################################
# Getting SQL DB detail
##################################
$SQLDBInfoURL = $baseURL+"mssql/db/"+$SQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Pulling detail
$SQLDBSnapshotCount = $SQLDBInfo.snapshotCount
$SQLDBInstanceID = $SQLDBInfo.primaryClusterId
$SQLDBSLADomainID = $SQLDBInfo.effectiveSlaDomainId
$SQLDBStorageUsed = $SQLDBInfo.localStorage
$SQLDBStorageUsedGB = $SQLDBStorageUsed / 1000 / 1000 / 1000
# Rounding differently depending on size
IF ($SQLDBStorageUsedGB -lt 1)
{
$SQLDBStorageUsedGB = [Math]::Round($SQLDBStorageUsedGB, 3)
}
ELSE
{
$SQLDBStorageUsedGB = [Math]::Round($SQLDBStorageUsedGB, 0)
} 
# The below is the latest recoverable point in time including logs (if logs are being protected)
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Getting the recovery model
$SQLDBRecoveryModel = $SQLDBInfo.replicas.recoveryModel
##################################
# IF no snapshots exist, not performing any snapshot work as the object is pending a first full backup
##################################
IF ($SQLDBSnapshotCount -eq 0)
{
# If DB is unprotected setting SLA compliance to "UNPROTECTED"
IF ($SQLDBEffectiveSLADomainID -eq "UNPROTECTED")
{
$SLACompliance = "Unprotected"
}
ELSE
{
# No snapshot exists, must be awaiting first full
$SLACompliance = "PendingFirstFull"
}
# Resetting values to ensure data isn't carried over between rows
$SQLDBLatestSnapshotAdjusted = $null
$SnapshotAdjustedGapInHours = $null
$SQLDBLatestRecoveryPointAdjusted = $null
$LogAdjustedGapInMinutes = $null
}
ELSE
{
# Snapshot exists, so performing actions
##################################
# Getting SQL DB snapshot detail
##################################
$SQLDBSnapshotsURL = $baseURL+"mssql/db/"+$SQLDBID+"/snapshot"
Try 
{
$SQLDBSnapshotsJSON = Invoke-RestMethod -Uri $SQLDBSnapshotsURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBSnapshots = $SQLDBSnapshotsJSON.data
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
##################################
# Selecting snapshots
##################################
$SQLDBLatestSnapshot1 = $SQLDBSnapshots | Sort-Object -Descending date | select -ExpandProperty date -First 1
$SQLDBLatestSnapshotID = $SQLDBSnapshots | Sort-Object -Descending date | select -ExpandProperty id -First 1
$SQLDBOldestSnapshot1 = $SQLDBSnapshots | Sort-Object date | select -ExpandProperty date -First 1
##################################
# Converting Latest Snapshot
##################################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBLatestSnapshot2 = $SQLDBLatestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBLatestCharCount = $SQLDBLatestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBLatestCharSubtract = $SQLDBLatestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBLatestSnapshot3 = $SQLDBLatestSnapshot2.Substring(0,$SQLDBLatestSnapshot2.Length-$SQLDBLatestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBLatestSnapshot = ([datetime]::ParseExact($SQLDBLatestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Oldest Snapshot
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBOldestSnapshot2 = $SQLDBOldestSnapshot1.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBOldestCharCount = $SQLDBOldestSnapshot2 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBOldestCharSubtract = $SQLDBOldestCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBOldestSnapshot3 = $SQLDBOldestSnapshot2.Substring(0,$SQLDBOldestSnapshot2.Length-$SQLDBOldestCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBOldestSnapshot = ([datetime]::ParseExact($SQLDBOldestSnapshot3,”yyyy-MM-dd HH:mm:ss”,$null))
##########################
# Converting Latest Log Backup
##########################
# Step 1 - Removing characters and trimming snapshot string for conversion
$SQLDBLatestRecoveryPoint1 = $SQLDBLatestRecoveryPoint.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Counting characters past 19 (required amount for conversion)
$SQLDBLatestRPCharCount = $SQLDBLatestRecoveryPoint1 | Measure-Object -Character | Select -ExpandProperty Characters
$SQLDBLatestRPCharSubtract = $SQLDBLatestRPCharCount - 19
# Step 3 - Subtracting the diff to ensure conversion works
$SQLDBLatestRecoveryPoint2 = $SQLDBLatestRecoveryPoint1.Substring(0,$SQLDBLatestRecoveryPoint1.Length-$SQLDBLatestRPCharSubtract)
# Step 4 - Converting string to PowerShell datetime object
$SQLDBLatestRecoveryPoint = ([datetime]::ParseExact($SQLDBLatestRecoveryPoint2,”yyyy-MM-dd HH:mm:ss”,$null))
##################################
# Calculating SLA compliance
##################################
# Calculating time gap from latest snap to current time
$SnapshotGap = NEW-TIMESPAN –Start $SQLDBLatestSnapshot –End $UTCDateTime
$SnapshotGapInHours = $SnapshotGap.TotalHours
$SnapshotGapInHours = [Math]::Round($SnapshotGapInHours, 1)
# Setting SLA outcome
IF (($SnapshotGapInHours -gt $BusinessSLAInHours) -or ($SnapshotGapInHours -eq $null))
{
$SLACompliance = "NotMeetingSLA"
}
ELSE
{
$SLACompliance = "MeetingSLA"
}
##################################
# Calculating Adjusted snapshots by timezone of machine running script for easier reading in the report
##################################
# Adjusting Latest Snapshot gap
$SQLDBLatestSnapshotAdjusted = $SQLDBLatestSnapshot.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$SnapshotAdjustedGap = NEW-TIMESPAN –Start $SQLDBLatestSnapshotAdjusted –End $SystemDateTime
$SnapshotAdjustedGapInHours = $SnapshotAdjustedGap.TotalHours
$SnapshotAdjustedGapInHours = [Math]::Round($SnapshotAdjustedGapInHours, 1)
# Adjusting Log backup
$SQLDBLatestRecoveryPointAdjusted = $SQLDBLatestRecoveryPoint.AddHours($SystemTimeGapToUTCInHours)
# Calculating diff
$LogAdjustedGap = NEW-TIMESPAN –Start $SQLDBLatestRecoveryPointAdjusted –End $SystemDateTime
$LogAdjustedGapInMinutes = $LogAdjustedGap.TotalMinutes
$LogAdjustedGapInMinutes = [Math]::Round($LogAdjustedGapInMinutes, 0)
##################################
# Setting log values to NULL if DB is in SIMPLE $SQLDBRecoveryModel
##################################
IF ($SQLDBRecoveryModel -eq "SIMPLE")
{
$SQLDBLatestRecoveryPointAdjusted = $null
$LogAdjustedGapInMinutes= $null
}
# End of ELSE action to IF snapshot count equals 0 above
}
# End of ELSE action to IF snapshot count equals 0 above
##################################
# Summarizing info into report
##################################
$SQLDBComplianceReportLine = New-Object PSObject
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value "$SLADomainName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Host" -Value "$SQLDBHostName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLDBInstanceName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "DB" -Value "$SQLDBName"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "SLACompliance" -Value "$SLACompliance"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastFullBackup" -Value "$SQLDBLatestSnapshotAdjusted"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "HoursSince" -Value "$SnapshotAdjustedGapInHours"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "RecoveryModel" -Value "$SQLDBRecoveryModel"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LastLogBackup" -Value "$SQLDBLatestRecoveryPointAdjusted"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "MinutesSince" -Value "$LogAdjustedGapInMinutes"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "TotalBackups" -Value "$SQLDBSnapshotCount"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "StorageUsedGB" -Value "$SQLDBStorageUsedGB"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "LiveMounted" -Value "$SQLInLM"
$SQLDBComplianceReportLine | Add-Member -MemberType NoteProperty -Name "InAvGroup" -Value "$SQLInAG"
# Adding row to array
$SQLDBComplianceReport += $SQLDBComplianceReportLine
# End of For Each SQL DB below
}
# End of For Each SQL DB above
#
##################################
# Show SLA compliance reports
##################################
"----------------------------------------------"
"SLA Compliance Report"
"----------------------------------------------"
$SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB | Format-Table
$SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB | Out-GridView -Title "SQL DB SLA Compliance Report"
##################################
# Output compliance report to CSV if enabled
##################################
IF ($EnableCSVOutput -eq $TRUE)
{
$CSVFile = $CSVOutputDirectory + "Rubrik-BusinessSLAReport-SQL-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$SQLDBComplianceReport | Sort-Object SLACompliance,SLADomain,Host,Instance,DB | Export-Csv $CSVFile -Force -NoTypeInformation
}
###############################################
# End of script
###############################################