﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script exports the SQL DB specified with either default file locations or new directories for multiple .mdf, .ndf, and .ldf files
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A SQL database protected with an SLA domain
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
# Either remove the below and pass them into the script as variables or hard code
# DB which to take export  FROM
$SourceSQLHostName = "SQL16-VM01.lab.local"
$SourceSQLInstanceName = "INSTANCE01"
$SourceSQLDBName = "SQL16-VM01-DemoDB01"
# Host/instance to export  TO, with DB name
$TargetSQLHostName = "SQL16-VM01.lab.local"
$TargetSQLInstanceName = "INSTANCE01"
$TargetSQLDBName = "SQL16-VM01-DemoDB01Export10"
# The below can be set to a directory on the host (make sure its valid first), or Default uses the DB default, just like a native SQL restore would
$DBPath = "Default"
$LogPath = "Default"
# Overwrite target DB if already exists, true or false, has to be one or the other
$OverwriteTargetDB = "true"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
##################################
$SQLInstanceArray = @()
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Adding to array
$SQLInstanceArrayLine = new-object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostNameID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
ForEach ($SQLDB in $SQLDBList)
{
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
}
# Adding to array
$SQLDBArrayLine = New-Object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArray += $SQLDBArrayLine
}
######################
# Converting the paths to usable format for JSON, if not default
######################
IF ($DBPath -ne "Default")
{
$DBPath = $DBPath.Replace("\","\\")
}
IF ($LogPath -ne "Default")
{
$LogPath = $LogPath.Replace("\","\\")
}
######################
# Selecting IDs required
######################
# Selecting the database ID
$SourceSQLDBID = $SQLDBArray | Where-Object {(($_.HostName -eq $SourceSQLHostName) -and ($_.InstanceName -eq $SourceSQLInstanceName) -and ($_.DatabaseName -eq $SourceSQLDBName))} | Select -ExpandProperty DatabaseID
# Selecting the target instance ID
$TargetInstanceID = $SQLInstanceArray | Where-Object {(($_.InstanceName -eq $TargetSQLInstanceName) -and ($_.Hostname -eq $TargetSQLHostName))} | Select -ExpandProperty InstanceID
######################
# Getting DB info
######################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SourceSQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting latest point in time
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
# Converting latestRecoveryPoint to values needed
$SQLDBLatestRecoveryPointURL= $SQLDBLatestRecoveryPoint.Replace(":","%3A")
$SQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($SQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
######################
# Getting DB File info
######################
$SQLDBFilesURL = $InternalURL+"mssql/db/"+$SourceSQLDBID+"/restore_files?time="+$SQLDBLatestRecoveryPointURL
Try 
{
$SQLDBFiles = Invoke-RestMethod -Uri $SQLDBFilesURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
######################
# Starting JSON build
######################
$SQLDBJSONStart = 
"{
""recoveryPoint"":
  {
  ""timestampMs"":$SQLDBLatestRecoveryPointMS
  },
""targetInstanceId"": ""$TargetInstanceID"",
""targetDatabaseName"": ""$TargetSQLDBName"",
""targetFilePaths"":[
"
######################
# Middle JSON build
######################
$SQLDBJSONMiddle = $null
# Selecting random ID to prevent file conflict
$Range = 1..10000
$RandomID = $Range | Get-Random
# Starting counters from ID
$SQLDBFileCounter = 1
$SQLNDFFileCounter = 1
$SQLDBLogFileCounter = 1
# Creating JSON VM array for all the VMs in the VPG
ForEach ($SQLDBFile in $SQLDBFiles)
{
# Setting the DB file variables
$logicalName = $SQLDBFile.logicalName
$originalPath = $SQLDBFile.originalPath
$originalName = $SQLDBFile.originalName
# Converting original path
$originalPathConverted = $originalPath.Replace("\","\\")
######################
# MDF settings
######################
IF ($originalName -match ".mdf")
{
# Setting DB path
If ($DBPath -eq "Default")
{
$ExportPath = $originalPathConverted
}
ELSE
{
$ExportPath = $DBPath
}
# Setting New DB File Name
$NewFilename = $TargetSQLDBName + "_ID" + $RandomID +"_" + + $SQLDBFileCounter + ".mdf"
# Incrementing DB counter to prevent duplicates
$SQLDBFileCounter ++
}
######################
# NDF settings
######################
IF ($originalName -match ".ndf")
{
# Setting DB path
If ($DBPath -eq "Default")
{
$ExportPath = $originalPathConverted
}
ELSE
{
$ExportPath = $DBPath
}
# Setting New DB File Name
$NewFilename = $TargetSQLDBName +  "_ID" + $RandomID +"_" + $SQLNDFFileCounter + ".ndf"
# Incrementing DB counter to prevent duplicates
$SQLNDFFileCounter ++
}
######################
# LDF settings
######################
IF ($originalName -match ".ldf")
{
# Setting DB path
If ($LogPath -eq "Default")
{
$ExportPath = $originalPathConverted
}
ELSE
{
$ExportPath = $LogPath
}
# Setting New DB File Name
$NewFilename = $TargetSQLDBName + "_ID" + $RandomID +"_Log" + $SQLDBLogFileCounter + ".ldf"
# Incrementing DB counter to prevent duplicates
$SQLDBLogFileCounter ++
}
######################
# Middle JSON
######################
$SQLDBJSONFile = 
"{
""exportPath"":""$ExportPath"",
""logicalName"":""$logicalName"",
""newFilename"":""$NewFilename""
}"
# Running if statement to check if this is the first file in the array, if not then a comma is added to string
if ($SQLDBJSONMiddle -ne $null)
{
$SQLDBJSONFile = ",
" + $SQLDBJSONFile
}
# Adding the DB file to the JSON middle
$SQLDBJSONMiddle = $SQLDBJSONMiddle + $SQLDBJSONFile
}
######################
# End JSON & Compiling
######################
# Creating the end of the JSON request
$SQLDBJSONEnd = "],
""finishRecovery"": true,
""allowOverwrite"": $OverwriteTargetDB
}"
# Putting the JSON request together and outputting the request
$SQLDBJSON = $SQLDBJSONStart + $SQLDBJSONMiddle + $SQLDBJSONEnd
######################
# POST SQL Export
######################
# Creating POST URL
$SQLDBExportURL = $BaseURL+"mssql/db/"+$SourceSQLDBID+"/export"
# POST to API
Try 
{
$SQLDBExportResponse = Invoke-RestMethod -Method Post -Uri $SQLDBExportURL -Body $SQLDBJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBExportResponse.links.href
##################################
# Getting Job status on a loop
##################################
# Setting counter
$SQLJobStatusCount = 0
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLJobStatusURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
$SQLJobStatus = "FAILED"
}
# Output to host
"SQLJobStatus: $SQLJobStatus"
# Waiting 15 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 15
}
# Will run until it succeeds, fails, or hits 24 hours (5760 is number of seconds in a day / 15)
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "FAILED") -OR ($SQLJobStatusCount -eq 5760))
##################################
# Perform any next actions you want here
##################################
IF ($SQLJobStatus -eq "SUCCEEDED")
{
# Put your next actions, if needed, here
}
##################################
# End of script
##################################