﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script live mounts the SQL DB specified from the most recent time available (including log backups if available) to the same or a different host/instance
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A SQL database protected with an SLA domain
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse and is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
# Either remove the below and pass them into the script as variables or hard code
# DB which to take a live mount FROM
$SourceSQLHostName = "SQL16-VM01.lab.local"
$SourceSQLInstanceName = "INSTANCE01"
$SourceSQLDBName = "SQL16-VM01-DemoDB01"
# Host/instance to live mount TO, with DB name
$TargetSQLHostName = "SQL16-VM01.lab.local"
$TargetSQLInstanceName = "INSTANCE01"
$TargetSQLDBName = "LastLogBackupLiveMount"
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of MS SQL DBs
##################################
$SQLDBListURL = $BaseURL+"mssql/db?limit=5000"
Try 
{
$SQLDBListJSON = Invoke-RestMethod -Uri $SQLDBListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBList = $SQLDBListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SQL Instances
##################################
$SQLInstanceListURL = $BaseURL+"mssql/instance?limit=5000"
Try 
{
$SQLInstanceJSON = Invoke-RestMethod -Uri $SQLInstanceListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLInstanceList = $SQLInstanceJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of SQL Availability groups
###############################################
$SQLAvailabilityGroupListURL = $InternalURL+"mssql/availability_group"
Try 
{
$SQLAvailabilityGroupListJSON = Invoke-RestMethod -Uri $SQLAvailabilityGroupListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLAvailabilityGroupList = $SQLAvailabilityGroupListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Building a list of SQL instances by hostname, needed to enable selection of the correct instance
##################################
$SQLInstanceArray = @()
ForEach ($SQLInstance in $SQLInstanceList)
{
$SQLInstanceName = $SQLInstance.name
$SQLInstanceID = $SQLInstance.id
$SQLInstanceHostName = $SQLInstance.rootProperties.rootName
$SQLInstanceHostID = $SQLInstance.rootProperties.rootId
# Getting DBs for each instance, to aid with troubleshooting
$SQLInstanceDBs = $SQLDBList | Where-Object {$_.instanceID -eq $SQLInstanceID} | Select -ExpandProperty Name
$SQLInstanceDBCount = $SQLInstanceDBs.Count
# Adding to array
$SQLInstanceArrayLine = New-Object PSObject
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLInstanceName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLInstanceID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLInstanceHostName"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "HostNameID" -Value "$SQLInstanceHostID"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBCount" -Value "$SQLInstanceDBCount"
$SQLInstanceArrayLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceDBs"
$SQLInstanceArray += $SQLInstanceArrayLine
}
###############################################
# Building a list of SQL DBS with hostname
###############################################
$SQLDBArray = @()
ForEach ($SQLDB in $SQLDBList)
{
$SQLDBName = $SQLDB.name
$SQLDBID = $SQLDB.id
$SQLDBInstanceName = $SQLDB.instanceName
$SQLDBInstanceID = $SQLDB.instanceId
$SQLDBInAvailabilityGroup = $SQLDB.isInAvailabilityGroup
IF ($SQLDBInAvailabilityGroup -eq "True")
{
$SQLDBHostName = "AvailabilityGroup"
$SQLAvailabilityGroupID = $SQLDB.availabilityGroupId
$SQLDBInstanceName = $SQLAvailabilityGroupList | Where-Object {$_.id -eq $SQLAvailabilityGroupID} | Select -ExpandProperty name
}
ELSE
{
# Selecting hostname by instanceID
$SQLDBHostName = $SQLInstanceArray | Where-Object {$_.InstanceID -eq $SQLDBInstanceID} | Select -ExpandProperty Hostname
}
# Adding to array
$SQLDBArrayLine = New-Object PSObject
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "HostName" -Value "$SQLDBHostName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceName" -Value "$SQLDBInstanceName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "InstanceID" -Value "$SQLDBInstanceID"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseName" -Value "$SQLDBName"
$SQLDBArrayLine | Add-Member -MemberType NoteProperty -Name "DatabaseID" -Value "$SQLDBID"
$SQLDBArray += $SQLDBArrayLine
}
###############################################
# Getting target Instance ID
###############################################
$TargetInstanceID = $SQLInstanceArray | Where-Object {(($_.HostName -eq $TargetSQLHostName) -AND ($_.InstanceName -eq $TargetSQLInstanceName))} | Select -ExpandProperty InstanceID -First 1
##################################
# Selecting the source DB ID whcih to take a live mount from
##################################
$SourceSQLDBID = $SQLDBArray | Where-Object {(($_.HostName -eq $SourceSQLHostName) -AND ($_.InstanceName -eq $SourceSQLInstanceName) -AND ($_.DatabaseName -eq $SourceSQLDBName))} | Select -ExpandProperty DatabaseID
##################################
# Only progressing if required IDs found
##################################
IF (($TargetInstanceID -eq $null) -or ($SourceSQLDBID -eq $null))
{
"SourceDB or $TargetInstance IDs not found, check your spelling and try again"
}
ELSE
{
######################
# Getting DB info
######################
$SQLDBInfoURL = $BaseURL+"mssql/db/"+$SourceSQLDBID
Try 
{
$SQLDBInfo = Invoke-RestMethod -Uri $SQLDBInfoURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting latest point in time
$SQLDBLatestRecoveryPoint = $SQLDBInfo.latestRecoveryPoint
"UsingRecoveryPoint: $SQLDBLatestRecoveryPoint"
# Converting latestRecoveryPoint to values needed
$SQLDBLatestRecoveryPointURL= $SQLDBLatestRecoveryPoint.Replace(":","%3A")
$SQLDBLatestRecoveryPointMS = (Get-Date (Get-Date -Date ([DateTime]::Parse($SQLDBLatestRecoveryPoint)).ToUniversalTime()) -UFormat %s) + "000"
###########################################
# Create JSON then POST to Live Mount REST API URL 
###########################################
# Creating POST URL
$SQLDBLiveMountURL = $baseURL+"mssql/db/"+$SourceSQLDBID+"/mount"
# Creating JSON for POST
$SQLDBLiveMountJSON =
"{
""recoveryPoint"":
        {
       ""timestampMs"":$SQLDBLatestRecoveryPointMS
        },
""targetInstanceId"": ""$TargetInstanceID"",
""mountedDatabaseName"": ""$TargetSQLDBName""
}"
# POST to API
Try 
{
$SQLDBLiveMountResponse = Invoke-RestMethod -Method Post -Uri $SQLDBLiveMountURL -Body $SQLDBLiveMountJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SQLDBLiveMountSuccess = $TRUE
}
Catch 
{
$SQLDBLiveMountSuccess = $FALSE
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting the URL to monitor the Job status
##################################
$SQLJobStatusURL = $SQLDBLiveMountResponse.links.href
##################################
# Getting Job status on a loop
##################################
# Setting counter
$SQLJobStatusCount = 0
DO
{
$SQLJobStatusCount ++
# Getting status
Try 
{
$SQLJobStatusResponse = Invoke-RestMethod -Uri $SQLJobStatusURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
# Setting status
$SQLJobStatus = $SQLJobStatusResponse.status
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
$SQLJobStatus = "FAILED"
}
# Output to host
"SQLJobStatus: $SQLJobStatus"
# Waiting 15 seconds before trying again, but only if not succeeded
IF ($SQLJobStatus -ne "SUCCEEDED")
{
sleep 15
}
# Will run until it succeeds, fails, or hits 24 hours (5760 is number of seconds in a day / 15)
} Until (($SQLJobStatus -eq "SUCCEEDED") -OR ($SQLJobStatus -eq "FAILED") -OR ($SQLJobStatusCount -eq 5760))
##################################
# Perform any next actions you want here
##################################
IF ($SQLJobStatus -eq "SUCCEEDED")
{
# Put your next actions, if needed, here
}
##################################
# End of bypass for sourceDB ID and target instance ID not found below
##################################
}
# End of bypass for sourceDB ID and target instance ID not found above
##################################
# End of script
##################################