﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script adds a Windows host to a Rubrik cluster
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - At least 1 Windows host with the Rubrik connector installed
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, 
# even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$SourceRubrikCluster = "192.168.0.102"
$TargetRubrikCluster = "192.168.5.103"
# Script dir
$ScriptDirectory = "C:\RubrikSQLAutomationv1\"
# SQL backup settings
$SQLSLADomain = "SQL"
$SQLLogBackupFrequencyMins = "3600"
$SQLLogBackupRetentionHours = "168"
$SQLCopyOnlyBackup = "false"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Adding security assembly
Add-Type -AssemblyName System.Security
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
# Prompting for credentials
$RubrikCredentials = Get-Credential -Message "Enter your login credentials for Rubrik"
# Setting user
$RubrikUser = $RubrikCredentials.UserName
# Encrypting user using machine key
$UserBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikUser)
$SecureUser = [Security.Cryptography.ProtectedData]::Protect($UserBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUserToStore = [System.Convert]::ToBase64String($SecureUser)
# Setting password to use in connection string
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
# Encrypting password using machine key
$PasswordBytes = [System.Text.Encoding]::Unicode.GetBytes($RubrikPassword)
$SecurePassword = [Security.Cryptography.ProtectedData]::Protect($PasswordBytes, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPasswordToStore = [System.Convert]::ToBase64String($SecurePassword)
# Creating password file name
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
$RubrikUserToStore | Out-File $RubrikCredentialsFile -Force
$RubrikPasswordToStore | Out-File $RubrikCredentialsFile -Append
}
ELSE
{
# Importing credentials
$RubrikCredentials = Get-Content $RubrikCredentialsFile
# Getting username
$RubrikUser = $RubrikCredentials[0]
$SecureStr = [System.Convert]::FromBase64String($RubrikUser)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikUser = [System.Text.Encoding]::Unicode.GetString($StringBytes)
# Getting password
$RubrikPasswordSecureStr = $RubrikCredentials[1]
$SecureStr = [System.Convert]::FromBase64String($RubrikPasswordSecureStr)
$StringBytes = [Security.Cryptography.ProtectedData]::Unprotect($SecureStr, $null, [Security.Cryptography.DataProtectionScope]::LocalMachine)
$RubrikPassword = [System.Text.Encoding]::Unicode.GetString($StringBytes)
}
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]'Ssl3,Tls,Tls11,Tls12'
##################################
# Step 1 - Removing Hosts
##################################
"----------------------------------------
Step 1 - RemovingHostsFrom: $SourceRubrikCluster
----------------------------------------"
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $SourceRubrikCluster + "/api/v1/"
$InternalURL = "https://" + $SourceRubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
###############################################
# Getting List of Hosts 
###############################################
$HostListURL = $BaseURL+"host"
# Getting Hosts
Try 
{
$HostList = Invoke-RestMethod -Uri $HostListURL -Method GET -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type | Select -ExpandProperty data 
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
$HostsToMigrate = $HostList | Where {$_.status -ne "Connected"} | Select -ExpandProperty hostname
$HostsToMigrateCount = $HostsToMigrate | Measure | Select -ExpandProperty Count
$HostsToMigrateCounter = 0
###############################################
# Removing Each Host
###############################################
# Removing each host
ForEach ($HostToMigrate in $HostsToMigrate)
{
# Host output
$HostsToMigrateCounter ++
"$HostsToMigrateCounter/$HostsToMigrateCount RemovingHost: $HostToMigrate"
# Getting ID
$HostID = $HostList | Where {$_.hostname -eq $HostToMigrate} | Select -ExpandProperty id
$HostIDShort = $HostID.Replace("Host:::","")
# Creating URL
$RemoveHostURL = $BaseURL+"host/"+$HostID
# POST To Add Host
Try 
{
$RemoveHostResponse = Invoke-RestMethod -Uri $RemoveHostURL -Method DELETE -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$HostRemoved = $TRUE
}
Catch 
{
$HostRemoved = $FALSE
}
# Output of result
"$HostsToMigrateCounter/$HostsToMigrateCount HostRemoved: $HostRemoved"
}
##################################
# Step 2 - Removing Hosts
##################################
"----------------------------------------
Step 2 - AddingHostsTo: $TargetRubrikCluster
----------------------------------------"
Sleep 10
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $TargetRubrikCluster + "/api/v1/"
$InternalURL = "https://" + $TargetRubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
###############################################
# Adding Each Host
###############################################
$HostsToMigrateCounter = 0
ForEach ($HostToMigrate in $HostsToMigrate)
{
# Host output
$HostsToMigrateCounter ++
"$HostsToMigrateCounter/$HostsToMigrateCount AddingHost: $HostToMigrate"
# Creating JSON
$AddHostJSON =
"{
""hostname"": ""$HostToMigrate""
}"
# Creating URL
$AddHostURL = $BaseURL+"host"
# POST To Add Host
Try 
{
Invoke-RestMethod -Uri $AddHostURL -Method POST -Body $AddHostJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type | Out-Null
$HostAdded = $TRUE
}
Catch 
{
$HostAdded = $FALSE
}
# Output of result
"$HostsToMigrateCounter/$HostsToMigrateCount HostAdded: $HostAdded"
}
##################################
# Getting list of Windows hosts
##################################
$WindowsHostsURL = $BaseURL+"host?operating_system_type=Windows"
Try 
{
$WindowsHostsJSON = Invoke-RestMethod -Uri $WindowsHostsURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$WindowsHosts = $WindowsHostsJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainsURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainsJSON = Invoke-RestMethod -Uri $SLADomainsURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomains = $SLADomainsJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting SLA Domain ID
$SQLSLADomainID = $SLADomains | Where-Object {$_.name -eq $SQLSLADomain} | Select -ExpandProperty id
##################################
# Step 3 - Assigning SLAs To Hosts
##################################
"----------------------------------------
Step 3 - AssigningSLAsTo: $TargetRubrikCluster
----------------------------------------"
Sleep 10
##################################
# Applying SLA To Each Host
##################################
$HostsToMigrateCounter = 0
ForEach ($HostToMigrate in $HostsToMigrate)
{
$HostID = $WindowsHosts | Where-Object {$_.name -eq $HostToMigrate} | Select -ExpandProperty id
# Host output
$HostsToMigrateCounter ++
"$HostsToMigrateCounter/$HostsToMigrateCount AssigningSLAToHost: $HostToMigrate"
# Checking if both exist, otherwise throwing error and not continuing as no point
IF (($SQLSLADomainID -eq $null) -or ($HostID -eq $null))
{
# Output to host
"
HostID or SLADomainID not found from the names specified, check the variables and try again
"
}
ELSE
{
##################################
# Assigning SLA Domain to SQL Host
##################################
# Creating URL
$AssignSQLSLAURL = $BaseURL+"mssql/sla_domain/assign"
# Creating JSON body
$AssignSQLSLAJSON = 
"{
  ""logBackupFrequencyInSeconds"": $SQLLogBackupFrequencyMins,
  ""logRetentionHours"": $SQLLogBackupRetentionHours,
  ""copyOnly"": $SQLCopyOnlyBackup,
  ""configuredSlaDomainId"": ""$SQLSLADomainID"",
  ""ids"": [""$HostID""]
}"
# Posting to the API
Try 
{
Invoke-RestMethod -Method Post -Uri $AssignSQLSLAURL -Body $AssignSQLSLAJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type | Out-Null
$SLADomainAssigned = $TRUE
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
$SLADomainAssigned = $FALSE
}
# Output to host
"$HostsToMigrateCounter/$HostsToMigrateCount AssignedSLA: $SLADomainAssigned"
# End of bypass for no hostid or sladomainid below
}
# End of bypass for no hostid or sladomainid above
#
# End of for each host below
}
# End of for each host above
##################################
# End of script
##################################