﻿#####################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
#####################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
##############################################
# Description:
# This script uses a CSV list of instances to query every DB size, files, backups, then summarizes the data into useful tables which can be used for reporting, email, alerts etc.
##############################################
# Requirements:
# - Internet access on the computer running this script to download the SQL module if not installed
# - At least 1 SQL instance configured in the CSV with credentails to query all the DBs on the instance
# - CSV with the following column; SQLInstance
# - Example line 1: SQL16-VM01.lab.local
# - Example line 2: SQL16-VM01.lab.local\SQLINSTANCE01
# - The instance must include the hostname of the server, otherwise the script won't be able to connect to it! In the first example I'm using MSSQLSERVER so don't need to name the instance 
# - A user (SA or SQL) with permissions to connect to and query every instance configured in the CSV
##############################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever. 
# Including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss arising out of the use of,
# or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
##############################################
# Configure the variables below
##############################################
# CSV list for SQL instances to query
$SQLInstanceCSV = "C:\MSSQLDBReportingv1\MSSQLInstances.csv"
# Configure Backup SLAs, if backup is not within window marked as not meeting SLA
$FullBackupSLAInhours = 24
$DiffBackupSLAInhours = 24
$LogBackupSLAInMinutes = 120
# Exlude Master,Model,MSDB, and TempDB from reporting
$ExcludeSysDBs = $TRUE
#####################################################################
# Nothing to change below this line, commented throughout to explain
#####################################################################
###############################################
# Creating/importing SQL credentials
###############################################
# Setting credential file
$SQLCredentialsFile = ".\SQLCredentials.xml"
# Testing if file exists
$SQLCredentialsFileTest =  Test-Path $SQLCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($SQLCredentialsFileTest -eq $False)
{
$SQLCredentials = Get-Credential -Message "Enter SQL login credentials"
$SQLCredentials | EXPORT-CLIXML $SQLCredentialsFile -Force
}
# Importing credentials
$SQLCredentials = IMPORT-CLIXML $SQLCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$SQLUsername = $SQLCredentials.UserName
$SQLPassword = $SQLCredentials.GetNetworkCredential().Password
##############################################
# Importing the CSV
##############################################
$SQLInstanceCSVTest = Test-Path $SQLInstanceCSV
# IF SQL file doesn't exist, skipping script and giving error
IF ($SQLInstanceCSVTest -eq $TRUE)
{
$SQLInstances = Import-CSV $SQLInstanceCSV
##############################################
# Getting current date and substracting time for SLA calcs on backups
##############################################
$CurrentDate = Get-Date
$FullBackupSLADate = $CurrentDate.AddHours(-$FullBackupSLAInhours)
$DiffBackupSLADate = $CurrentDate.AddHours(-$DiffBackupSLAInhours)
$LogBackupSLADate = $CurrentDate.AddMinutes(-$LogBackupSLAInMinutes)
##############################################
# Checking to see if the SqlServer module is already installed, if not installing it
##############################################
$SQLModuleCheck = Get-Module -ListAvailable SqlServer
if ($SQLModuleCheck -eq $null)
{
write-host "SqlServer Module Not Found - Installing"
# Not installed, trusting PS Gallery to remove prompt on install
Set-PSRepository -Name PSGallery -InstallationPolicy Trusted
# Installing module
Install-Module -Name SqlServer –Scope AllUsers -Confirm:$false -AllowClobber
}
##############################################
# Importing the SqlServer module
##############################################
Import-Module SqlServer
##############################################
# Creating table object to store the results
##############################################
$SQLDBSizeTable = @()
$SQLDBFileTable = @()
$SQLDBBackupTable = @()
$SQLInstanceTable = @()
$SQLDBSummaryTable = @()
##############################################
# Performing for each SQL Instance specified
##############################################
ForEach ($SQLInstance in $SQLInstances)
{
$SQLInstance = $SQLInstance.SQLInstance
##############################################
# Getting DB file info
##############################################
# Building query
$SQLDBFileQuery = "SELECT
    db.name AS DBName,
    type_desc AS FileType,
    Physical_Name AS Location
    FROM
    sys.master_files mf
    INNER JOIN 
    sys.databases db ON db.database_id = mf.database_id
    ORDER BY DBName"
# Running query
Try 
{
$SQLDBFileResult = Invoke-SQLcmd -Query $SQLDBFileQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
########################
# Excluding System DBs if Enabled
########################
IF ($ExcludeSysDBs -eq $TRUE)
{
$SQLDBFileResultFiltered = $SQLDBFileResult | Where-Object {(($_.DBName -ne "master") -and ($_.DBName -ne "model") -and ($_.DBName -ne "msdb") -and ($_.DBName -ne "tempdb"))}
}
########################
# For Each SQL DB adding SQLDBFileResult to table
########################
ForEach ($SQLDBFile in $SQLDBFileResultFiltered)
{
# Setting variables
$SQLDBFileDBName = $SQLDBFile.DBName
$SQLDBFileFileType = $SQLDBFile.FileType
$SQLDBFileLocation = $SQLDBFile.Location
# Adding result to table
$SQLDBFileLine = new-object PSObject
$SQLDBFileLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLInstance"
$SQLDBFileLine | Add-Member -MemberType NoteProperty -Name "DBName" -Value "$SQLDBFileDBName"
$SQLDBFileLine | Add-Member -MemberType NoteProperty -Name "FileType" -Value "$SQLDBFileFileType"
$SQLDBFileLine | Add-Member -MemberType NoteProperty -Name "FileLocation" -Value "$SQLDBFileLocation"
$SQLDBFileTable += $SQLDBFileLine
}
##############################################
# Getting DB sizes
##############################################
# Building query
$SQLDBSizeQuery = "SELECT
    DB_NAME(db.database_id) DBName,
    (CAST(mfrows.RowSize AS FLOAT)*8)/1024 DBSizeMB,
    (CAST(mflog.LogSize AS FLOAT)*8)/1024 LogSizeMB
FROM sys.databases db
    LEFT JOIN (SELECT database_id, SUM(size) RowSize FROM sys.master_files WHERE type = 0 GROUP BY database_id, type) mfrows ON mfrows.database_id = db.database_id
    LEFT JOIN (SELECT database_id, SUM(size) LogSize FROM sys.master_files WHERE type = 1 GROUP BY database_id, type) mflog ON mflog.database_id = db.database_id"
# Running query
Try 
{
$SQLDBSizeResult = Invoke-SQLcmd -Query $SQLDBSizeQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
########################
# Excluding System DBs if Enabled
########################
IF ($ExcludeSysDBs -eq $TRUE)
{
$SQLDBSizeResultFiltered = $SQLDBSizeResult | Where-Object {(($_.DBName -ne "master") -and ($_.DBName -ne "model") -and ($_.DBName -ne "msdb") -and ($_.DBName -ne "tempdb"))}
}
########################
# Counting DBs per Instance
########################
$SQLInstanceDBCount = $SQLDBSizeResult.Count
########################
# For Each SQL DB adding SQLDBSizeResult to table
########################
ForEach ($SQLDB in $SQLDBSizeResultFiltered)
{
# Settng variables
$SQLDBName = $SQLDB.DBName
$SQLDBSizeMB = $SQLDB.DBSizeMB
$SQLDBLogSizeMB = $SQLDB.LogSizeMB
# Converting MB to GB
$SQLDBSizeGB = $SQLDBSizeMB / 1000 
$SQLDBSizeGB = [Math]::Round($SQLDBSizeGB,2)
$SQLDBLogSizeGB = $SQLDBLogSizeMB / 1000
$SQLDBLogSizeGB = [Math]::Round($SQLDBLogSizeGB,2)
# Getting file counts
$SQLDBFileCount = $SQLDBFileTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.DBName -eq $SQLDBName) -and ($_.FileType -eq "ROWS"))} | Measure | Select -ExpandProperty Count
$SQLLogFileCount = $SQLDBFileTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.DBName -eq $SQLDBName) -and ($_.FileType -eq "LOG"))} | Measure | Select -ExpandProperty Count
# Adding result to table
$SQLDBSizeTableLine = New-Object PSObject
$SQLDBSizeTableLine | Add-Member -MemberType NoteProperty -Name "DBName" -Value "$SQLDBName"
$SQLDBSizeTableLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLInstance"
$SQLDBSizeTableLine | Add-Member -MemberType NoteProperty -Name "DBSizeGB" -Value "$SQLDBSizeGB"
$SQLDBSizeTableLine | Add-Member -MemberType NoteProperty -Name "LogSizeGB" -Value "$SQLDBLogSizeGB"
$SQLDBSizeTableLine | Add-Member -MemberType NoteProperty -Name "DBFileCount" -Value "$SQLDBFileCount"
$SQLDBSizeTableLine | Add-Member -MemberType NoteProperty -Name "LogFileCount" -Value "$SQLLogFileCount"
$SQLDBSizeTable += $SQLDBSizeTableLine
}
##############################################
# Getting DB backup info
##############################################
$SQLDBBackupQuery = "SELECT DISTINCT a.Name AS DBName 
        , CONVERT(SYSNAME, DATABASEPROPERTYEX(a.name, 'Recovery')) RecoveryModel 
        , COALESCE(( SELECT   CONVERT(VARCHAR(20), MAX(backup_finish_date), 120)
                   FROM     msdb.dbo.backupset
                   WHERE    database_name = a.name
                            AND type = 'd'
                            AND is_copy_only = '0' ), 'No Full') AS 'Full' 
        , COALESCE(( SELECT   CONVERT(VARCHAR(20), MAX(backup_finish_date), 120)
                   FROM     msdb.dbo.backupset
                   WHERE    database_name = a.name
                            AND type = 'i'
                            AND is_copy_only = '0'), 'No Diff') AS 'Diff' 
        , COALESCE(( SELECT   CONVERT(VARCHAR(20), MAX(backup_finish_date), 120)
                   FROM     msdb.dbo.backupset
                   WHERE    database_name = a.name
                            AND type = 'l'), 'No Log') AS 'LastLog1' 
        , COALESCE(( SELECT   CONVERT(VARCHAR(20), backup_finish_date, 120)
                   FROM     ( SELECT    ROW_NUMBER() OVER ( ORDER BY backup_finish_date DESC ) AS 'rownum' ,
                                        backup_finish_date
                              FROM      msdb.dbo.backupset
                              WHERE     database_name = a.name
                                        AND type = 'l'
                            ) withrownum
                   WHERE    rownum = 2), 'No Log') AS 'LastLog2'
FROM sys.databases a 
LEFT OUTER JOIN msdb.dbo.backupset b 
ON b.database_name = a.name
WHERE   a.name <> 'tempdb' 
        AND a.state_desc = 'online'
GROUP BY a.Name ,
        a.compatibility_level
ORDER BY a.name"
# Running query
Try 
{
$SQLDBBackupResult = Invoke-SQLcmd -Query $SQLDBBackupQuery -ServerInstance $SQLInstance -Username $SQLUsername -Password $SQLPassword
}
Catch 
{
Write-Host $_.Exception.ToString()
$error[0] | Format-List -Force
}
########################
# Excluding System DBs if Enabled
########################
IF ($ExcludeSysDBs -eq $TRUE)
{
$SQLDBBackupResultFiltered = $SQLDBBackupResult | Where-Object {(($_.DBName -ne "master") -and ($_.DBName -ne "model") -and ($_.DBName -ne "msdb") -and ($_.DBName -ne "tempdb"))}
}
########################
# For Each SQL DB adding SQLDBBackupResult to table
########################
ForEach ($SQLDBBackup in $SQLDBBackupResultFiltered)
{
# Setting variables
$SQLDBBackupName = $SQLDBBackup.DBName
$SQLDBBackupRecoveryModel = $SQLDBBackup.RecoveryModel
#########################
# Calculating Last full backup and SLA compliance
#########################
$SQLDBBackupLastFull = $SQLDBBackup.Full
IF ($SQLDBBackupLastFull -ne "No Full")
{
$SQLDBBackupLastFull = [DateTime]$SQLDBBackupLastFull
IF ($SQLDBBackupLastFull -gt $FullBackupSLADate)
{
$FullBackupSLACompliance = $TRUE
}
ELSE
{
$FullBackupSLACompliance = $FALSE
}
}
ELSE
{
$FullBackupSLACompliance = "N/A"
}
#########################
# Calculating Last diff backup and SLA compliance
#########################
$SQLDBBackupLastDiff = $SQLDBBackup.Diff
IF ($SQLDBBackupLastDiff -ne "No Diff")
{
$SQLDBBackupLastDiff = [DateTime]$SQLDBBackupLastDiff
IF ($SQLDBBackupLastDiff -gt $DiffBackupSLAInhours)
{
$DiffBackupSLACompliance = $TRUE
}
ELSE
{
$DiffBackupSLACompliance = $FALSE
}
}
ELSE
{
$DiffBackupSLACompliance = "N/A"
}
#########################
# Calculating Last Log backup and SLA compliance
#########################
# Last log 1
$SQLDBBackupLastLog1 = $SQLDBBackup.LastLog1
IF ($SQLDBBackupLastLog1 -ne "No Log")
{
$SQLDBBackupLastLog1 = [DateTime]$SQLDBBackupLastLog1
IF ($SQLDBBackupLastLog1 -gt $LogBackupSLADate)
{
$LogBackupSLACompliance = $TRUE
}
ELSE
{
$LogBackupSLACompliance = $FALSE
}
}
ELSE
{
$LogBackupSLACompliance = "N/A"
}
#########################
# Calculating 2nd Last Log backup and difference
#########################
$SQLDBBackupLastLog2 = $SQLDBBackup.LastLog2
IF ($SQLDBBackupLastLog2 -ne "No Log")
{$SQLDBBackupLastLog2 = [DateTime]$SQLDBBackupLastLog2}
# If 2 log backups exist calculating diff
IF (($SQLDBBackupLastLog1 -ne "No Log") -AND ($SQLDBBackupLastLog2 -ne "No Log"))
{
$LastLogDiff = New-Timespan -Start $SQLDBBackupLastLog2 -End $SQLDBBackupLastLog1 | Select -ExpandProperty TotalMinutes
$LastLogDiff = [Math]::Round($LastLogDiff,0)
}
ELSE
{
$LastLogDiff = $null
}
#########################
# Adding results to table
#########################
$SQLDBBackupTableLine = New-Object PSObject
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "DBName" -Value "$SQLDBBackupName"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLInstance"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "RecoveryModel" -Value "$SQLDBBackupRecoveryModel"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "MeetingFullSLA" -Value "$FullBackupSLACompliance"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "LastFull" -Value "$SQLDBBackupLastFull"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "MeetingDiffSLA" -Value "$DiffBackupSLACompliance"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "LastDiff" -Value "$SQLDBBackupLastDiff"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "MeetingLogSLA" -Value "$LogBackupSLACompliance"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "LastLog1" -Value "$SQLDBBackupLastLog1"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "LastLog2" -Value "$SQLDBBackupLastLog2"
$SQLDBBackupTableLine | Add-Member -MemberType NoteProperty -Name "LogDiff" -Value "$LastLogDiff"
$SQLDBBackupTable += $SQLDBBackupTableLine
# End of for each DB below
}
# End of for each DB above
##############################################
# Calculating totals for each instance
##############################################
# Counting DBs
$SQLInstanceTotalDBs = $SQLDBSizeTable | Where-Object {$_.Instance -eq $SQLInstance} | Measure | Select -ExpandProperty Count
# DBs in recovery modes
$SQLInstanceTotalSimple = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.RecoveryModel -eq "SIMPLE"))} | Measure | Select -ExpandProperty Count
$SQLInstanceTotalFull = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.RecoveryModel -eq "FULL"))} | Measure | Select -ExpandProperty Count
# Total DBs files
$SQLInstanceTotalDBFiles = $SQLDBFileTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.FileType -eq "ROWS"))} | Measure | Select -ExpandProperty Count
$SQLInstanceTotalLogFiles = $SQLDBFileTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.FileType -eq "LOG"))} | Measure | Select -ExpandProperty Count
# Getting total DBs and Logs size on disk
$SQLInstanceTotalDBSizeGB = $SQLDBSizeTable | Where-Object {$_.Instance -eq $SQLInstance} | Select -ExpandProperty DBSizeGB | Measure -Sum | Select -ExpandProperty Sum
$SQLInstanceTotalLogSizeGB = $SQLDBSizeTable | Where-Object {$_.Instance -eq $SQLInstance} | Select -ExpandProperty LogSizeGB | Measure -Sum | Select -ExpandProperty Sum
# DBs meeting Full SLA
$SQLInstanceTotalDBFullMeetingSLA = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.MeetingFullSLA -eq $TRUE))} | Measure | Select -ExpandProperty Count
$SQLInstanceTotalDBFullNotMeetingSLA = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.MeetingFullSLA -eq $FALSE))} | Measure | Select -ExpandProperty Count
# DBS meeting Diff SLA
$SQLInstanceTotalDBDiffMeetingSLA = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.MeetingDiffSLA -eq $TRUE))} | Measure | Select -ExpandProperty Count
$SQLInstanceTotalDBDiffNotMeetingSLA = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.MeetingDiffSLA -eq $FALSE))} | Measure | Select -ExpandProperty Count
# DBS meeting Log SLA
$SQLInstanceTotalDBLogMeetingSLA = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.MeetingLogSLA -eq $TRUE))} | Measure | Select -ExpandProperty Count
$SQLInstanceTotalDBLogNotMeetingSLA = $SQLDBBackupTable | Where-Object {(($_.Instance -eq $SQLInstance) -and ($_.MeetingLogSLA -eq $FALSE))} | Measure | Select -ExpandProperty Count
# Adding results to table
$SQLInstanceTableLine = New-Object PSObject
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "Instance" -Value "$SQLInstance"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLInstanceTotalDBs"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "Simple" -Value "$SQLInstanceTotalSimple"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "Full" -Value "$SQLInstanceTotalFull"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "DBFiles" -Value "$SQLInstanceTotalDBFiles"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "LogFiles" -Value "$SQLInstanceTotalLogFiles"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "TotalDBSizeGB" -Value "$SQLInstanceTotalDBSizeGB"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "TotalLogSizeGB" -Value "$SQLInstanceTotalLogSizeGB"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "MeetingFullSLA" -Value "$SQLInstanceTotalDBFullMeetingSLA"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "NotMeetingFullSLA" -Value "$SQLInstanceTotalDBFullNotMeetingSLA"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "MeetingDiffSLA" -Value "$SQLInstanceTotalDBDiffMeetingSLA"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "NotMeetingDiffSLA" -Value "$SQLInstanceTotalDBDiffNotMeetingSLA"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "MeetingLogSLA" -Value "$SQLInstanceTotalDBLogMeetingSLA"
$SQLInstanceTableLine | Add-Member -MemberType NoteProperty -Name "NotMeetingLogSLA" -Value "$SQLInstanceTotalDBLogNotMeetingSLA"
$SQLInstanceTable += $SQLInstanceTableLine
##############################################
# End of for each SQL instance below
##############################################
}
# End of for each SQL instance above
##############################################
# Calculating totals for the environment
##############################################
# Counting Instances
$SQLInstanceCount = $SQLInstances.Count
# Counting DBs
$SQLTotalDBs = $SQLDBSizeTable | Measure | Select -ExpandProperty Count
# DBs in recovery modes
$SQLTotalSimple = $SQLDBBackupTable | Where-Object {$_.RecoveryModel -eq "SIMPLE"} | Measure | Select -ExpandProperty Count
$SQLTotalFull = $SQLDBBackupTable | Where-Object {$_.RecoveryModel -eq "FULL"} | Measure | Select -ExpandProperty Count
# Total DBs files
$SQLTotalDBFiles = $SQLDBFileTable | Where-Object {$_.FileType -eq "ROWS"} | Measure | Select -ExpandProperty Count
$SQLTotalLogFiles = $SQLDBFileTable | Where-Object {$_.FileType -eq "LOG"} | Measure | Select -ExpandProperty Count
# Getting total DBs and Logs size on disk
$SQLTotalDBSizeGB = $SQLDBSizeTable.DBSizeGB | Measure -Sum | Select -ExpandProperty Sum
$SQLTotalLogSizeGB = $SQLDBSizeTable.LogSizeGB | Measure -Sum | Select -ExpandProperty Sum
# DBs meeting Full SLA
$SQLTotalDBFullMeetingSLA = $SQLDBBackupTable | Where-Object {$_.MeetingFullSLA -eq $TRUE} | Measure | Select -ExpandProperty Count
$SQLTotalDBFullNotMeetingSLA = $SQLDBBackupTable | Where-Object {$_.MeetingFullSLA -eq $FALSE} | Measure | Select -ExpandProperty Count
# DBS meeting Diff SLA
$SQLTotalDBDiffMeetingSLA = $SQLDBBackupTable | Where-Object {$_.MeetingDiffSLA -eq $TRUE} | Measure | Select -ExpandProperty Count
$SQLTotalDBDiffNotMeetingSLA = $SQLDBBackupTable | Where-Object {$_.MeetingDiffSLA -eq $FALSE} | Measure | Select -ExpandProperty Count
# DBS meeting Log SLA
$SQLTotalDBLogMeetingSLA = $SQLDBBackupTable | Where-Object {$_.MeetingLogSLA -eq $TRUE} | Measure | Select -ExpandProperty Count
$SQLTotalDBLogNotMeetingSLA = $SQLDBBackupTable | Where-Object {$_.MeetingLogSLA -eq $FALSE} | Measure | Select -ExpandProperty Count
# Adding results to table
$SQLDBSummaryTableLine = New-Object PSObject
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "Instances" -Value "$SQLInstanceCount"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "DBs" -Value "$SQLTotalDBs"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "Simple" -Value "$SQLTotalSimple"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "Full" -Value "$SQLTotalFull"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "DBFiles" -Value "$SQLTotalDBFiles"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "LogFiles" -Value "$SQLTotalLogFiles"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "TotalDBSizeGB" -Value "$SQLTotalDBSizeGB"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "TotalLogSizeGB" -Value "$SQLTotalLogSizeGB"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "MeetingFullSLA" -Value "$SQLTotalDBFullMeetingSLA"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "NotMeetingFullSLA" -Value "$SQLTotalDBFullNotMeetingSLA"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "MeetingDiffSLA" -Value "$SQLTotalDBDiffMeetingSLA"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "NotMeetingDiffSLA" -Value "$SQLTotalDBDiffNotMeetingSLA"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "MeetingLogSLA" -Value "$SQLTotalDBLogMeetingSLA"
$SQLDBSummaryTableLine | Add-Member -MemberType NoteProperty -Name "NotMeetingLogSLA" -Value "$SQLTotalDBLogNotMeetingSLA"
$SQLDBSummaryTable += $SQLDBSummaryTableLine
##############################################
# Showing results, you can send this data by email or take other actions
##############################################
$SQLDBSizeTable | Sort-Object Instance,DBName |  Out-GridView -Title "SQL DB Sizes"
$SQLDBFileTable | Sort-Object Instance,DBName |  Out-GridView -Title "SQL DB Files"
$SQLDBBackupTable | Sort-Object Instance,DBName |  Out-GridView -Title "SQL DB Backups"
$SQLInstanceTable | Sort-Object Instance |  Out-GridView -Title "SQL Instance Summary"
$SQLDBSummaryTable | Out-GridView -Title "SQL Environment Summary"
##############################################
# End bypass for CSV not found below
##############################################
}
ELSE
{
"CSV Not Found: $SQLInstanceCSV"
"Skipped running the script as no CSV list of instances found"
sleep 3
}
##############################################
# End of script
##############################################