########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script EXPORTS all of the VMNIC settings for customization before running a separate IMPORT script
################################################
# Requirements:
# - PowerShell 5.0, will not work on 6.0, and not tested on lower PowerShell version. If you need a 6.0 version go back to the blog post where you downloaded this!
# - Zerto 6.0+, might work on lower versions after 5.0 U3 but not tested
# - Change $ZertoServer to be the protected site ZVM
# - You have to run this script first to create a CSV export of the current NIC settings for the import script. 
# - Once run, edit the CSV with your desired settings
# - Make sure you have VPGs configured otherwise there is nothing to export!
# - THIS SCRIPT WILL ONLY WORK FOR VMWARE TO VMWARE replication, Zerto does not have an API call for editing VPGs to and from anything else (still)
# - VMware Tools installed in all protected VMs
# - In Windows, run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted",
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any Zerto support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author and Zerto further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall Zerto, its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author or Zerto has been advised of the possibility of such damages.
################################################
# Configure the variables below
################################################
$ExportDataDir = "C:\ZertoBulkReIPv6\"
$ZertoServer = "192.168.1.9"
# Prompting for username and password to authenicate with ZVM
$Credentials = Get-Credential -Credential $null
$ZertoUser = $Credentials.UserName
$Credentials.Password | ConvertFrom-SecureString
$ZertoPassword = $Credentials.GetNetworkCredential().password
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Setting certificate exception to prevent authentication issues to the ZVM
################################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
################################################
# Building Zerto API string and invoking API
################################################
$BaseURL = "https://" + $ZertoServer + ":9669/v1/"
$ZertoSessionURL = $BaseURL + "session/add"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($ZertoUser+":"+$ZertoPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$ZertoSessionResponse = Invoke-WebRequest -Uri $ZertoSessionURL -Headers $Header -Method POST -ContentType $Type 
$ZertoSessionSuccess = $TRUE
}
Catch 
{
$ZertoSessionSuccess = $FALSE
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$ZertoSession = $ZertoSessionResponse.headers.get_item("x-zerto-session")
$ZertoSessionHeader = @{"Accept"="application/json"
"x-zerto-session"="$ZertoSession"}
################################################
# IF not authenticated with Zerto no point continuing
################################################
IF ($ZertoSessionSuccess -eq $TRUE)
{
################################################
# Creating Arrays for populating ZVM info from the API
################################################
$VPGArray = @()
$VMArray = @()
$VMVolumeArray = @()
$VMNICArray = @()
################################################
# Getting list of VPGs
################################################
$VPGListUrl = $BaseURL+"vpgs"
Try 
{
$VPGList = Invoke-RestMethod -Uri $VPGListUrl -TimeoutSec 100 -Headers $ZertoSessionHeader -ContentType $Type 
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
################################################
# Getting list of VMs 
################################################
$VMListUrl = $BaseURL+"vms"
Try 
{
$VMList = Invoke-RestMethod -Uri $VMListUrl -TimeoutSec 100 -Headers $ZertoSessionHeader -ContentType $Type 
}
Catch 
{
$_.Exception.ToString()
$Error[0] | Format-List -Force
}
################################################
# Starting for each VPG action of collecting ZVM VPG data
################################################
ForEach ($VPG in $VPGList)
{
$VPGID = $VPG.vpgidentifier
$VPGOrganization = $VPG.OrganizationName
$VPGVMCount = $VPG.VmsCount
$VPGJSON = 
"{
""VpgIdentifier"":""$VPGID""
}"
################################################
# Posting the VPG JSON Request to the API to get a settings ID (like clicking edit on a VPG in the GUI)
################################################
# URL to Edit VPG settings
$EditVPGURL = $BaseURL+"vpgSettings"
# POST
Try 
{
$VPGSettingsID = Invoke-RestMethod -Method POST -Uri $EditVPGURL -Body $VPGJSON -ContentType $Type -Headers $ZertoSessionHeader 
$ValidVPGSettingsID = $True
}
Catch 
{
$ValidVPGSettingsID = $False
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Getting VPG settings from API
################################################
# Skipping if unable to obtain valid VPG setting identifier
IF ($ValidVPGSettingsID -eq $True)
{
# Getting VPG settings
$VPGSettingsURL = $BaseURL+"vpgSettings/"+$VPGSettingsID
Try 
{
$VPGSettings = Invoke-RestMethod -Method GET -Uri $VPGSettingsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Getting VPG Settings
$VPGName = $VPGSettings.Basic.Name
# Getting VM IDs in VPG
$VPGVMIDs = $VPGSettings.VMs.VmIdentifier
################################################
# Getting Recovery Site settings from API
################################################
# Setting recovery site ID (needed for network settings)
$VPGRecoverySiteID = $VPGSettings.Basic.RecoverySiteIdentifier
# Getting network info 
$VPGPortGroupsURL = $baseURL+"virtualizationsites/$VPGRecoverySiteID/networks"
Try 
{
$VPGPortGroups = Invoke-RestMethod -Method GET -Uri $VPGPortGroupsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Starting for each VM ID action for collecting ZVM VM data
################################################
ForEach ($VPGVMID in $VPGVMIDs)
{
$VMID = $VPGVMID
# Getting the VM name from the VM list
$VMName = $VMList | Where-Object {$_.VMIdentifier -eq $VMID} | Select -ExpandProperty VMName
# Get VMs settings
$VMSettingsURL = $baseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VMID
Try 
{
$VMSettings = Invoke-RestMethod -Method GET -Uri $VMSettingsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Get VM NIC IDs for the current VM
################################################
$VMNICsURL = $baseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VMID+"/nics"
Try 
{
$VMNICs = Invoke-RestMethod -Method GET -Uri $VMNICsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
$VMNICIDs = $VMNICs | Select -ExpandProperty NicIdentifier
################################################
# Starting for each VM NIC ID action for collecting ZVM VM NIC data
################################################
ForEach ($VMNICID in $VMNICIDs)
{
# Getting VM NIC settings
$VMNICSettingsURL = $baseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VMID+"/nics/"+$VMNICID
Try 
{
$VMNICSettings = Invoke-RestMethod -Method GET -Uri $VMNICSettingsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Building arrays of data from API
$VMSettingNICIDArray1 = $VMNICSettings.Failover.Hypervisor
$VMSettingNICIDArray2 = $VMNICSettings.Failover.Hypervisor.IpConfig
$VMSettingNICIDArray3 = $VMNICSettings.FailoverTest.Hypervisor
$VMSettingNICIDArray4 = $VMNICSettings.FailoverTest.Hypervisor.IpConfig
# Setting failover values
$VMNICFailoverDNSSuffix = $VMSettingNICIDArray1.DnsSuffix
$VMNICFailoverNetworkID = $VMSettingNICIDArray1.NetworkIdentifier
$VMNICFailoverShouldReplaceMacAddress = $VMSettingNICIDArray1.ShouldReplaceMacAddress
$VMNICFailoverGateway = $VMSettingNICIDArray2.Gateway
$VMNICFailoverDHCP = $VMSettingNICIDArray2.IsDhcp
$VMNICFailoverPrimaryDns = $VMSettingNICIDArray2.PrimaryDns
$VMNICFailoverSecondaryDns = $VMSettingNICIDArray2.SecondaryDns
$VMNICFailoverStaticIp = $VMSettingNICIDArray2.StaticIp
$VMNICFailoverSubnetMask = $VMSettingNICIDArray2.SubnetMask
# Nulling blank content
IF ($VMNICFailoverDNSSuffix.nil -eq $true){$VMNICFailoverDNSSuffix = $null}
IF ($VMNICFailoverGateway.nil -eq $true){$VMNICFailoverGateway = $null}
IF ($VMNICFailoverPrimaryDns.nil -eq $true){$VMNICFailoverPrimaryDns = $null}
IF ($VMNICFailoverSecondaryDns.nil -eq $true){$VMNICFailoverSecondaryDns = $null}
IF ($VMNICFailoverStaticIp.nil -eq $true){$VMNICFailoverStaticIp = $null}
IF ($VMNICFailoverSubnetMask.nil -eq $true){$VMNICFailoverSubnetMask = $null}
# Setting failover test values
$VMNICFailoverTestDNSSuffix = $VMSettingNICIDArray3.DnsSuffix
$VMNICFailoverTestNetworkID = $VMSettingNICIDArray3.NetworkIdentifier
$VMNICFailoverTestShouldReplaceMacAddress = $VMSettingNICIDArray3.ShouldReplaceMacAddress
$VMNICFailoverTestGateway = $VMSettingNICIDArray4.Gateway
$VMNICFailoverTestDHCP = $VMSettingNICIDArray4.IsDhcp
$VMNICFailoverTestPrimaryDns = $VMSettingNICIDArray4.PrimaryDns
$VMNICFailoverTestSecondaryDns = $VMSettingNICIDArray4.SecondaryDns
$VMNICFailoverTestStaticIp = $VMSettingNICIDArray4.StaticIp
$VMNICFailoverTestSubnetMask = $VMSettingNICIDArray4.SubnetMask
# Nulling blank content
IF ($VMNICFailoverTestDNSSuffix.nil -eq $true){$VMNICFailoverTestDNSSuffix = $null}
IF ($VMNICFailoverTestGateway.nil -eq $true){$VMNICFailoverTestGateway = $null}
IF ($VMNICFailoverTestPrimaryDns.nil -eq $true){$VMNICFailoverTestPrimaryDns = $null}
IF ($VMNICFailoverTestSecondaryDns.nil -eq $true){$VMNICFailoverTestSecondaryDns = $null}
IF ($VMNICFailoverTestStaticIp.nil -eq $true){$VMNICFailoverTestStaticIp = $null}
IF ($VMNICFailoverTestSubnetMask.nil -eq $true){$VMNICFailoverTestSubnetMask = $null}
# Mapping Network IDs to Names
$VMNICFailoverNetworkName = $VPGPortGroups | Where-Object {$_.NetworkIdentifier -eq $VMNICFailoverNetworkID}  | Select -ExpandProperty VirtualizationNetworkName 
$VMNICFailoverTestNetworkName = $VPGPortGroups | Where-Object {$_.NetworkIdentifier -eq $VMNICFailoverTestNetworkID}  | Select -ExpandProperty VirtualizationNetworkName 
################################################
# Adding all VM NIC setting info to $VMNICArray
################################################
$VMNICArrayLine = new-object PSObject
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "VPGName" -Value $VPGName
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "VPGID" -Value $VPGID
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "VMName" -Value $VMName
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VMID
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "NICID" -Value $VMNICID
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverNetworkName" -Value $VMNICFailoverNetworkName
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverDHCP" -Value $VMNICFailoverDHCP
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverStaticIp" -Value $VMNICFailoverStaticIp
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverSubnetMask" -Value $VMNICFailoverSubnetMask
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverGateway" -Value $VMNICFailoverGateway
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverPrimaryDns" -Value $VMNICFailoverPrimaryDns
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverSecondaryDns" -Value $VMNICFailoverSecondaryDns
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverDNSSuffix" -Value $VMNICFailoverDNSSuffix
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverShouldReplaceMacAddress" -Value $VMNICFailoverShouldReplaceMacAddress
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestNetworkName" -Value $VMNICFailoverTestNetworkName
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestDHCP" -Value $VMNICFailoverTestDHCP
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestStaticIp" -Value $VMNICFailoverTestStaticIp
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestSubnetMask" -Value $VMNICFailoverTestSubnetMask
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestGateway" -Value $VMNICFailoverTestGateway
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestPrimaryDns" -Value $VMNICFailoverTestPrimaryDns
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestSecondaryDns" -Value $VMNICFailoverTestSecondaryDns
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestDNSSuffix" -Value $VMNICFailoverTestDNSSuffix
$VMNICArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverTestShouldReplaceMacAddress" -Value $VMNICFailoverTestShouldReplaceMacAddress
$VMNICArray += $VMNICArrayLine
# End of per VM NIC actions below
}
# End of per VM NIC actions above
#
# End of per VM actions below
}
# End of per VM actions above
################################################
# Deleting VPG edit settings ID (same as closing the edit screen on a VPG in the ZVM without making any changes)
################################################
Try 
{
Invoke-RestMethod -Method Delete -Uri $VPGSettingsURL -TimeoutSec 100 -ContentType $Type -Headers $ZertoSessionHeader
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
#
# End of check for valid VPG settings ID below
}
# End of check for valid VPG settings ID above
#
# End of per VPG actions below
}
# End of per VPG actions above
#
################################################
# Exporting VM Nic Settings
################################################
$CSVExportFile = $ExportDataDir + "ZertoVMNICS.csv" 
$VMNICArray | Sort-Object VPGName,VMName,NICID | Export-CSV $CSVExportFile -NoTypeInformation
################################################
# IF not authenticated with Zerto no point continuing
################################################
}
ELSE
{
"Failed to login to Zerto API, check server and credentials"
}
################################################
# End of script
################################################