########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script IMPORTS all of the VMNIC settings for customization, this should be run using the CSV created from the EXPORT script
################################################
# Requirements:
# - PowerShell 5.0, will not work on 6.0, and not tested on lower PowerShell version. If you need a 6.0 version go back to the blog post where you downloaded this!
# - Zerto 6.0+, might work on lower versions after 5.0 U3 but not tested
# - Change $ZertoServer to be the protected site ZVM
# - You have to run the -Export.ps1 script first to create a CSV export of the current NIC settings. Edit the CSV, ensure the $CSVExportFile is correct and run the script.
# - THIS SCRIPT WILL ONLY WORK FOR VMWARE TO VMWARE replication, Zerto does not have an API call for editing VPGs to and from anything else (still)
# - VMware Tools installed in all protected VMs
# - In Windows, run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted",
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any Zerto support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author and Zerto further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall Zerto, its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author or Zerto has been advised of the possibility of such damages.
################################################
# Configure the variables below
################################################
$CSVExportFile = "C:\ZertoBulkReIPv6\ZertoVMNICS.csv" 
$ZertoServer = "192.168.1.9"
# Prompting for username and password to authenicate with ZVM
$Credentials = Get-Credential -Credential $null
$ZertoUser = $Credentials.UserName
$Credentials.Password | ConvertFrom-SecureString
$ZertoPassword = $Credentials.GetNetworkCredential().password
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Importing CSV and building list of VPGs
################################################
$CSVImport = Import-Csv $CSVExportFile
$VPGsToConfigure = $CSVImport | select -ExpandProperty VPGName -Unique
################################################
# Setting certificate exception to prevent authentication issues to the ZVM
################################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
################################################
# Building Zerto API string and invoking API
################################################
$BaseURL = "https://" + $ZertoServer + ":9669/v1/"
$ZertoSessionURL = $BaseURL + "session/add"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($ZertoUser+":"+$ZertoPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$ZertoSessionResponse = Invoke-WebRequest -Uri $ZertoSessionURL -Headers $Header -Method POST -ContentType $Type 
$ZertoSessionSuccess = $TRUE
}
Catch 
{
$ZertoSessionSuccess = $FALSE
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$ZertoSession = $ZertoSessionResponse.headers.get_item("x-zerto-session")
$ZertoSessionHeader = @{"Accept"="application/json"
"x-zerto-session"="$ZertoSession"}
################################################
# IF not authenticated with Zerto no point continuing
################################################
IF (($ZertoSessionSuccess -eq $TRUE) -and ($CSVImport -ne $null))
{
################################################
# Starting for each VPG action
################################################
ForEach ($VPGName in $VPGsToConfigure)
{
# Output to host
"------------------------------"
"Configuring VPG: $VPGName"
# Getting VPG Identifier
$VPGID = $CSVImport | Where-Object {$_.VPGName -eq $VPGName} | select -ExpandProperty VPGID -Unique
# Getting list of VMs to reconfigure
$VMsToConfigure = $CSVImport | Where-Object {$_.VPGName -eq $VPGName} | select -ExpandProperty VMName -Unique
"Configuring VMs: $VMsToConfigure"
# Creating edit VPG JSON
$VPGJSON = 
"{
""VpgIdentifier"":""$VPGID""
}"
################################################
# Posting the VPG JSON Request to the API
################################################
$CreateVPGURL = $BaseURL+"vpgSettings"
Try 
{
$VPGSettingsID = Invoke-RestMethod -Method POST -Uri $CreateVPGURL -Body $VPGJSON -ContentType $Type -Headers $ZertoSessionHeader 
$ValidVPGSettingsID = $True
}
Catch 
{
$ValidVPGSettingsID = $False
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Skipping if unable to obtain valid VPG setting identifier
################################################
IF ($ValidVPGSettingsID -eq $True)
{
################################################
# Getting ZVR IDs for the VPG
################################################
# Getting VPG settings
$VPGSettingsURL = $BaseURL+"vpgSettings/"+$VPGSettingsID
Try 
{
$VPGSettings = Invoke-RestMethod -Method GET -Uri $VPGSettingsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Getting Recovery Site settings from API
################################################
# Setting recovery site ID (needed for network settings)
$VPGRecoverySiteID = $VPGSettings.Basic.RecoverySiteIdentifier
# Getting network info 
$VPGPortGroupsURL = $BaseURL+"virtualizationsites/$VPGRecoverySiteID/networks"
Try 
{
$VPGPortGroups = Invoke-RestMethod -Method GET -Uri $VPGPortGroupsURL -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Starting per VM actions
################################################
ForEach ($VMName in $VMsToConfigure)
{
# Getting VM settings from the CSV
$VMSettings = $CSVImport | Where-Object {$_.VPGName -eq $VPGName -and $_.VMName -eq $VMName} 
$VMID = $CSVImport | Where-Object {$_.VPGName -eq $VPGName -and $_.VMName -eq $VMName} | select -ExpandProperty VMID  -Unique
$VMNICIDs = $VMSettings.NICID
################################################
# Starting per VM NIC actions
################################################   
ForEach ($VMNICID in $VMNICIDs)
{
# Getting VM NIC settings
$VMNICSettings = $VMSettings | Where-Object {$_.NICID -eq $VMNICID} 
$VMNICFailoverNetworkName = $VMNICSettings.FailoverNetworkName
$VMNICFailoverDHCP = $VMNICSettings.FailoverDHCP
$VMNICFailoverStaticIp = $VMNICSettings.FailoverStaticIp
$VMNICFailoverSubnetMask = $VMNICSettings.FailoverSubnetMask
$VMNICFailoverGateway = $VMNICSettings.FailoverGateway
$VMNICFailoverPrimaryDns = $VMNICSettings.FailoverPrimaryDns
$VMNICFailoverSecondaryDns = $VMNICSettings.FailoverSecondaryDns
$VMNICFailoverDNSSuffix = $VMNICSettings.FailoverDNSSuffix
$VMNICFailoverShouldReplaceMacAddress = $VMNICSettings.FailoverShouldReplaceMacAddress
$VMNICFailoverTestNetworkName = $VMNICSettings.FailoverTestNetworkName
$VMNICFailoverTestDHCP = $VMNICSettings.FailoverTestDHCP
$VMNICFailoverTestStaticIp = $VMNICSettings.FailoverTestStaticIp
$VMNICFailoverTestSubnetMask = $VMNICSettings.FailoverTestSubnetMask
$VMNICFailoverTestGateway = $VMNICSettings.FailoverTestGateway
$VMNICFailoverTestPrimaryDns = $VMNICSettings.FailoverTestPrimaryDns
$VMNICFailoverTestSecondaryDns = $VMNICSettings.FailoverTestSecondaryDns
$VMNICFailoverTestDNSSuffix = $VMNICSettings.FailoverTestDNSSuffix
$VMNICFailoverTestShouldReplaceMacAddress = $VMNICSettings.FailoverTestShouldReplaceMacAddress
# Setting default DHCP to false if blank, or if static IP is configured, to prevent API errors
IF (($VMNICFailoverDHCP -eq "") -or ($VMNICFailoverStaticIp -ne ""))
{
$VMNICFailoverDHCP = "FALSE"
}
IF (($VMNICFailoverTestDHCP -eq "") -or ($VMNICFailoverTestStaticIp -ne ""))
{
$VMNICFailoverTestDHCP = "FALSE"
}
# Setting answers to lower case for API to process
$VMNICFailoverShouldReplaceMacAddress = $VMNICFailoverShouldReplaceMacAddress.ToLower()
$VMNICFailoverDHCP = $VMNICFailoverDHCP.ToLower()
$VMNICFailoverTestShouldReplaceMacAddress = $VMNICFailoverTestShouldReplaceMacAddress.ToLower()
$VMNICFailoverTestDHCP = $VMNICFailoverTestDHCP.ToLower()
# Translating network names to ZVR Network Identifiers
$VMNICFailoverNetworkID = $VPGPortGroups | where-object {$_.VirtualizationNetworkName -eq $VMNICFailoverNetworkName} | select -ExpandProperty NetworkIdentifier
IF (!$VMNICFailoverNetworkID){$VMNICFailoverNetworkID = "NotFound"}
$VMNICFailoverTestNetworkID = $VPGPortGroups | where-object {$_.VirtualizationNetworkName -eq $VMNICFailoverTestNetworkName} | select -ExpandProperty NetworkIdentifier
IF (!$VMNICFailoverTestNetworkID){$VMNICFailoverTestNetworkID = "NotFound"}
# Output to host if null to help with troubleshooting
IF (($VMNICFailoverNetworkID -eq "NotFound") -OR ($VMNICFailoverTestNetworkID -eq "NotFound"))
{
"---------------------"
"A valid port group setting ID was not found"
"To help you troubleshoot, here are the settings you specified in the CSV:"

"VM: $VMName"
"NICID: $VMNICID"
"FailoverNetworkName: $VMNICFailoverNetworkName"
"FailoverTestNetworkName: $VMNICFailoverTestNetworkName"

"Here is what the ZVM returned as valid port groups:"
$VPGPortGroups | Format-Table -Autosize
}
ELSE
{
################################################
# Building VMNIC JSON
################################################ 
$VMNICJSON = 
"    {
        ""Failover"":{
        ""Hypervisor"":{
            ""DnsSuffix"":""$VMNICFailoverDNSSuffix"",
            ""IpConfig"":{
                   ""Gateway"":""$VMNICFailoverGateway"",
                   ""IsDhcp"":$VMNICFailoverDHCP,
                   ""PrimaryDns"":""$VMNICFailoverPrimaryDns"",
                   ""SecondaryDns"":""$VMNICFailoverSecondaryDns"",
                   ""StaticIp"":""$VMNICFailoverStaticIp"",
                   ""SubnetMask"":""$VMNICFailoverSubnetMask""
                },
                ""NetworkIdentifier"":""$VMNICFailoverNetworkID"",
                ""ShouldReplaceMacAddress"":$VMNICFailoverShouldReplaceMacAddress
                }
            },
       ""FailoverTest"":{
        ""Hypervisor"":{
            ""DnsSuffix"":""$VMNICFailoverTestDNSSuffix"",
            ""IpConfig"":{
                   ""Gateway"":""$VMNICFailoverTestGateway"",
                   ""IsDhcp"":$VMNICFailoverTestDHCP,
                   ""PrimaryDns"":""$VMNICFailoverTestPrimaryDns"",
                   ""SecondaryDns"":""$VMNICFailoverTestSecondaryDns"",
                   ""StaticIp"":""$VMNICFailoverTestStaticIp"",
                   ""SubnetMask"":""$VMNICFailoverTestSubnetMask""
                },
                ""NetworkIdentifier"":""$VMNICFailoverTestNetworkID"",
                ""ShouldReplaceMACAddress"":$VMNICFailoverTestShouldReplaceMacAddress
                }
            },
        ""NicIdentifier"":""$VMNICID""
                }"
################################################
# Creating URL and sending PUT command to API
################################################ 
$EditVMNICURL = $BaseURL+"vpgSettings/"+$VPGSettingsID+"/vms/"+$VMID+"/nics/"+$VMNICID
"Editing NIC settings with PUT to: $EditVMNICURL"
Try 
{
Invoke-RestMethod -Method PUT -Uri $EditVMNICURL -Body $VMNICJSON -ContentType $Type -Headers $ZertoSessionHeader 
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Waiting for API processing
sleep 3
# End of skip if no valid Port group name specified for VMNIC below
}
# End of skip if no valid Port group name specified for VMNIC above
#
# End of for each VMNIC below
}
# End of for each VMNIC above
#
# End of for each VM below
}
# End of for each VM above
################################################
# Committing VPG settings
################################################
$CommitVPGSettingURL = $BaseURL+"vpgSettings/"+"$VPGSettingsID"+"/commit"
"Committing VPG Settings with POST to URL: $CommitVPGSettingURL"
Try 
{
Invoke-RestMethod -Method POST -Uri $CommitVPGSettingURL -Headers $ZertoSessionHeader -ContentType $Type -TimeoutSec 100
$VPGEditOutcome = "PASSED"
}
Catch 
{
$VPGEditOutcome = "FAILED"
$_.Exception.ToString()
$error[0] | Format-List -Force
}
"VPGEditOutcome: $VPGEditOutcome"
# Sleeping before processing next VPG
"Waiting 5 seconds before processing next VPG"
sleep 5
# End of check for valid VPG settings ID below
}
# End of check for valid VPG settings ID above
#
# End of per VPG actions below
}
# End of per VPG actions above
################################################
# IF not authenticated with Zerto no point continuing
################################################
}
ELSE
{
"Failed to login to Zerto API or couldn't load CSV, check server, credentials and the CSV config from the export script."
}
################################################
# End of script
################################################