﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script shows you how to connect to a vROPS 6.7 REST API, get a list of alerts and create a list of objects by alert
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - vROPS 6.7.0 (tested), should work with newer versions, probably not older
# - A username and password that can authenticate with
# - To view what REST APIs are avaible use https://yourvROPSIP/suite-api/docs/rest/index.html
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for vROPs
################################################
$vROPsServer = "192.168.1.4"
# Prompting for credentials, I was only able to login with a local user account in vROPs
$vROPsCredentials = Get-Credential -Message "Enter your vROPs credentials"
$vROPSUser = $vROPsCredentials.UserName
$vROPsCredentials.Password | ConvertFrom-SecureString
$vROPsPassword = $vROPsCredentials.GetNetworkCredential().password
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
# Adding certificate exception to prevent API errors
################################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
################################################
# Building vROPS API string & invoking REST API
################################################
$BaseURL = "https://" + $vROPsServer + "/suite-api/api/"
$BaseAuthURL = "https://" + $vROPsServer + "/suite-api/api/auth/token/acquire"
$Type = "application/json"
# Creating JSON for Auth Body
$AuthJSON =
"{
  ""username"": ""$vROPSUser"",
  ""password"": ""$vROPsPassword""
}"
# Authenticating with API
Try 
{
$vROPSSessionResponse = Invoke-RestMethod -Method POST -Uri $BaseAuthURL -Body $AuthJSON -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the session ID from the response
$vROPSSessionHeader = @{"Authorization"="vRealizeOpsToken "+$vROPSSessionResponse.'auth-token'.token
"Accept"="application/json"}
###############################################
# Getting Resources (used "$ResourcesJSON | Get-Member" to identify .resourcelist, API docs said nothing
###############################################
$ResourcesURL = $BaseURL+"resources?pageSize=5000"
Try 
{
$ResourcesJSON = Invoke-RestMethod -Method GET -Uri $ResourcesURL -Headers $vROPSSessionHeader -ContentType $Type
$Resources = $ResourcesJSON.resourcelist
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Building table of resources for lookup
$ResourceList = @()
ForEach($Resource in $Resources)
{
# Setting values
$ResourceHealth = $Resource.resourceHealth
$ResourceHealthValue = $Resource.resourceHealthValue
$ResourceID = $Resource.identifier
$ResourceName = $Resource.resourcekey.name
$ResourceType = $Resource.resourcekey.resourceKindKey
# Adding to table
$ResourceListRow = new-object PSObject
$ResourceListRow | Add-Member -MemberType NoteProperty -Name "Name" -Value "$ResourceName"
$ResourceListRow | Add-Member -MemberType NoteProperty -Name "Type" -Value "$ResourceType"
$ResourceListRow | Add-Member -MemberType NoteProperty -Name "Health" -Value "$ResourceHealth"
$ResourceListRow | Add-Member -MemberType NoteProperty -Name "HealthValue" -Value "$ResourceHealthValue"
$ResourceListRow | Add-Member -MemberType NoteProperty -Name "ID" -Value "$ResourceID"
$ResourceList += $ResourceListRow
}
$ResourceList | Sort-Object Type,Name | Format-Table
###############################################
# Getting Current Alerts
###############################################
$AlertsURL = $BaseURL+"alerts?pageSize=5000"
Try 
{
$AlertsJSON = Invoke-RestMethod -Method GET -Uri $AlertsURL -Headers $vROPSSessionHeader -ContentType $Type
$Alerts = $AlertsJSON.alerts
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
$ActiveAlerts = $Alerts | Where-Object {$_.status -eq "Active"}
$ActiveAlertsCount = $ActiveAlerts.count
# Output of result
"ActiveAlerts:$ActiveAlertsCount"
###############################################
# Building list of alerts with resource name rather than just ID
###############################################
$AlertList = @()
ForEach($ActiveAlert in $ActiveAlerts)
{
# Setting values
$AlertName = $ActiveAlert.alertDefinitionId
$AlertDescription = $ActiveAlert.alertDefinitionName
$AlertResourceID = $ActiveAlert.resourceId
$AlertLevel = $ActiveAlert.alertLevel
$AlertImpact = $ActiveAlert.alertImpact
# Converting date times from Epoch to readable format
$AlertStartTimeUTC = $ActiveAlert.startTimeUTC
$AlertUpdateTimeUTC = $ActiveAlert.updateTimeUTC
$AlertStartTime = [TimeZone]::CurrentTimeZone.ToLocalTime(([datetime]'1/1/1970').AddMilliSeconds($AlertStartTimeUTC))
$AlertUpdateTime = [TimeZone]::CurrentTimeZone.ToLocalTime(([datetime]'1/1/1970').AddMilliSeconds($AlertStartTimeUTC))
# Getting name and type of resource impacted
$AlertResourceName = $ResourceList | Where-Object {$_.ID -eq $AlertResourceID} | Select -ExpandProperty Name
$AlertResourceType = $ResourceList | Where-Object {$_.ID -eq $AlertResourceID} | Select -ExpandProperty Type
# Adding to table
$AlertListRow = new-object PSObject
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Name" -Value "$AlertResourceName"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Type" -Value "$AlertResourceType"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Level" -Value "$AlertLevel"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Impact" -Value "$AlertImpact"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Alert" -Value "$AlertName"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Description" -Value "$AlertDescription"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Start" -Value "$AlertStartTime"
$AlertListRow | Add-Member -MemberType NoteProperty -Name "Update" -Value "$AlertUpdateTime"
$AlertList += $AlertListRow
}
# Output of result
$AlertList | Sort-Object Type,Name,Level,Alert | Format-Table
# Critical only, you can use this output for other actions such as creating incidents in SNOW!
$CriticalAlertList = $AlertList | Where-Object {$_.Level -eq "CRITICAL"} | Sort-Object Name
$CriticalAlertList | Format-Table
###############################################
# End of script
###############################################