﻿#####################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
#####################################################################
# Written by: Joshua Stenhouse @ virtuallysober.com
##############################################
# Description:
# This script shows you how to connect to multiple different types/locations of SQL instances using the PowerShell SQL module, includes a ping test to help troubleshooting
# Script tested using Windows 10, PowerShell 5.1 and a mix of SQL Server 2016 and 2017
##############################################
# Requirements:
# - Internet access on the computer running this script to download the SQL module if not installed
# - At least 1 SQL instance to connect to!
##############################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever. 
# Including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss arising out of the use of,
# or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
#######################
# Configure the variables below
#######################
# Credential directory, you will be prompted to enter and save the username and password in the below directory (where required)
$CredentialDirectory = "C:\MSSQLConnectionExamplesv1\"
# Local SQL instance using Windows authentication of the user running the script
$LocalSQLInstance = "localhost\SQLEXPRESS"
# Remote SQL named instance using SQL server login
$RemoteSQLInstance = "SQL16-VM01.lab.local\INSTANCE01"
# Remote SQL default MSSQLSERVER instance using sa login
$RemoteDefaultSQLInstance  = "SQL16-VM01.lab.local"
#####################################################################
# Nothing to change below this line, commented throughout to explain
#####################################################################
##############################################
# Checking to see if the SqlServer module is already installed, if not installing it for the current user
##############################################
$SQLModuleCheck = Get-Module -ListAvailable SqlServer
if ($SQLModuleCheck -eq $null)
{
write-host "SqlServer Module Not Found - Installing"
# Not installed, trusting PS Gallery to remove prompt on install
Set-PSRepository -Name PSGallery -InstallationPolicy Trusted
# Installing module
Install-Module -Name SqlServer –Scope CurrentUser -Confirm:$false -AllowClobber
}
##############################################
# Importing the SqlServer module
##############################################
Import-Module SqlServer
##############################################
# Creating SQL Query to get the size of all DBs on each Instance
##############################################
# Change this to whatever you want to test, but this is a good one to start with!
$SQLDBSizeQuery = "SELECT
    DB_NAME(db.database_id) DatabaseName,
    (CAST(mfrows.RowSize AS FLOAT)*8)/1024 DBSizeMB,
    (CAST(mflog.LogSize AS FLOAT)*8)/1024 LogSizeMB
FROM sys.databases db
    LEFT JOIN (SELECT database_id, SUM(size) RowSize FROM sys.master_files WHERE type = 0 GROUP BY database_id, type) mfrows ON mfrows.database_id = db.database_id
    LEFT JOIN (SELECT database_id, SUM(size) LogSize FROM sys.master_files WHERE type = 1 GROUP BY database_id, type) mflog ON mflog.database_id = db.database_id"
############################################################################################
# Example 1 - Local SQL with Windows Authentication
############################################################################################
##############################################
# Testing query against Local SQL instance using Windows authentication of the user running the script
##############################################
# Getting the hostname of the SQL server by selectign everything before \
$LocalSQLInstanceHost = $LocalSQLInstance.Substring(0, $LocalSQLInstance.lastIndexOf('\'))
# Running ping test to SQL hostname
$SQLHostPingSuccess = Test-Connection -ComputerName $LocalSQLInstanceHost -Quiet -Count 2 
# Running the SQL Query, setting result of query to $False if any errors caught
Try 
{
# Setting to null first to prevent seeing previous query results
$SQLDBSizeResult = $null
# Running the query
$SQLDBSizeResult = Invoke-SqlCmd -Query $SQLDBSizeQuery -ServerInstance $LocalSQLInstance
# Setting the query result
$SQLQuerySuccess = $TRUE
}
Catch 
{
# Overwriting result if it failed
$SQLQuerySuccess = $FALSE
}
# Output of the results for you to see
"SQLInstance: $LocalSQLInstance"
"SQLHostPingResult: $SQLHostPingSuccess"
"SQLQueryResult: $SQLQuerySuccess"
"SQLQueryOutput:"
$SQLDBSizeResult
############################################################################################
# Example 2 - Remote SQL with SQL User Authentication
############################################################################################
##################################
# Importing Remote SQL User credentials, if not already created prompting user to create
##################################
# Setting credential file
$SQLUserCredentialsFile = $CredentialDirectory + "SQLUserCredentials.xml"
# Testing if file exists
$SQLUserCredentialsFileTest =  Test-Path $SQLUserCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($SQLUserCredentialsFileTest -eq $False)
{
$SQLUserCredentials = Get-Credential -Message "Enter SQL User login credentials"
$SQLUserCredentials | EXPORT-CLIXML $SQLUserCredentialsFile -Force
}
# Importing credentials
$SQLUserCredentials = IMPORT-CLIXML $SQLUserCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RemoteSQLUser = $SQLUserCredentials.UserName
$RemoteSQLPassword = $SQLUserCredentials.GetNetworkCredential().Password
##############################################
# Testing query against Remote SQL named instance using SQL server login
##############################################
# Getting the hostname of the SQL server by selectign everything before \
$RemoteSQLInstanceHost = $RemoteSQLInstance.Substring(0, $RemoteSQLInstance.lastIndexOf('\'))
# Running ping test to SQL hostname
$SQLHostPingSuccess = Test-Connection -ComputerName $RemoteSQLInstanceHost -Quiet -Count 2 
# Running the SQL Query, setting result of query to $False if any errors caaught
Try 
{
# Setting to null first to prevent seeing previous query results
$SQLDBSizeResult = $null
# Running the query
$SQLDBSizeResult = Invoke-SqlCmd -Query $SQLDBSizeQuery -ServerInstance $RemoteSQLInstance -Username $RemoteSQLUser -Password $RemoteSQLPassword
# Setting the query result
$SQLQuerySuccess = $TRUE
}
Catch 
{
# Overwriting result if it failed
$SQLQuerySuccess = $FALSE
}
# Output of the results for you to see
"SQLInstance: $RemoteSQLInstance"
"SQLHostPingResult: $SQLHostPingSuccess"
"SQLQueryResult: $SQLQuerySuccess"
"SQLQueryOutput:"
$SQLDBSizeResult
############################################################################################
# Example 3 - Remote SQL with SA Authentication
############################################################################################
##################################
# Importing Remote SA User credentials, if not already created prompting user to create
##################################
# Setting credential file
$SQLSAUserCredentialsFile = $CredentialDirectory + "SQLSAUserCredentials.xml"
# Testing if file exists
$SQLSAUserCredentialsFileTest =  Test-Path $SQLSAUserCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($SQLSAUserCredentialsFileTest -eq $False)
{
$SQLSAUserCredentials = Get-Credential -User "sa" -Message "Enter SQL SA login password"
$SQLSAUserCredentials | EXPORT-CLIXML $SQLSAUserCredentialsFile -Force
}
# Importing credentials
$SQLSAUserCredentials = IMPORT-CLIXML $SQLSAUserCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RemoteSQLSAUser = $SQLSAUserCredentials.UserName
$RemoteSQLSAPassword = $SQLSAUserCredentials.GetNetworkCredential().Password
##############################################
# Testing query against Remote SQL default MSSQLSERVER instance using sa login
##############################################
# Running ping test to SQL hostname
$SQLHostPingSuccess = Test-Connection -ComputerName $RemoteDefaultSQLInstance -Quiet -Count 2 
# Running the SQL Query, setting result of query to $False if any errors caaught
Try 
{
# Setting to null first to prevent seeing previous query results
$SQLDBSizeResult = $null
# Running the query
$SQLDBSizeResult = Invoke-SqlCmd -Query $SQLDBSizeQuery -ServerInstance $RemoteDefaultSQLInstance -Username $RemoteSQLSAUser -Password $RemoteSQLSAPassword
# Setting the query result
$SQLQuerySuccess = $TRUE
}
Catch 
{
# Overwriting result if it failed
$SQLQuerySuccess = $FALSE
}
# Output of the results for you to see
"SQLInstance: $RemoteSQLInstance"
"SQLHostPingResult: $SQLHostPingSuccess"
"SQLQueryResult: $SQLQuerySuccess"
"SQLQueryOutput:"
$SQLDBSizeResult
##############################################
# End of script
##############################################