﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script gets a list of VMs, then protects those with the matching string which are not already protected by the SLA domain name configured
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A matching VM string and SLA domain name with which to protect them
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
# File input settings
$ScriptDirectory = "C:\RubrikAutoVMProtectv1\"
# String to search for to protect VMs
$VMString = "AWS"
# SLA domain name to apply to the VMs matching the string
$SLADomain = "DemoSLA1"
# Time delay between protecting VMs in seconds, 1 is fine
$TimeDelay = 1
# Dry run, won't apply the change
$DryRunEnabled = $TRUE
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of VMs (to convert VM name to ID later)
##################################
$VMListURL = $BaseURL+"vmware/vm?limit=5000&is_relic=false"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainListURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainListJSON = Invoke-RestMethod -Uri $SLADomainListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainList = $SLADomainListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting SLA Domain ID
$SLADomainID = $SLADomainList | Where-Object {$_.name -eq $SLADomain} | Select -ExpandProperty id
##################################
# Finding VMs to Protect where not already assigned to SLA specified
##################################
$VMsToProtect = $VMList | Where-Object {(($_.name -match $VMString) -and ($_.effectiveSlaDomainName -ne $SLADomain))}
$VMNamesToProtect = $VMsToProtect | Select -ExpandProperty name
# Counting VMs
$VMsToProtectCount = $VMsToProtect | Measure-Object | Select -ExpandProperty Count
# Output to host for user
"
--------------------------------
VMsFound: $VMsToProtectCount
SLADomainIDFound: $SLADomainID
--------------------------------
Protecting:"
$VMNamesToProtect
##################################
# Bypassing if no SLA Domain ID found
##################################
IF (!$SLADomainID)
{
"SLADomainNotFound: $SLADomain
Check name and try again..."
}
ELSE
{
##################################
# Performing For Each VM Action
##################################
$Iteration = 0
ForEach ($VM in $VMsToProtect)
{
# Incrementing interation
$Iteration ++
# Assigning variables from the CSV
$VMName = $VM.name
$VMID = $VM.id
# Output to host for user
"--------------------------
VM: $Iteration
Name: $VMName
SLADomain: $SLADomain"
########################
# Assigning SLA Domain to VM
########################
IF ($DryRunEnabled -eq $False)
{
# Creating URL
$AssignSLADomainURL = $InternalURL+"sla_domain/"+$SLADomainID+"/assign"
# Creating JSON body
$SLADomainJSON = 
"{
  ""managedIds"": [""$VMID""]
}"
# Posting to the API
Try 
{
$AssignSLADomain = Invoke-RestMethod -Method POST -Uri $AssignSLADomainURL -Body $SLADomainJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$ProtectionJob = "SUCCESS"
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
$ProtectionJob = "FAIL"
}
# Sleeping wait time specified
sleep $TimeDelay
# Output to host for user
"Task: $ProtectionJob"
# End of bypass for DryRunEnabledBelow
}
ELSE
{
# Output to host as dry run was enabled
"DryRun: Enabled"
}
# End of per VM action below
}
# End of per VM action above
#
# End of bypass for no SLA domain ID found below
}
# End of bypass for no SLA domain ID found above
##################################
# End of script
##################################