﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script automatically exports the all the VMware VMs and SLA domains in Rubrik to a CSV for editing then import (to change settings), anything with an explicity Do Not Protect is assigned with an SLA called DoNotProtect
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance with login credentials and a registered vCenter 
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
# File input settings
$ScriptDirectory = "C:\RubrikAutoVMProtectv1\"
$VMCSV = "C:\RubrikAutoVMProtectv1\RubrikVMList.csv"
# Toggle the below if you only want to export Unprotected VMs
$ExportUnprotectedOnly = $FALSE
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
################################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
################################################
# Building Rubrik API string & invoking REST API
################################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authentication with API
Try 
{
$RubrikSessionResponse = Invoke-WebRequest -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSession = (ConvertFrom-Json -InputObject $RubrikSessionResponse.Content)
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSession.token)"}
###############################################
# Getting list of VMs
###############################################
$VMListURL = $BaseURL+"vmware/vm?primary_cluster_id=local&limit=5000&is_relic=false&sort_by=name&sort_order=asc"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Removing relic VMs as they no longer exist in the vCenter
$VMListFiltered = $VMList | Where-Object {$_.isRelic -ne "True"}
##################################
# If ExportUnprotectedOnly is TRUE, removing VMs already protected
##################################
IF ($ExportUnprotectedOnly -eq $TRUE)
{
$VMListFiltered = $VMListFiltered | Where-Object {(($_.effectiveSlaDomainName -eq "Unprotected") -and ($_.slaAssignment -eq "Unassigned"))} 
}
##################################
# Getting list of SLA Domains
##################################
$SLAListURL = $BaseURL+"sla_domain"
Try 
{
$SLAListJSON = Invoke-RestMethod -Uri $SLAListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLAList = $SLAListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
##################################
# Getting list of vCenters
##################################
$vCenterListURL = $BaseURL+"vmware/vcenter"
Try 
{
$vCenterListJSON = Invoke-RestMethod -Uri $vCenterListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$vCenterList = $vCenterListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Building a table of all VM info to export
###############################################
$VMExportList=@()
# Interating through each VM
ForEach ($VM in $VMListFiltered)
{
# Setting values
$VMName = $VM.name
$VMSLADomain = $VM.effectiveSlaDomainName
$VMPowerStatus = $VM.powerStatus
$VMCluster = $VM.clusterName
$VMID = $VM.id
$VMGUestIP = $VM.ipAddress
$VMSLAAssignment = $VM.slaAssignment
$VMvCenterID = $VM.vcenterId
# Translating vCenter ID into friendly name
$VMvCenterName = $vCenterList | Where-Object {$_.id -eq $VMvCenterID} | Select -ExpandProperty name
# Showing SLA domain as DoNotProtect if SLA is Unprotected by assignment
IF (($VMSLADomain -eq "Unprotected") -and ($VMSLAAssignment -eq "Direct"))
{
$VMSLADomain = "DoNotProtect"
}
# Creating table row
$VMExportListLine = New-Object PSObject
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "VMName" -Value "$VMName"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "NewSLADomain" -Value $null
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "ExistingSLADomain" -Value "$VMSLADomain"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "SLAAssignment" -Value "$VMSLAAssignment"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "vCenter" -Value "$VMvCenterName"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "Cluster" -Value "$VMCluster"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "PowerStatus" -Value "$VMPowerStatus"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "GuestIP" -Value "$VMGUestIP"
$VMExportListLine | Add-Member -MemberType NoteProperty -Name "ID" -Value "$VMID"
# Adding row to table
$VMExportList += $VMExportListLine
}
###############################################
# Exporting VM CSV
###############################################
$VMExportList | Sort-Object ExistingSLADomain,Cluster,VM| Export-Csv $VMCSV -Force -NoTypeInformation
"CreatedCSV: $VMCSV"
###############################################
# Showing a list of the available SLA domains to enter into your CSV export
###############################################
"----------------------------------------------
- SLA Domains Available For Selection
- Use DoNotProtect to explicity stop an object being protected
----------------------------------------------"
$SLAList | Sort-Object name | Select -ExpandProperty name | Format-Table
###############################################
# End of script
###############################################