﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script processes VMName, NewSLADomain in the CSV and protects the VM with the SLA specified 
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A CSV with the following columns: VMName,SLADomain
# - Example CSV line: SQLServer-VM01,GlacierArchive
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
# File input settings
$ScriptDirectory = "C:\RubrikAutoVMProtectv1\"
$VMCSV = "C:\RubrikAutoVMProtectv1\RubrikVMList.csv"
# Time delay between protecting VMs in seconds, 1 is fine
$TimeDelay = 1
# Dry run, won't apply the change
$DryRunEnabled = $TRUE
################################################
# Nothing to configure below this line - Starting the main function of the script
################################################
##################################
# Importing Rubrik credentials
##################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
# Setting the username and password from the credential file (run at the start of each script)
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
##################################
# Importing the CSV & Filtering out objects with no SLA to assign
##################################
$VMsToProtectImport = Import-CSV $VMCSV 
$VMsToProtect = $VMsToProtectImport | Where-Object {$_.NewSLADomain -ne ""}
##################################
# Building Rubrik API string & invoking REST API
##################################
$BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
##################################
# Getting list of VMs (to convert VM name to ID later)
##################################
$VMListURL = $BaseURL+"vmware/vm?primary_cluster_id=local&limit=5000&is_relic=false&sort_by=name&sort_order=asc"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$Error[0] | Format-List -Force
}
##################################
# Getting list of SLA Domains (to convert SLA domain name to ID later)
##################################
$SLADomainListURL = $BaseURL+"sla_domain"
Try 
{
$SLADomainListJSON = Invoke-RestMethod -Uri $SLADomainListURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$SLADomainList = $SLADomainListJSON.data
}
Catch 
{
$Error[0] | Format-List -Force
}
##################################
# Performing For Each VM On Demand Snapshot Action
##################################
$Iteration = 0
$VMsToProtectCount = $VMsToProtect | Measure-Object | Select -ExpandProperty Count
# Output to host
"--------------------------
Configuring Protection for $VMsToProtectCount VMs"
# For each VM configuring SLA
ForEach ($VM in $VMsToProtect)
{
# Assigning variables from the CSV
$VMName = $VM.VMName
$VMNewSLADomain = $VM.NewSLADomain
# Incrementing interation
$Iteration ++
# Selecting VM ID
$VMID = $VMList | Where-Object {$_.name -eq "$VMName"} | Select -ExpandProperty id
IF (!$VMID)
{
$VMID = "IDNotFound"
}
# Output to host
"--------------------------
ProtectingVM: $Iteration
VM: $VMName
NewSLADomain: $VMNewSLADomain
VMID: $VMID"
##################################
# Assigning VM DoNotProtect if valid VM ID found and NewSLADomain/VMSLADomain set to DoNotProtect
##################################
IF (($VMNewSLADomain -eq "DoNotProtect") -and ($VMID -ne "NotFound"))
{
# Creating URL
$AssignSLADomainURL = $InternalURL+"sla_domain/UNPROTECTED/assign"
# Creating JSON body
$SLADomainJSON = 
"{
  ""managedIds"": [""$VMID""]
}"
# Posting to the API
Try 
{
Invoke-RestMethod -Method Post -Uri $AssignSLADomainURL -Body $SLADomainJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
}
##################################
# Selecting SLA domain ID, as it isn't set to DoNotProtect
##################################
IF (($VMNewSLADomain -ne "DoNotProtect") -and ($VMID -ne "NotFound"))
{
$VMSLADomainID = $SLADomainList | Where-Object {$_.Name -eq $VMNewSLADomain} | Select -ExpandProperty id
IF (!$VMSLADomainID){$VMSLADomainID = "NotFound"}
##################################
# Skipping if not all valid IDs found
##################################
IF (($VMID -eq "IDNotFound") -or ($VMSLADomainID -eq "NotFound"))
{
"Failed to find Valid VM ID, didn't run SLA assignment"
}
ELSE
{
########################
# Posting To URL to configure backup of VM
########################
IF ($DryRunEnabled -eq $False)
{
# Creating URL
$AssignSLADomainURL = $InternalURL+"sla_domain/"+$VMSLADomainID+"/assign"
# Creating JSON body
$SLADomainJSON = 
"{
  ""managedIds"": [""$VMID""]
}"
# Posting to the API
Try 
{
$ProtectVM = Invoke-RestMethod -Method Post -Uri $AssignSLADomainURL -Body $SLADomainJSON -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$Protected = $TRUE
}
Catch 
{
$Error[0] | Format-List -Force
$Protected = $FALSE
}
# Output to host
"VMProtected: $Protected"
# Sleeping wait time specified
sleep $TimeDelay
}
ELSE
{
# Output to host as dry run was enabled
"DryRun: Enabled"
}
##################################
# End of skip if not all valid IDs found below
##################################
}
# End of skip if not all valid IDs found above
#
##################################
# End of Selecting SLA domain ID, as it isn't set to DoNotProtect
##################################
}
# End of Selecting SLA domain ID, as it isn't set to DoNotProtect
#
##################################
# End of per VM action below
##################################
}
# End of per VM action above
##################################
# End of script
##################################