﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script runs a Recovery Plan for VMs specified in a CSV to be recovered by Rubrik, validates the job outcome, then sends an email of the result
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A CSV with the following fields: VMName,HostSelection,DisableNetwork,RemoveNetworkDevices,KeepMACAddress,RecoverTags,RubrikPowerOn,NextVMFailoverDelay
# - Example for a DR Test with no NIC = DemoApp1-VM01,192.168.2.21,FALSE,TRUE,FALSE,FALSE,TRUE,10
# - Example for a DR Test with NIC disconnected = DemoApp1-VM01,192.168.2.21,TRUE,FALSE,FALSE,FALSE,TRUE,10
# - Example for a DR Failover (live) = DemoApp1-VM01,192.168.2.21,FALSE,FALSE,FALSE,FALSE,TRUE,10
# - Valid options for HostSelection are RANDOM (uses the existing ESXi host of the VM if recovering to prod) or the name of the ESXi host as registered in the vCenter
# - Valid options for DisableNetwork,RemoveNetworkDevices,RubrikPowerOn are TRUE or FALSE
# - This script always fails over to the latest snapshot available
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "rubrik.lab.local"
$RecoveryPlanCSV = "C:\RubrikRecoveryPlanv5\RubrikRecoveryPlan.csv"
$LogDirectory = "C:\RubrikRecoveryPlanv5\Logs"
# Logo file, change to your logo if desired (recommended 150 by x pixels)
$LogoFile = "C:\RubrikRecoveryPlanv5\Tools\Logo.png"
# Configure the max time to wait for a VM to power on before moving onto the next and failing the VM result
$MaxVMWaitTimeSeconds = 300
# VM suffix is added to the end of each VM name as its registered in the vCenter, set to $null if you just want the VM name as is, I.E for recovery with existing VM gone
$VMSuffix = "-FailoverTest"
# Optional settings, CSV goes to the log directory
$ExportCSV = $TRUE
$EmailOnFailures = $TRUE
$EmailOnAll = $TRUE
# HTML Color codes used for report rows
$HTMLColorSuccess = "#696969"
$HTMLColorFailure = "#e60000"
# Email settings
$EmailTo = "joshua@lab.local"
$EmailFrom = "rubrik@lab.local"
$EmailServer = "localhost"
# Email subject
$EmailSubject = "Rubrik VM Recovery Report"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Creating the Convert-UNIXTime function
################################################
Function Convert-UNIXTime {
Param ($UNIXTime)
# Example: $PSTime = Convert-UNIXTime -UNIXTime $UNIXTime
# Step 1 - Removing characters and trimming snapshot string for conversion
$PSTimeStep1 = $UNIXTime.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Removing last 4 characters
$PSTimeStep2 = $PSTimeStep1.Substring(0,$PSTimeStep1.Length-4)
# Step 3 - Converting string to PowerShell datetime object
$PSTimeStep3 = ([datetime]::ParseExact($PSTimeStep2,”yyyy-MM-dd HH:mm:ss”,$null))
# Returning Result
Return $PSTimeStep3
}
################################################
# Starting logging & importing the CSV
################################################
# Checking log path exists, if not trying to create it
$LogFilePathTest = Test-Path $LogDirectory
# Creating if false
IF ($LogFilePathTest -eq $False)
{
New-Item -Path $LogDirectory -ItemType "directory"
}
# Getting time
$Now = Get-Date
# Creating log file name
$Log = $LogDirectory + "\Rubrik-RecoveryPlanLog-Auto-" + $Now.ToString("yyyy-MM-dd") + "@" + $Now.ToString("HH-mm-ss") + ".log"
# Starting logging
Start-Transcript -Path $Log -NoClobber 
$RecoveryPlanVMs = Import-CSV $RecoveryPlanCSV
# Filtering CSV to remove any lines where VMName is empty (sometimes people leave half finished rows)
$RecoveryPlanVMs= $RecoveryPlanVMs | Where-Object {$_.VMName -ne $null}
# Counting VMs to test
$RecoveryPlanVMCount = $RecoveryPlanVMs.Count
###############################################
# Importing Rubrik credentials
###############################################
# Setting credential file
$RubrikCredentialsFile = $LogDirectory + "\RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
ELSE
{
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
}
# Setting credentials
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception and TLS 1.2 to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
################################################
# Building Rubrik API string & invoking REST API
################################################
$v1BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$v2BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $v1BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
###############################################
# Getting list of VMs
###############################################
$VMListURL = $v1BaseURL+"vmware/vm?limit=5000"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# For troubleshooting output the list of VMs found: $VMList | Select Name | Sort-Object Name,id | Format-Table -AutoSize
###############################################
# Getting list of Hosts
###############################################
$VMHostListURL = $v1BaseURL+"vmware/host"
Try 
{
$VMHostListJSON = Invoke-RestMethod -Uri $VMHostListURL -Headers $RubrikSessionHeader -ContentType $Type
$VMHostList = $VMHostListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of vCenters
###############################################
$vCenterListURL = $v1BaseURL+"vmware/vcenter"
Try 
{
$vCenterListJSON = Invoke-RestMethod -Uri $vCenterListURL -Headers $RubrikSessionHeader -ContentType $Type
$vCenterList = $vCenterListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of Rubrik Nodes
###############################################
$RubrikNodesURL = $InternalURL+"node"
Try 
{
$RubrikNodesJSON = Invoke-RestMethod -Uri $RubrikNodesURL -Headers $RubrikSessionHeader -ContentType $Type
$RubrikNodes = $RubrikNodesJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###################################################################
# Step 1: Start Per VM Recovery Actions here
###################################################################
"----------------------------------------------------------------------------------------"
"Step 1: Starting per VM RecoveryPlan Actions for $RecoveryPlanVMCount VMs using VMSuffix:$VMSuffix"
"----------------------------------------------------------------------------------------"
"UsingCSV: $RecoveryPlanCSV
RubrikCluster: $RubrikCluster
RubrikUser: $RubrikUser
"
"VMs to test in order of testing:"
$RecoveryPlanVMs | Format-Table -AutoSize
"--------------------------------------------"
# Creating array for live mount results
$VMLiveMountArray =@()
$VMLiveMountCounter = 0
$VMLiveMountCount = $RecoveryPlanVMs | Measure | Select -ExpandProperty Count
# Starting per vm action
ForEach ($VM in $RecoveryPlanVMs)
{
# Incrementing counter
$VMLiveMountCounter ++
# Output to host
"Processing: $VMLiveMountCounter / $VMLiveMountCount" 
###############################################
# Setting the variables for the current VM
###############################################
$VMName = $VM.VMName
$VMHostSelection = $VM.HostSelection
$VMDisableNetwork = $VM.DisableNetwork
$VMRemoveNetworkDevices = $VM.RemoveNetworkDevices
$VMKeepMACAddress = $VM.KeepMACAddress
$VMRecoverTags = $VM.RecoverTags
$VMRubrikPowerOn = $VM.RubrikPowerOn
$VMNextVMFailoverDelay = $VM.NextVMFailoverDelay
# Setting VM live mount name
$VMLiveMountName = $VMName + $VMSuffix
# Counting if same VM has been specified multiple times in plan
$VMDuplicateCount = $RecoveryPlanVMs | Where-Object {$_.VMName -eq $VMName} | Measure | Select -ExpandProperty Count
IF ($VMDuplicateCount -ge 2)
{
# Generating random ID to prevent conflicts
$VMRandomID = 1..9999 | Get-Random
# Adding random ID to name
$VMLiveMountName = $VMLiveMountName + " " + $VMRandomID
}
# Getting SLA domain name
$VMSLADomain = $VMList | Where-Object {(($_.name -eq $VMName) -and ($_.effectiveSlaDomainId -ne "UNPROTECTED"))} | Select -ExpandProperty effectiveSlaDomainName
# Inserting space in log for readability
"VM: $VMName"
# Giving the user 3 seconds to see
sleep 3
###############################################
# Getting VM ID and VM snapshot info
###############################################
# Creating array to store VM snapshots for all matching VM IDs
$VMSnapshots = @()
# Selecting VM ID, if multiple will cycle through each to find a snapshot (can happen if multiple VMs exist with the same name)
$VMIDs = $VMList | Where-Object {(($_.name -eq $VMName) -and ($_.effectiveSlaDomainId -ne "UNPROTECTED"))}
ForEach ($VMID in $VMIDs)
{
# Setting values
$VMIDName = $VMID.name
$VMID = $VMID.id
$VMSnapshotID = $null
$VMSnapshotDate = $null
# Building Snapshot URL
$VMSnapshotListURL = $v1BaseURL+"vmware/vm/"+$VMID+"/snapshot"
# Getting list of snapshots for the VMID
Try 
{
$VMSnapshotListJSON = Invoke-RestMethod -Uri $VMSnapshotListURL -Headers $RubrikSessionHeader -ContentType $Type
$VMSnapshotList = $VMSnapshotListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Selecting the first (and most recent snapshot) for the VMID, converting then adding to the array for selection
$VMLatestSnapshot = $VMSnapshotList | Select -First 1
# Getting ID and UNIX date time
$VMLatestSnapshotID = $VMLatestSnapshot.id
$VMLatestSnapshotDateUNIX = $VMLatestSnapshot.date
$VMLatestSnapshotIsOnDemand = $VMLatestSnapshot.isOnDemandSnapshot
# Converting from string to datetime format
$VMLatestSnapshotDate = Convert-UNIXTime -UNIXTime $VMLatestSnapshotDateUNIX
# Adding to array
$VMSnapshot = New-Object PSObject
$VMSnapshot | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VMID
$VMSnapshot | Add-Member -MemberType NoteProperty -Name "Date" -Value $VMLatestSnapshotDate
$VMSnapshot | Add-Member -MemberType NoteProperty -Name "SnapshotID" -Value $VMLatestSnapshotID
$VMSnapshot | Add-Member -MemberType NoteProperty -Name "IsOnDemand" -Value $VMLatestSnapshotIsOnDemand
$VMSnapshot | Add-Member -MemberType NoteProperty -Name "VMIDName" -Value $VMIDName
$VMSnapshots += $VMSnapshot
}
###############################################
# Selecting VMID and VMSnapshotID where a VMSnapshotID exists
###############################################
# Selecting most recent snapshot
$VMSnapshotSelected = $VMSnapshots | Where-Object {$_.SnapshotID -ne ""} | Sort-Object -Descending SnapshotDate | Select -First 1
# From latest snapshot across all matching VM IDs found setting variables
$VMSnapshotID = $VMSnapshotSelected.SnapshotID 
$VMSnapshotDate = $VMSnapshotSelected.Date
$VMIDSelected = $VMSnapshotSelected.VMID
# Setting VMID value if not found (for logging)
IF ($VMIDSelected -eq $null)
{
$VMIDSelected = "VM and/or Snapshot Not Found In Rubrik"
}
###########################################
# Selecting VM host, only selecting hosts with datastores ensures that you aren't selecting a host ID which is replicated vs usable by the Rubrik cluster
###########################################
IF ($VMHostSelection -ne "RANDOM")
{
# ESXi hostname has been specified, selecting the host
$VMHostID = $VMHostList |  Where-Object {($_.name -eq $VMHostSelection) -and ($_.datastores -ne $null)} | Select -ExpandProperty id -First 1
$VMHostName = $VMHostList |  Where-Object {$_.id -eq $VMHostID} | Select -ExpandProperty name -First 1
}
# Setting to RANDOM if no ESXi host found or set to RANDOM
IF (($VMHostID -eq $null) -or ($VMHostSelection -eq "RANDOM"))
{
$VMHostID = $VMHostList | Where-Object {$_.datastores -ne $null} | Get-Random | Select -ExpandProperty id
$VMHostName = $VMHostList |  Where-Object {$_.id -eq $VMHostID} | Select -ExpandProperty name -First 1
}
###########################################
# Getting vCenter for the ESXi host and VM selected
###########################################
$VMHostvCenterID = $VMHostList | Where-Object {$_.id -eq $VMHostID} | Select -ExpandProperty effectiveSlaSourceObjectId
# Getting vCenter name
$VMHostvCenterName = $vCenterList | Where-Object {$_.id -eq $VMHostvCenterID } | Select -ExpandProperty name
###########################################
# Getting source vCenter and Host names
###########################################
# Getting source ESXi host name
$VMSourceHostName = $VMIDs | Where-Object {$_.id -eq $VMIDSelected} | Select -ExpandProperty hostName
# Getting source vCenter
$VMSourcevCenterID = $VMIDs | Where-Object {$_.id -eq $VMIDSelected} | Select -ExpandProperty vcenterId
$VMSourcevCenterName = $vCenterList | Where-Object {$_.id -eq $VMSourcevCenterID } | Select -ExpandProperty name
###########################################
# Logging settings
###########################################
"Using the following values:
VMID: $VMIDSelected
vCenter: $VMHostvCenterName
HostName: $VMHostName
SnapshotID: $VMSnapshotID
SnapshotDate: $VMSnapshotDate
DisableNetwork: $VMDisableNetwork
RemoveNetworkDevices: $VMRemoveNetworkDevices
KeepMACAddress: $VMKeepMACAddress
RecoverTags: $VMRecoverTags
RubrikPowerOn: $VMRubrikPowerOn
NextVMFailoverDelay: $VMNextVMFailoverDelay"
###########################################
# Creating JSON & configuring URL
###########################################
# Setting default if not specified in CSV, or not configured correctly, defaulting to safe set of options
IF (($VMDisableNetwork -eq "") -or ($VMDisableNetwork -ne "FALSE")){$VMDisableNetwork = "true"}
IF (($VMRemoveNetworkDevices -eq "") -or ($VMRemoveNetworkDevices -ne "TRUE")){$VMRemoveNetworkDevices = "false"}
IF (($VMRubrikPowerOn -eq "") -or ($VMRubrikPowerOn -ne "TRUE")){$VMRubrikPowerOn = "false"}
IF (($VMKeepMACAddress -eq "") -or ($VMKeepMACAddress -ne "TRUE")){$VMKeepMACAddress = "false"}
IF (($VMRecoverTags -eq "") -or ($VMRecoverTags -ne "TRUE")){$VMRecoverTags = "false"}
# Forcing to lower case to compensate for excel auto-correct capitalizing 
$VMDisableNetwork = $VMDisableNetwork.ToLower()
$VMRemoveNetworkDevices = $VMRemoveNetworkDevices.ToLower()
$VMRubrikPowerOn = $VMRubrikPowerOn.ToLower()
$VMLMJSON =
"{
  ""vmName"": ""$VMLiveMountName"",
  ""hostId"": ""$VMHostID"",
  ""disableNetwork"": $VMDisableNetwork,
  ""removeNetworkDevices"": $VMRemoveNetworkDevices,
  ""powerOn"": $VMRubrikPowerOn,
  ""keepMacAddresses"": $VMKeepMACAddress,
  ""shouldRecoverTags"": $VMRecoverTags
}"
###########################################
# POST to REST API URL with VMJSON, but only if a VMID is found
###########################################
# URL
$VMLiveMountURL = $v1BaseURL+"vmware/vm/snapshot/"+$VMSnapshotID+"/mount"
# POST
IF ($VMSnapshotID -ne $null)
{
Try 
{
"StartingVMLiveMount: $VMLiveMountName"
$VMLiveMountRequest = Invoke-RestMethod -Method Post -Uri $VMLiveMountURL -Body $VMLMJSON -Headers $RubrikSessionHeader -ContentType $Type
$VMOperationSuccess = $TRUE
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
$VMOperationSuccess = $FALSE
}
# Output to host
"OperationSuccess: $VMOperationSuccess"
###########################################
# Adding VM LiveMount Task ID to table for subsequent verification
###########################################
# Getting status URL and job instance ID
$VMLiveMountStatusURL = $VMLiveMountRequest.links.href
$VMLiveMountJobInstanceID = $VMLiveMountRequest.id
# Getting start time
$VMLiveMountStartUNIX = $VMLiveMountRequest.startTime
$VMLiveMountStart = Convert-UNIXTime -UNIXTime $VMLiveMountStartUNIX
# Output to host
"Started: $VMLiveMountStart"
"--------------------------------------------"
# Adding info to table/array
$VMLiveMountArrayLine = New-Object PSObject
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "Order" -Value $VMLiveMountCounter
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "VM" -Value $VMIDName
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "vCenter" -Value $VMHostvCenterName
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "ESXiHost" -Value $VMHostName
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "SourcevCenter" -Value $VMSourcevCenterName
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "SourceESXiHost" -Value $VMSourceHostName
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value $VMSLADomain
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VMIDSelected
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "FailoverVMDelay" -Value $VMNextVMFailoverDelay
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "LiveMountName" -Value $VMLiveMountName
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "LiveMountRequest" -Value $VMOperationSuccess
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "Start" -Value $VMLiveMountStart
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "SnapshotDate" -Value $VMSnapshotDate
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "StatusURL" -Value $VMLiveMountStatusURL
$VMLiveMountArrayLine | Add-Member -MemberType NoteProperty -Name "JobInstanceID" -Value $VMLiveMountJobInstanceID
$VMLiveMountArray += $VMLiveMountArrayLine
# End of live mount operation
}
ELSE
{
$VMOperationSuccess = $FALSE
# Logging result
"OperationSuccess: $VMOperationSuccess"
}
###########################################
# Waiting for VMNextVMFailoverDelay if start test was a success
###########################################
IF ($VMOperationSuccess -eq $TRUE)
{
# Ignoring sleep delay if VM isn't set to power on, boot delay will be observed only when powering on VMs
IF ($VMRubrikPowerOn -eq "true")
{
"Sleeping $VMNextVMFailoverDelay seconds for NextVMFailoverDelay"
sleep $VMNextVMFailoverDelay
}
}
# Output to host
"--------------------------------------------"
#
# End of per VM recovery actions below
}
# End of per VM recovery actions above
###################################################################
# Step 2: Start Per VM Validation Actions here
###################################################################
# Inserting space in log for readability
"----------------------------------------------------------------------------------------"
"Step 2: Starting Recovery Validations"
"----------------------------------------------------------------------------------------"
# Creating table/array for results
$VMLiveMountResults = @()
$VMLiveMountTasks = @()
# Counting
$VMLiveMountCounter = 0
$VMLiveMountCount = $VMLiveMountArray | Measure | Select -ExpandProperty Count
# For Each VM action
ForEach ($VMLiveMount in $VMLiveMountArray)
{
# Incrementing counter
$VMLiveMountCounter ++
# Setting variables
$VMLiveMountOrder = $VMLiveMount.Order
$VMLiveMountVMName = $VMLiveMount.VM
$VMLiveMountvCenter = $VMLiveMount.vCenter
$VMLiveMountHost = $VMLiveMount.ESXiHost
$VMLiveMountSourcevCenter = $VMLiveMount.SourcevCenter
$VMLiveMountSourceHost = $VMLiveMount.SourceESXiHost
$VMLiveMountSLADomain = $VMLiveMount.SLADomain
$VMLiveMountFailoverVMDelay = $VMLiveMount.FailoverVMDelay
$VMLiveMountVMID = $VMLiveMount.VMID
$VMLiveMountVMIDShort = $VMLiveMountVMID.Replace("VirtualMachine:::","")
$VMLiveMountTestName = $VMLiveMount.LiveMountName
$VMLiveMountStart = $VMLiveMount.Start
$VMLiveMountSnapshot = $VMLiveMount.SnapshotDate
$VMLiveMountStatusURL = $VMLiveMount.StatusURL
$VMLiveMountJobInstanceID = $VMLiveMount.JobInstanceID
###########################################
# Getting Job status
###########################################
$RubrikJobStatusCount = 0
DO
{
$RubrikJobStatusCount = $RubrikJobStatusCount + 10
# Getting status
Try 
{
$RubrikJobStatusResponse = Invoke-RestMethod -Method GET -Uri $VMLiveMountStatusURL -Headers $RubrikSessionHeader
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Setting status
$RubrikJobStatus = $RubrikJobStatusResponse.status
# Output to host
"$VMLiveMountCounter/$VMLiveMountCount VM:$VMLiveMountVMName LiveMountStatus:$RubrikJobStatus"
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($RubrikJobStatus -ne "SUCCEEDED")
{
sleep 10
}
# Getting the error if FAILED
IF ($RubrikJobStatus -eq "FAILED")
{
$VMLiveMountError = $RubrikJobStatusResponse.error
$VMLiveMountErrorMessage = $VMLiveMountError | Select -ExpandProperty message  | ConvertFrom-Json | Select -ExpandProperty message
}
ELSE
{
$VMLiveMountErrorMessage = $null
}
# Will run until it succeeds, fails, or hits MaxVMWaitTimeSeconds
} Until (($RubrikJobStatus -eq "SUCCEEDED") -OR ($RubrikJobStatus -eq "FAILED") -OR ($RubrikJobStatusCount -gt $MaxVMWaitTimeSeconds))
# If $RubrikJobStatusCount eq MaxVMWaitTimeSeconds forcing RubrikJobStatus to FAILED
IF ($RubrikJobStatusCount -gt $MaxVMWaitTimeSeconds)
{
$RubrikJobStatus = "FAILED"
$VMLiveMountErrorMessage = "TimedOut - Considering increasing MaxVMWaitTimeSeconds"
}
####################
# Getting Node Data
####################
$VMLiveMountNodeID = $RubrikJobStatusResponse.nodeId.Replace("cluster:::","")
$VMLiveMountNodeIP = $RubrikNodes | Where-Object {$_.id -eq $VMLiveMountNodeID} | Select -ExpandProperty ipAddress
####################
# Getting Event Data
####################
# Building URL to get event data
$VMLiveMountEventURL = $InternalURL + "event?event_type=Recovery&object_ids=VirtualMachine%3A%3A%3A" + $VMLiveMountVMIDShort + "&show_only_latest=true&filter_only_on_latest=true"
# Getting event
Try 
{
$VMLiveMountEventJSON = Invoke-RestMethod -Method GET -Uri $VMLiveMountEventURL -Headers $RubrikSessionHeader
$VMLiveMountEvent = $VMLiveMountEventJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Filtering to ensure matching live mount request
$VMLiveMountEventFiltered = $VMLiveMountEvent | Where-Object {$_.JobInstanceID -eq $VMLiveMountJobInstanceID}
# Getting event series ID
$VMLiveMountEventSeriesID = $VMLiveMountEventFiltered.eventSeriesID
####################
# Getting Event Series
####################
# Building URL to get event series data
$VMEventSeriesURL = $InternalURL + "event_series/" + $VMLiveMountEventSeriesID
# Getting event series
Try 
{
$VMEventSeries = Invoke-RestMethod -Method GET -Uri $VMEventSeriesURL -Headers $RubrikSessionHeader
$VMEventSeriesDetail = $VMEventSeries.eventDetailList
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Getting recovery steps
$VMLiveMountRecoverySteps = $VMEventSeries.total
$VMLiveMountRecoveryRetries = $VMEventSeries.numberOfRetries
# For Each Event Series Entry Adding to array
ForEach ($VMEvent in $VMEventSeriesDetail)
{
# Setting variables
$VMEventTimeString = $VMEvent.time
$VMEventStatus = $VMEvent.status
$VMEventInfo = $VMEvent.eventInfo
$VMEventMessage = $VMEventInfo | ConvertFrom-Json | Select -ExpandProperty message
$VMEventMessageID = $VMEventInfo | ConvertFrom-Json | Select -ExpandProperty id
# Splitting message ID into category and action
$VMEventeCategory = $VMEventMessageID.Split(".")[0]
$VMEventAction = $VMEventMessageID.Split(".")[1]
# Converting time
$VMEventTime = ([datetime]::ParseExact($VMEventTimeString,”ddd MMM dd HH:mm:ss UTC yyyy”,$null))
# Adding to array
$VMLiveMountTask = New-Object PSObject
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Time" -Value $VMEventTime
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "VM" -Value $VMLiveMountVMName
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Status" -Value $VMEventStatus
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Details" -Value $VMEventMessage
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Category" -Value $VMEventeCategory
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Action" -Value $VMEventAction
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Host" -Value $VMLiveMountHost
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VMLiveMountVMID
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "Node" -Value $VMLiveMountNodeID
$VMLiveMountTask | Add-Member -MemberType NoteProperty -Name "NodeIP" -Value $VMLiveMountNodeIP
$VMLiveMountTasks += $VMLiveMountTask
}
# Hostoutput
"--------------------------------------------"
###########################################
# Summarizing test result
###########################################
# Getting node
$VMLiveMountNodeID = $RubrikJobStatusResponse.nodeId
# Setting times
$VMLiveMountEndUNIX = $RubrikJobStatusResponse.endTime
$VMLiveMountEnd = Convert-UNIXTime -UNIXTime $VMLiveMountEndUNIX
# Getting timespan and runtimes
$VMLiveMountTimespan = New-TimeSpan -Start $VMLiveMountStart -End $VMLiveMountEnd
$VMLiveMountSeconds = $VMLiveMountTimespan | Select -ExpandProperty TotalSeconds
$VMLiveMountSeconds = [Math]::Round($VMLiveMountSeconds, 0)
$VMLiveMountMinutes = $VMLiveMountTimespan | Select -ExpandProperty TotalMinutes
$VMLiveMountMinutes = [Math]::Round($VMLiveMountMinutes, 2)
$VMLiveMountTime = "{0:g}" -f $VMLiveMountTimespan
# Adding info to table/array
$VMLiveMountResultsLine = New-Object PSObject
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Order" -Value $VMLiveMountOrder
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "VM" -Value $VMLiveMountVMName
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "vCenter" -Value $VMLiveMountvCenter
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Host" -Value $VMLiveMountHost
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "SourcevCenter" -Value $VMLiveMountSourcevCenter
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "SourceHost" -Value $VMLiveMountSourceHost
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value $VMliveMountSLADomain
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "LiveMountName" -Value $VMLiveMountTestName
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "SnapshotDate" -Value $VMLiveMountSnapshot
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "LiveMountRequest" -Value $VMOperationSuccess
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "LiveMountStatus" -Value $RubrikJobStatus
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Start" -Value $VMLiveMountStart
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "End" -Value $VMLiveMountEnd
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "RTO" -Value $VMLiveMountTime
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Seconds" -Value $VMLiveMountSeconds
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Minutes" -Value $VMLiveMountMinutes
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "BootDelay" -Value $VMLiveMountFailoverVMDelay
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VMLiveMountVMID
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Node" -Value $VMLiveMountNodeID
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "NodeIP" -Value $VMLiveMountNodeIP
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Details" -Value $VMLiveMountErrorMessage
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Retries" -Value $VMLiveMountRecoveryRetries
$VMLiveMountResultsLine | Add-Member -MemberType NoteProperty -Name "Steps" -Value $VMLiveMountRecoverySteps
$VMLiveMountResults += $VMLiveMountResultsLine
# End of per VM validation of live mount below
}
# End of per VM validation of live mount above
###################################################################
# Step 3: Summarizing result for log
###################################################################
"----------------------------------------------------------------------------------------"
" Step 3: Summarizing Test Result"
"----------------------------------------------------------------------------------------"
###########################################
# Building totals and setting result
###########################################
$TotalVMsTested = $VMLiveMountResults.Count
$TotalVMsSucceeded = $VMLiveMountResults | Where-Object {$_.LiveMountStatus -eq "SUCCEEDED"} | Measure-Object | Select -ExpandProperty Count
$TotalVMsFailed = $VMLiveMountResults | Where-Object {$_.LiveMountStatus -eq "FAILED"} | Measure-Object | Select -ExpandProperty Count
# Timings
$JobStart = $VMLiveMountResults | Sort-Object Start | Select -ExpandProperty Start -First 1
$JobEnd = $VMLiveMountResults | Sort-Object End | Select -ExpandProperty End -Last 1
$JobTimespan = New-TimeSpan -Start $JobStart -End $JobEnd 
$JobMinutes = $JobTimespan | Select -ExpandProperty TotalMinutes
$JobMinutes = [Math]::Round($JobMinutes, 2)
$JobTime = "{0:g}" -f $JobTimespan
# Steps
$JobTotalSteps = $VMLiveMountTasks | Measure | Select -ExpandProperty Count
$JobTotalStepsSuccess = $VMLiveMountTasks | Where-Object {$_.Status -ne "Failure"} | Measure | Select -ExpandProperty Count
$JobTotalStepsFail = $VMLiveMountTasks | Where-Object {$_.Status -eq "Failure"} | Measure | Select -ExpandProperty Count
# Average minutes per VM
$JobAverageMinutesPerVM = $JobMinutes / $TotalVMsTested
$JobAverageMinutesPerVM = [Math]::Round($JobAverageMinutesPerVM, 2)
# Total vCenters, hosts and nodes
$JobTotalvCenters = $VMLiveMountResults | Select vCenter -Unique | Measure | Select -ExpandProperty Count
$JobTotalHosts = $VMLiveMountResults | Select Host -Unique | Measure | Select -ExpandProperty Count
$JobTotalNodes = $VMLiveMountResults | Select Node -Unique | Measure | Select -ExpandProperty Count
# Average VMs per host
$JobAverageVMsPerHost = $TotalVMsTested / $JobTotalHosts
$JobAverageVMsPerHost = [Math]::Round($JobAverageVMsPerHost, 2)
# Max VMs per host
$JobMaxVMsOnHost = $VMLiveMountResults | Group-Object Host | Sort-Object Count -descending | Select -ExpandProperty Count -First 1
$JobMaxVMsOnHostName = $VMLiveMountResults | Group-Object Host | Sort-Object Count -descending | Select -ExpandProperty Name -First 1
# Average VMs per node
$JobAverageVMsPerNode = $TotalVMsTested / $JobTotalNodes
$JobAverageVMsPerNode = [Math]::Round($JobAverageVMsPerNode, 2)
# Longest RTO
$JobLongestRTO = $VMLiveMountResults | Sort-Object {[int]$_.Seconds} -Descending | Select -ExpandProperty RTO -First 1
$JobLongestRTOVM = $VMLiveMountResults | Sort-Object {[int]$_.Seconds} -Descending | Select -ExpandProperty VM -First 1
# Setting test result
IF ($TotalVMsTested -eq $TotalVMsSucceeded)
{
$TestResult = "PASS"
$HTMLFontColor =  $HTMLColorSuccess
}
ELSE
{
$TestResult = "FAIL"
$HTMLFontColor =  $HTMLColorFailure
}
# Overriding if no VMs were tested (I.E faulty CSV config)
IF ($TotalVMsTested -eq 0)
{
$TestResult = "FAIL"
$HTMLFontColor =  $HTMLColorFailure
}
# If you want to test failing the report to see the email in red, uncomment the below
# $TestResult = "Fail"
# $HTMLFontColor =  $HTMLColorFailure
# Results for log
"VMsTested: $TotalVMsTested
VMsPassed: $TotalVMsSucceeded
TestResult: $TestResult"
###########################################
# Exporting CSV if set to true
###########################################
IF ($ExportCSV -eq $TRUE)
{
# Creating CSV name
$RecoveryPlanResultCSV = $LogDirectory + "\Rubrik-RecoveryPlan-Result-" + $Now.ToString("yyyy-MM-dd") + "@" + $Now.ToString("HH-mm-ss") + ".csv"
$RecoveryPlanDetailCSV = $LogDirectory + "\Rubrik-RecoveryPlan-Detail-" + $Now.ToString("yyyy-MM-dd") + "@" + $Now.ToString("HH-mm-ss") + ".csv"
# Host output
"Exporting Result To CSVs: 
$RecoveryPlanResultCSV
$RecoveryPlanDetailCSV"
# Exporting CSV
$VMLiveMountResults | Sort-Object End | Export-CSV -Path $RecoveryPlanResultCSV -NoTypeInformation -Force
$VMLiveMountTasks | Sort-Object Time | Export-CSV -Path $RecoveryPlanDetailCSV -NoTypeInformation -Force
}
###################################################################
# Step 4: Sending emails if OnlyEmailOnFailures is True
###################################################################
IF (($EmailOnFailures -eq $TRUE) -and ($TestResult -eq "Fail") -or ($EmailOnAll -eq $True))
{
# Host output
"----------------------------------------------------------------------------------------"
" Step 4: Sending Email as EmailOnFailures equals TRUE and TestResult equals Fail, or if EmailOnAll equals TRUE"
"----------------------------------------------------------------------------------------"
####################################################################
# SMTP Body - HTML Email style settings
####################################################################
# Start of HTML structure
$HTMLStart = @"
<!DOCTYPE html>
<html>
<head>
<style>

a {
    color: black;
}

a:link {
    text-decoration: none;
    color: #696969;
}

table.table1 {
  border-collapse: collapse;
  width: 100%;
}
table.table1 th {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #696969;
  font-size:16px
}
table.table1 td {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

table.table2 {
  border-collapse: collapse;
  width: 100%;
}
table.table2 th {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #00B2A9;
  font-size:14px
}
table.table2 td {
  text-align: center;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

table.table3 {
  border-collapse: collapse;
  width: 100%;
}
table.table3 th {
  text-align: left;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  background-color: white;
  color: #00B2A9;
  font-size:14px
}
table.table3 td {
  text-align: left;
  padding: 8px;
  border-bottom: 1px solid #ddd;
  font-size:12px
}

</style>
</head>
<body>

<div style="overflow-x:auto;">

<table style="text-align: right; width: 100%;" border="0"; cellpadding="2"; cellspacing="2">
<tbody>
<tr>
<td style="vertical-align: top;"><img style="width: 350px; height: 95px;" alt="Logo" src="logo.png"><br>
<br>
</td>
</tr>
</tbody>
</table>

<br></br>
"@
# End of HTML structure
$HTMLEnd = @"
</div>

</body>
</html>
"@
##################################
# Creating HTML Summary table
##################################
$SystemDateTime = Get-Date
$RubrikClusterURL = "https://" + $RubrikCluster + "/web/bin/index.html#/dashboard"
$HTMLSummaryTable = @"
<table class="table1">
  <tr>
    <th>VM Failover Result: <font color=$HTMLFontColor>$TestResult</font></th>
  </tr>
  <tr>
    <th>Created $SystemDateTime using <a href=$RubrikClusterURL>$RubrikCluster</a></th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>RTO</th>
    <td>$JobTime</td>
    <th>VMs</th>
    <td>$TotalVMsTested</td>
    <th>Steps</th>
    <td>$JobTotalSteps</td>
    <th>MinsPerVM</th>
    <td>$JobAverageMinutesPerVM</td>
    <th>VMsPerNode</th>
    <td>$JobAverageVMsPerNode</td>
  </tr>
 <tr>
    <th>Start</th>
    <td>$JobStart</td>
    <th>Success</th>
    <td>$TotalVMsSucceeded</td>
    <th>Success</th>
    <td>$JobTotalStepsSuccess</td>
    <th>LongestVM</th>
    <td>$JobLongestRTOVM</td>
    <th>VMsPerHost</th>
    <td>$JobAverageVMsPerHost</td>
 </tr>
 <tr>
    <th>End</th>
    <td>$JobEnd</td>
    <th>Fail</th>
    <td>$TotalVMsFailed</td>
    <th>Fail</th>
    <td>$JobTotalStepsFail</td>
    <th>LongestRTO</th>
    <td>$JobLongestRTO</td>
    <th>MaxVMsOnHost</th>
    <td>$JobMaxVMsOnHost</td>
 </tr>
 </table>
 <br>
"@
##################################
# Creating Table 1 HTML structure
##################################
$HTMLTable1Start = @"
<table class="table1">
  <tr>
    <th>Recovery Plan</th>
  </tr>
</table>

<table class="table2">
  <tr>
    <th>#</th>
    <th>VM</th>
    <th>vCenter</th>
    <th>ESXiHost</th>
    <th>SLADomain</th>
    <th>Result</th>
    <th>RTO</th>
    <th>Start</th>
    <th>End</th>
    <th>Delay</th>
    <th>RubrikNode</th>
    <th>Snapshot</th>
    <th>Details</th>
  </tr>
"@
$HTMLTable1End = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
# Building email list by task to put the most important objects for viewing at the top
$ReportList = $VMLiveMountResults | Sort-Object End
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTable1Middle = $null
# Creating table row for each line
ForEach ($Row in $ReportList) 
{
# Setting values
$HTML1Order = $Row.Order
$HTML1VM = $Row.VM
$HTML1VMID = $Row.VMID
$HTML1SLADomain = $Row.SLADomain
$HTML1VMLiveMountName = $Row.LiveMountName
$HTML1vCenter = $Row.vCenter
$HTML1ESXiHost = $Row.Host
$HTML1NodeIP = $Row.NodeIP
$HTML1Snapshot = $Row.SnapshotDate
$HTML1Result = $Row.LiveMountStatus
$HTML1RTO = $Row.RTO
$HTML1Start = $Row.Start
$HTML1End = $Row.End
$HTML1BootDelay = $Row.BootDelay
$HTML1Details = $Row.Details
# Building URL for VM
$RubrikVMURL = "https://" + $RubrikCluster + "/web/bin/index.html#/object_details/vmware/" + $HTML1VMID
# Setting colors for failed VMs
IF ($HTML1Result -eq "SUCCEEDED") {$HTMLFontColor =  $HTMLColorSuccess}
IF ($HTML1Result -ne "SUCCEEDED") {$HTMLFontColor =  $HTMLColorFailure}
# Building HTML table row
$HTMLReportTable1Row = "
<tr>
    <td><font color=$HTMLFontColor>$HTML1Order</font></td>
	<td><a href=$RubrikVMURL><font color=$HTMLFontColor>$HTML1VM</font></a></td>
    <td><font color=$HTMLFontColor>$HTML1vCenter</font></td>
    <td><font color=$HTMLFontColor>$HTML1ESXiHost</font></td>
    <td><font color=$HTMLFontColor>$HTML1SLADomain</font></td>
    <td><font color=$HTMLFontColor>$HTML1Result</font></td> 
    <td><font color=$HTMLFontColor>$HTML1RTO</font></td>
    <td><font color=$HTMLFontColor>$HTML1Start</font></td>
    <td><font color=$HTMLFontColor>$HTML1End</font></td> 
    <td><font color=$HTMLFontColor>$HTML1BootDelay</font></td>  
    <td><font color=$HTMLFontColor>$HTML1NodeIP</font></td>  
    <td><font color=$HTMLFontColor>$HTML1Snapshot</font></td>
    <td><font color=$HTMLFontColor>$HTML1Details</font></td>
  </tr>
"
# Adding row to table
$HTMLReportTable1Middle += $HTMLReportTable1Row
}
##################################
# Putting Table 1 together
##################################
$HTMLTable1 = $HTMLTable1Start + $HTMLReportTable1Middle + $HTMLTable1End
##################################
# Creating Table 2 HTML structure
##################################
$HTMLTable2Start = @"
<table class="table1">
  <tr>
    <th>Detail</th>
  </tr>
</table>

<table class="table3">
  <tr>
    <th>Step</th>
    <th>Time</th>
    <th>Elapsed</th>
    <th>Category</th>
    <th>Action</th>
    <th>Description</th>
    <th>Result</th>
    <th>VM</th>
    <th>RubrikNode</th>
    <th>ESXiHost</th>
  </tr>
"@
$HTMLTable2End = @"
</table>
<br>
"@
##################################
# Creating the Report Email Body
##################################
$ReportList = $VMLiveMountTasks | Sort-Object Time
# Adding counters and timers
$HTML2StepCounter = 0
$HTML2TimeStart = $ReportList | Select -ExpandProperty Time -First 1
# Nulling out table, protects against issues with multiple runs in PowerShell ISE
$HTMLReportTable2Middle = $null
# Creating table row for each line
ForEach ($Row in $ReportList) 
{
# Incrementing step counter
$HTML2StepCounter ++
# Setting values
$HTML2Time = $Row.Time
$HTML2VM = $Row.VM
$HTML2VMID = $Row.VMID
$HTML2Host = $Row.Host
$HTML2Node = $Row.NodeIP
$HTML2Category = $Row.Category
$HTML2Action = $Row.Action
$HTML2Details = $Row.Details
$HTML2Status = $Row.Status
# Getting elapsed time
$HTML2TimeSpan = New-TimeSpan -Start $HTML2TimeStart -End $HTML2Time
$HTML2TimeElapsed = "{0:g}" -f $HTML2TimeSpan
# Building URL for VM
$RubrikVMURL = "https://" + $RubrikCluster + "/web/bin/index.html#/object_details/vmware/" + $HTML2VMID
# Setting colors for failed VMs
IF ($HTML2Status -ne "Failure") {$HTMLFontColor =  $HTMLColorSuccess}
IF ($HTML2Status -eq "Failure") {$HTMLFontColor =  $HTMLColorFailure}
# Building HTML table row
$HTMLReportTable2Row = "
<tr>
    <td><font color=$HTMLFontColor>$HTML2StepCounter</font></td>
    <td><font color=$HTMLFontColor>$HTML2Time</font></td>
    <td><font color=$HTMLFontColor>$HTML2TimeElapsed</font></td>
    <td><font color=$HTMLFontColor>$HTML2Category</font></td>
    <td><font color=$HTMLFontColor>$HTML2Action</font></td>
    <td><font color=$HTMLFontColor>$HTML2Details</font></td>
    <td><font color=$HTMLFontColor>$HTML2Status</font></td> 
    <td><a href=$RubrikVMURL><font color=$HTMLFontColor>$HTML2VM</font></a></td>
    <td><font color=$HTMLFontColor>$HTML2Node</font></td>
    <td><font color=$HTMLFontColor>$HTML2Host</font></td>
  </tr>
"
# Adding row to table
$HTMLReportTable2Middle += $HTMLReportTable2Row
}
##################################
# Putting Table 2 together
##################################
$HTMLTable2 = $HTMLTable2Start + $HTMLReportTable2Middle + $HTMLTable2End
##################################
# Building Email Attachments
##################################
$Attachments = "$RecoveryPlanResultCSV","$RecoveryPlanDetailCSV","$LogoFile"
##################################
# Creating and sending HTML Email
##################################
# Building HTML report:
$HTMLReport = $HTMLStart + $HTMLSummaryTable + $HTMLTable1 + $HTMLTable2 + $HTMLEnd
# Sending report
Try
{
$EmailSent = $TRUE
Send-MailMessage -To $EmailTo -From $EmailFrom -Subject $EmailSubject -BodyAsHtml -Body $HTMLReport -Attachments $Attachments -SmtpServer $EmailServer -Port 25
}
Catch
{
$EmailSent = $FALSE
$_.Exception.ToString()
$error[0] | Format-List -Force
}
# Host output
"SMTPServer: $EmailServer
EmailFrom: $EmailFrom
EmailTo: $EmailTo
EmailSent: $EmailSent"
# End of email section below
}
# End of email section above
###################################################################
# Step 5: Start Per VM UnMount Action here
###################################################################
"----------------------------------------------------------------------------------------"
" Step 5: UnMounting VMs"
"----------------------------------------------------------------------------------------"
###############################################
# Getting list of VM Live Mounts - For unmounting GET 
###############################################
$VMActiveLiveMountsURL = $v1BaseURL+"vmware/vm/snapshot/mount"
Try 
{
$VMActiveLiveMountsJSON = Invoke-RestMethod -Uri $VMActiveLiveMountsURL -Headers $RubrikSessionHeader -ContentType $Type
$VMActiveLiveMounts = $VMActiveLiveMountsJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Starting For Each VM Live Mount
###############################################
# Creating counters
$VMLiveMountCounter = 0
$VMLiveMountCount =  $VMActiveLiveMounts | Measure | Select -ExpandProperty Count
# Starting action per live mounted VM
ForEach ($VMActiveLiveMount in $VMActiveLiveMounts)
{
# Incrementing counter
$VMLiveMountCounter ++
###############################################
# Setting the variables for the current VM
###############################################
$VMActiveLiveMountID = $VMActiveLiveMount.id
$VMActiveLiveMountSourceID = $VMActiveLiveMount.vmId
# Checking Live Mount Source VM ID was requested by this script
$VMActiveLiveMountFound = $VMLiveMountResults | Where-Object {$_.VMID -eq $VMActiveLiveMountSourceID} | Select -First 1
$VMActiveLiveMountSourceVMName = $VMActiveLiveMountFound.VM
# Setting action
IF ($VMActiveLiveMountFound -ne $null)
{
$VMRemoveLiveMount = $TRUE
}
ELSE
{
$VMRemoveLiveMount = $FALSE
}
# Setting VM live mount name
$VMLiveMountName = $VMActiveLiveMountSourceVMName + $VMSuffix
###########################################
# Setting URL running DELETE to REST API
###########################################
$VMUnMountURL = $v1BaseURL+"vmware/vm/snapshot/mount/"+$VMActiveLiveMountID
###########################################
# POST to REST API URL with VMJSON
###########################################
# Only trying unmount if VMID found
IF ($VMActiveLiveMountID -ne $null)
{
# DELETE request
Try 
{
$VMUnMountRequest = Invoke-RestMethod -Method DELETE -Uri $VMUnMountURL -Headers $RubrikSessionHeader -ContentType $Type -TimeoutSec 100
$VMUnMountStatusURL = $VMUnMountRequest.links.href
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Waiting for VM unmount to finish before moving to the next
Do{
# Getting status
Try 
{
$VMUnMountStatusResponse = Invoke-RestMethod -Method GET -Uri $VMUnMountStatusURL -Headers $RubrikSessionHeader
$VMUnMountStatus = $VMUnMountStatusResponse.status
}
Catch 
{
# For unmount jobs an error on the API call means its finished succesfully, as they just dissapear and it says "Could not find VmwareMount with id=xxx"
$VMUnMountStatus = "SUCCEEDED"
}
# Output of result
"$VMLiveMountCounter/$VMLiveMountCount VM:$VMLiveMountName UnmountStatus:$VMUnMountStatus"
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($VMUnMountStatus -ne "SUCCEEDED")
{
sleep 10
}
# Waiting for job to finish
}
Until(($VMUnMountStatus -eq "SUCCEEDED") -or ($VMUnMountStatus -eq "FAILED")) 
# End of IF live mount exists below
}
ELSE
{
"Live Mount ID Not Found for VM: $VMLiveMountName"
$VMUnMountStatus= "FAILED"
}
"--------------------------------------------"
# End of per VM actions below
}
# End of per VM actions above
################################################
# Stopping logging
################################################
# Host output
"End of RecoveryPlan Script"
"--------------------------------------------"
# Stopping logging
Stop-Transcript
###############################################
# End of script
###############################################