﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script stops all the active live mounts for the VMs specified in the RecoveryPlanCSV
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - A CSV with the following fields: VMName,HostSelection,DisableNetwork,RemoveNetworkDevices,KeepMACAddress,RecoverTags,RubrikPowerOn,NextVMFailoverDelay
# - Example for a DR Test with no NIC = DemoApp1-VM01,192.168.2.21,FALSE,TRUE,FALSE,FALSE,TRUE,10
# - Example for a DR Test with NIC disconnected = DemoApp1-VM01,192.168.2.21,TRUE,FALSE,FALSE,FALSE,TRUE,10
# - Example for a DR Failover (live) = DemoApp1-VM01,192.168.2.21,FALSE,FALSE,FALSE,FALSE,TRUE,10
# - Valid options for HostSelection are RANDOM (uses the existing ESXi host of the VM if recovering to prod) or the name of the ESXi host as registred in the vCenter
# - Valid options for DisableNetwork,RemoveNetworkDevices,RubrikPowerOn are TRUE or FALSE
# - This script always fails over to the latest snapshot available
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "192.168.1.201"
$RecoveryPlanCSV = "C:\RubrikRecoveryPlanv5\RubrikRecoveryPlan.csv"
$LogDirectory = "C:\RubrikRecoveryPlanv5\Logs"
# VM suffix is added to the end of each VM name as its registered in the vCenter, set to $null if you just want the VM name as is, I.E for recovery with existing VM gone
$VMSuffix = "-FailoverTest"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Starting logging & importing the CSV
################################################
# Checking log path exists, if not trying to create it
$LogFilePathTest = Test-Path $LogDirectory
# Creating if false
IF ($LogFilePathTest -eq $False)
{
New-Item -Path $LogDirectory -ItemType "directory"
}
# Getting time
$Now = Get-Date
# Creating log file name
$Log = $LogDirectory + "\Rubrik-RecoveryPlanLog-Stop-" + $Now.ToString("yyyy-MM-dd") + "@" + $Now.ToString("HH-mm-ss") + ".log"
# Starting logging
Start-Transcript -Path $Log -NoClobber 
$RecoveryPlanVMs = Import-CSV $RecoveryPlanCSV
# Filtering CSV to remove any lines where VMName is empty (sometimes people leave half finished rows)
$RecoveryPlanVMs= $RecoveryPlanVMs | Where-Object {$_.VMName -ne $null}
# Counting VMs to test
$RecoveryPlanVMCount = $RecoveryPlanVMs.Count
###############################################
# Importing Rubrik credentials
###############################################
# Setting credential file
$RubrikCredentialsFile = $LogDirectory + "\RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
ELSE
{
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
}
# Setting credentials
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception and TLS 1.2 to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
################################################
# Building Rubrik API string & invoking REST API
################################################
$v1BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$RubrikSessionURL = $v1BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
###############################################
# Getting list of VMs
###############################################
$VMListURL = $v1BaseURL+"vmware/vm?limit=5000"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of VM Live Mounts - For unmounting GET 
###############################################
$VMActiveLiveMountsURL = $v1BaseURL+"vmware/vm/snapshot/mount"
Try 
{
$VMActiveLiveMountsJSON = Invoke-RestMethod -Uri $VMActiveLiveMountsURL -Headers $RubrikSessionHeader -ContentType $Type
$VMActiveLiveMounts = $VMActiveLiveMountsJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Filtering Mounts to get only those initiated in recovery plan
###############################################
$VMLiveMountsToRemove = @()
ForEach($VMActiveLiveMount in $VMActiveLiveMounts)
{
# Setting ID
$VMActiveLiveMountVMID = $VMActiveLiveMount.vmID
# Getting VM name
$VMActiveLiveMountName = $VMList | Where-Object {$_.id -eq $VMActiveLiveMountVMID} | Select -ExpandProperty name -First 1
# Checking if name is in recovery plan
$VMActiveLiveMountFound = $RecoveryPlanVMs | Where-Object {$_.VMName -eq $VMActiveLiveMountName}
# Adding to array if found
IF ($VMActiveLiveMountFound -ne $null)
{
$VMLiveMountsToRemove += $VMActiveLiveMount
}
}
# Counting
$VMLiveMountCount = $VMLiveMountsToRemove | Measure | Select -ExpandProperty Count
$VMLiveMountCounter = 0
###################################################################
# Start Per VM  UnMount Action here
###################################################################
"----------------------------------------------------------------------------------------"
"Step 1: Starting per VM RecoveryPlan Unmount Action"
"----------------------------------------------------------------------------------------"
"LiveMounts: $VMLiveMountCount
RubrikCluster: $RubrikCluster
RubrikUser: $RubrikUser
Starting Unmounts:
--------------------------------------------"
###################################################################
# Start Per VM  UnMount Action here
###################################################################
# Creating counters
$VMLiveMountCounter = 0
$VMLiveMountCount =  $VMActiveLiveMounts | Measure | Select -ExpandProperty Count
# Starting For Each
ForEach ($VMLiveMountToRemove in $VMLiveMountsToRemove)
{
# Incremeing counter
$VMLiveMountCounter ++
# Setting ID
$VMLiveMountID = $VMLiveMountToRemove.id
$VMLiveMountVMID = $VMLiveMountToRemove.vmId
# Getting name
$VMLiveMountName = $VMList | Where-Object {$_.id -eq $VMLiveMountVMID} | Select -ExpandProperty name -First 1
$VMLiveMountName = $VMLiveMountName + $VMSuffix
# Building URL
$VMUnMountURL = $v1BaseURL+"vmware/vm/snapshot/mount/"+$VMLiveMountID
# Unmounting VM
Try 
{
$VMUnMountRequest = Invoke-RestMethod -Method DELETE -Uri $VMUnMountURL -TimeoutSec 100 -Headers $RubrikSessionHeader -ContentType $Type
$VMUnMountStatusURL = $VMUnMountRequest.links.href
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Waiting for VM unmount to finish before moving to the next
Do{
# Getting status
Try 
{
$VMUnMountStatusResponse = Invoke-RestMethod -Method GET -Uri $VMUnMountStatusURL -Headers $RubrikSessionHeader
$VMUnMountStatus = $VMUnMountStatusResponse.status
}
Catch 
{
# For unmount jobs an error on the API call means its finished succesfully, as they just dissapear and it says "Could not find VmwareMount with id=xxx"
$VMUnMountStatus = "SUCCEEDED"
}
# Output of result
"$VMLiveMountCounter/$VMLiveMountCount VM:$VMLiveMountName UnmountStatus:$VMUnMountStatus"
# Waiting 10 seconds before trying again, but only if not succeeded
IF ($VMUnMountStatus -ne "SUCCEEDED")
{
sleep 10
}
# Waiting for job to finish
}
Until(($VMUnMountStatus -eq "SUCCEEDED") -or ($VMUnMountStatus -eq "FAILED"))
# Host output of unmount operation
"$VMLiveMountCounter/$VMLiveMountCount VM:$VMLiveMountName UnmountStatus:$VMUnMountStatus"
"--------------------------------------------"
# End of per VM actions below
}
# End of per VM actions above
#
################################################
# Stopping logging
################################################
# Host output
"End of RecoveryPlan Script"
"--------------------------------------------"
# Stopping logging
Stop-Transcript
###############################################
# End of script
###############################################