﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script creates a CSV containing every VMware VM that the specified Rubrik cluster can see in the required format for a recovery plan
# Edit the generated CSV to only contain the VMs you want to recover/test
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik cluster or EDGE appliance, network access to it and credentials to login
# - This script creates a CSV with each VM containing the following fields: VMName,HostSelection,DisableNetwork,RemoveNetworkDevices,KeepMACAddress,RecoverTags,RubrikPowerOn,NextVMFailoverDelay
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Cluster
################################################
$RubrikCluster = "rubrik.lab.local"
$RecoveryPlanCSV = "C:\RubrikRecoveryPlanv5\RubrikRecoveryPlanTest.csv"
# Configure default settings
$HostSelection = "RANDOM"
$DisableNetwork = "FALSE"
$RemoveNetworkDevices = "TRUE"
$KeepMACAddress = "FALSE"
$RecoverTags = "FALSE"
$RubrikPowerOn = "TRUE"
$NextVMFailoverDelay = 10
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
###############################################
# Importing Rubrik credentials
###############################################
# Setting credential file
$RubrikCredentialsFile = $LogDirectory + "\RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
ELSE
{
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
}
# Setting credentials
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
##################################
# Adding certificate exception and TLS 1.2 to prevent API errors
##################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
################################################
# Building Rubrik API string & invoking REST API
################################################
$v1BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$v2BaseURL = "https://" + $RubrikCluster + "/api/v1/"
$InternalURL = "https://" + $RubrikCluster + "/api/internal/"
$RubrikSessionURL = $v1BaseURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($RubrikUser+":"+$RubrikPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$RubrikSessionResponse = Invoke-RestMethod -Uri $RubrikSessionURL -Headers $Header -Method POST -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response
$RubrikSessionHeader = @{'Authorization' = "Bearer $($RubrikSessionResponse.token)"}
###############################################
# Getting list of VMs
###############################################
$VMListURL = $v1BaseURL+"vmware/vm?limit=5000&is_relic=false"
Try 
{
$VMListJSON = Invoke-RestMethod -Uri $VMListURL -Headers $RubrikSessionHeader -ContentType $Type
$VMList = $VMListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of Hosts
###############################################
$VMHostListURL = $v1BaseURL+"vmware/host"
Try 
{
$VMHostListJSON = Invoke-RestMethod -Uri $VMHostListURL -Headers $RubrikSessionHeader -ContentType $Type
$VMHostList = $VMHostListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# Getting list of vCenters
###############################################
$vCenterListURL = $v1BaseURL+"vmware/vcenter"
Try 
{
$vCenterListJSON = Invoke-RestMethod -Uri $vCenterListURL -Headers $RubrikSessionHeader -ContentType $Type
$vCenterList = $vCenterListJSON.data
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
###############################################
# For Each VM creating entry for CSV
###############################################
# Creating array
$RecoveryPlanVMs = @()
# Counters
$RecoveryPlanVMCount = $VMList | Measure | Select -ExpandProperty Count
$RecoveryPlanVMCounter = 0
# Output to host
"VMSFound: $RecoveryPlanVMCount 
--------------------------------------------"
# For each VM
ForEach ($VM in $VMList)
{
# Incrementing counter
$RecoveryPlanVMCounter ++
# Output to host
"AddingVM: $RecoveryPlanVMCounter/$RecoveryPlanVMCount"
# Setting variables
$VMName = $VM.name
# Adding to array
$RecoveryPlanVM = New-Object PSObject
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "VMName" -Value $VMName
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "HostSelection" -Value $HostSelection
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "DisableNetwork" -Value $DisableNetwork
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "RemoveNetworkDevices" -Value $RemoveNetworkDevices
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "KeepMACAddress" -Value $KeepMACAddress
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "RecoverTags" -Value $RecoverTags
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "RubrikPowerOn" -Value $RubrikPowerOn
$RecoveryPlanVM | Add-Member -MemberType NoteProperty -Name "NextVMFailoverDelay" -Value $NextVMFailoverDelay
$RecoveryPlanVMs += $RecoveryPlanVM
}
###############################################
# Creating Recovery Plan CSV
###############################################
"--------------------------------------------
CreatingCSV:
$RecoveryPlanCSV"
# Creating CSV
$RecoveryPlanVMs | Sort-Object VMName | Export-Csv $RecoveryPlanCSV -NoTypeInformation -Force
# Host output
"--------------------------------------------"
"End of Script"
###############################################
# End of script
###############################################