﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script exports a CSV containing each VM NIC and its assigned port group using vSphere 6.5 REST APIs
# Run the script, edit the exported CSV to contain the VMs you want to reconfigure, change the State and PortGroup fields as desired then run the import script
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - Tested with a VMware vCenter 6.5 server, required as this script leverages the REST APIs in 6.5 onwards
# - Script collects all the VM NICs settings by REST API and exports a CSV
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the vCenter
################################################
$vCenterServer = "hchlv2vcenter.lab.local"
$LogDirectory = "C:\RubrikRecoveryPlanv5\Logs"
$VMNICSCSV = "C:\RubrikRecoveryPlanv5\Tools\RubrikRecoveryPlanNetworking.csv"
# Optional filter by VMnames containing text, I.E only VMs with -TestFailover in thier name
$VMFilterNames = "FALSE"
$VMFilterString = "-TestFailover"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
###############################################
# Importing vCenter credentials
###############################################
# Setting credential file
$vCenterCredentialsFile = $LogDirectory + "\vCenterCredentials.xml"
# Testing if file exists
$vCenterCredentialsFileTest =  Test-Path $vCenterCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($vCenterCredentialsFileTest -eq $False)
{
$vCenterCredentials = Get-Credential -Message "Enter vCenter login credentials"
$vCenterCredentials | EXPORT-CLIXML $vCenterCredentialsFile -Force
}
ELSE
{
# Importing credentials
$vCenterCredentials = IMPORT-CLIXML $vCenterCredentialsFile
}
# Setting credentials
$vCenterUser = $vCenterCredentials.UserName
$vCenterPassword = $vCenterCredentials.GetNetworkCredential().Password
################################################
# Adding certificate exception to prevent API errors
################################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
################################################
# Building vCenter API string & invoking REST API
################################################
$vCenterBaseAuthURL = "https://" + $vCenterServer + "/rest/com/vmware/cis/"
$vCenterBaseURL = "https://" + $vCenterServer + "/rest/vcenter/"
$vCenterSessionURL = $vCenterBaseAuthURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($vCenterUser+":"+$vCenterPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$vCenterSessionResponse = Invoke-RestMethod -Method POST -Uri $vCenterSessionURL -Headers $Header  -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
# Extracting the session ID from the response
$vCenterSessionHeader = @{'vmware-api-session-id' = $vCenterSessionResponse.value}
###############################################
# Getting list ESXi Hosts
###############################################
$ESXiHostsURL = $vCenterBaseURL+"host"
Try 
{
$ESXiHostsJSON = Invoke-RestMethod -Method GET -Uri $ESXiHostsURL -TimeoutSec 100 -Headers $vCenterSessionHeader -ContentType $Type
$ESXiHosts = $ESXiHostsJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
###############################################
# Getting VMs on each ESXi Host
###############################################
# Creating array to store results
$VMList = @()
# For each ESXi host getting VMs to overcome max 1,000 VM limit on VM API
ForEach ($ESXiHost in $ESXiHosts)
{
# Setting host ID
$ESXiHostID = $ESXiHost.host
$ESXiHostName = $ESXiHost.name
# Getting VMs on the ESXi host
$ESXiHostVMsURL = $vCenterBaseURL+"vm?filter.hosts="+$ESXiHostID
Try 
{
$ESXiHostVMsJSON = Invoke-RestMethod -Method GET -Uri $ESXiHostVMsURL -TimeoutSec 100 -Headers $vCenterSessionHeader -ContentType $Type
$ESXiHostVMs = $ESXiHostVMsJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
###############################################
# Adding each VM to table
###############################################
ForEach ($ESXiHostVM in $ESXiHostVMs)
{
# Setting variables
$VMID = $ESXiHostVM.vm
$VMName = $ESXiHostVM.name
$VMPowerState = $ESXiHostVM.power_state
$VMCPUCount = $ESXiHostVM.cpu_count
$VMRAMMB = $ESXiHostVM.memory_size_MiB
# Adding to table/array
$VM = New-Object PSObject
$VM | Add-Member -MemberType NoteProperty -Name "VM" -Value "$VMName"
$VM | Add-Member -MemberType NoteProperty -Name "VMID" -Value "$VMID"
$VM | Add-Member -MemberType NoteProperty -Name "State" -Value "$VMPowerState"
$VM | Add-Member -MemberType NoteProperty -Name "CPUs" -Value "$VMCPUCount"
$VM | Add-Member -MemberType NoteProperty -Name "RAMMB" -Value "$VMRAMMB"
$VM | Add-Member -MemberType NoteProperty -Name "Host" -Value "$ESXiHostName"
$VM | Add-Member -MemberType NoteProperty -Name "HostID" -Value "$ESXiHostID"
$VMList += $VM
# End of for each VM on each ESXi host below
}
# End of for each VM on each ESXi host above
#
# End of for each ESXi host below
}
# End of for each ESXi host above
###############################################
# Applying VM Name filter if enabled
###############################################
IF ($VMFilterNames -eq "TRUE")
{
$VMList = $VMList | Where-Object {$_.name -match $VMFilterString}
}
###############################################
# Getting list of Port Groups
###############################################
$PortGroupListURL = $vCenterBaseURL+"network"
Try 
{
$PortGroupListJSON = Invoke-RestMethod -Method GET -Uri $PortGroupListURL -Headers $vCenterSessionHeader -ContentType $Type
$PortGroupList = $PortGroupListJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
# Counting port groups
$PortGroupCount = $PortGroupList.Count
###############################################
# For Each VM creating entry for CSV
###############################################
# Creating array
$VMNICs = @()
# Counters
$VMCount = $VMList | Measure | Select -ExpandProperty Count
$VMCounter = 0
# Output to host
"VMSFound: $VMCount
Adding VM NICs:
--------------------------------------------"
# For each VM
ForEach ($VM in $VMList)
{
# Incrementing counter
$VMCounter ++
###############################################
# Setting the variables for the current VM
###############################################
$VMName = $VM.VM
$VMID = $VM.VMID
###############################################
# Getting NICs to configure for the VM if set to reconfigure
###############################################
$VMNICListURL = $vCenterBaseURL+"vm/"+$VMID+"/hardware/ethernet"
Try 
{
$VMNICListJSON = Invoke-RestMethod -Method GET -Uri $VMNICListURL -Headers $vCenterSessionHeader -ContentType $Type
$VMNICList = $VMNICListJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
$VMNICListCount = $VMNICList.count
###############################################
# Performing GET For Each VM NIC
###############################################
# Counting
$VMNICCount = $VMNICList.Count
$VMNICCounter = 0
# For each VM NIC
ForEach ($VMNICListed in $VMNICList)
{
# Incrementing counter
$VMNICCounter ++
# Output to host
"$VMCounter/$VMCount VM:$VMName NIC:$VMNICCounter/$VMNICCount"
# Setting NIC ID
$VMNICID = $VMNICListed.nic
# Building NIC URL
$VMNICURL = $vCenterBaseURL+"vm/"+$VMID+"/hardware/ethernet/"+$VMNICID 
# Getting NIC settings
Try 
{
$VMNICSettingsJSON = Invoke-RestMethod -Method GET -Uri $VMNICURL -Headers $vCenterSessionHeader -ContentType $Type
$VMNICSettings = $VMNICSettingsJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
# Setting VM nic variables
$VMNICName = $VMNICSettings.label
$VMNICMACAddress = $VMNICSettings.mac_address
$VMNICState = $VMNICSettings.state
# Setting port group variables
$VMNICPortGroupSettings = $VMNICSettings.backing
$VMNICPortGroupNetworkName = $VMNICPortGroupSettings.network_name
$VMNICPortGroupNetworkType = $VMNICPortGroupSettings.type
$VMNICPortGroupNetworkID = $VMNICPortGroupSettings.network
# Adding to array
$VMNIC = New-Object PSObject
$VMNIC | Add-Member -MemberType NoteProperty -Name "VMName" -Value $VMName
$VMNIC | Add-Member -MemberType NoteProperty -Name "VMID" -Value $VMID
$VMNIC | Add-Member -MemberType NoteProperty -Name "VMNIC" -Value $VMNICName
$VMNIC | Add-Member -MemberType NoteProperty -Name "VMNICID" -Value $VMNICID
$VMNIC | Add-Member -MemberType NoteProperty -Name "MACAddress" -Value $VMNICMACAddress
$VMNIC | Add-Member -MemberType NoteProperty -Name "State" -Value $VMNICState
$VMNIC | Add-Member -MemberType NoteProperty -Name "PortGroup" -Value $VMNICPortGroupNetworkName
$VMNICs += $VMNIC
# End of Per NIC actions below
}
# End of Per NIC actions above
#
# Output to host to separate VMs
"--------------------------------------------"
# End of per VM actions below
}
# End of per VM actions above
#
###############################################
# Creating CSVs
###############################################
"--------------------------------------------
CreatingCSV:
$VMNICSCSV"
# Creating CSV
$VMNICs | Sort-Object VMName,VMNIC | Export-Csv $VMNICSCSV -NoTypeInformation -Force
# Output to host
"--------------------------------------------"
###############################################
# PortGroupList output to host
###############################################
"Port Groups Available:"
$PortGroupList | Format-Table -AutoSize
# Host output
"--------------------------------------------"
"End of Script"
"--------------------------------------------"
###############################################
# End of script
###############################################