﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script imports the CSV generated from the Export script, then reconfigures each VMNIC specified with the new State and Port Group (as it only changes connection state and port group)
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A VMware vCenter 6.5 server, required as this script leverages the REST APIs in 6.5 onwards
# - The CSV created by the ExportVMNetworking.ps1 script
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the vCenter
################################################
$vCenterServer = "hchlv2vcenter.lab.local"
$LogDirectory = "C:\RubrikRecoveryPlanv5\Logs"
$VMNICSCSV = "C:\RubrikRecoveryPlanv5\Tools\RubrikRecoveryPlanNetworking.csv"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Starting logging & importing the CSV
################################################
# Checking log path exists, if not trying to create it
$LogFilePathTest = Test-Path $LogDirectory
# Creating if false
IF ($LogFilePathTest -eq $False)
{
New-Item -Path $LogDirectory -ItemType "directory"
}
# Getting time
$Now = Get-Date
$Log = $LogDirectory + "\vSphere-ConfigureNetworking-" + $Now.ToString("yyyy-MM-dd") + "@" + $Now.ToString("HH-mm-ss") + ".log"
Start-Transcript -Path $Log -NoClobber
# CSV import 
$VMNICS = Import-CSV $VMNICSCSV
# Getting unique VMs from VMNIC list
$VMsToConfigure = $VMNICS | Sort-Object VMID -Unique
###############################################
# Importing vCenter credentials
###############################################
# Setting credential file
$vCenterCredentialsFile = $LogDirectory + "\vCenterCredentials.xml"
# Testing if file exists
$vCenterCredentialsFileTest =  Test-Path $vCenterCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($vCenterCredentialsFileTest -eq $False)
{
$vCenterCredentials = Get-Credential -Message "Enter vCenter login credentials"
$vCenterCredentials | EXPORT-CLIXML $vCenterCredentialsFile -Force
}
ELSE
{
# Importing credentials
$vCenterCredentials = IMPORT-CLIXML $vCenterCredentialsFile
}
# Setting credentials
$vCenterUser = $vCenterCredentials.UserName
$vCenterPassword = $vCenterCredentials.GetNetworkCredential().Password
################################################
# Adding certificate exception to prevent API errors
################################################
add-type @"
    using System.Net;
    using System.Security.Cryptography.X509Certificates;
    public class TrustAllCertsPolicy : ICertificatePolicy {
        public bool CheckValidationResult(
            ServicePoint srvPoint, X509Certificate certificate,
            WebRequest request, int certificateProblem) {
            return true;
        }
    }
"@
[System.Net.ServicePointManager]::CertificatePolicy = New-Object TrustAllCertsPolicy
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
################################################
# Building vCenter API string & invoking REST API
################################################
$vCenterBaseAuthURL = "https://" + $vCenterServer + "/rest/com/vmware/cis/"
$vCenterBaseURL = "https://" + $vCenterServer + "/rest/vcenter/"
$vCenterSessionURL = $vCenterBaseAuthURL + "session"
$Header = @{"Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($vCenterUser+":"+$vCenterPassword))}
$Type = "application/json"
# Authenticating with API
Try 
{
$vCenterSessionResponse = Invoke-RestMethod -Method POST -Uri $vCenterSessionURL -Headers $Header  -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
# Extracting the session ID from the response
$vCenterSessionHeader = @{'vmware-api-session-id' = $vCenterSessionResponse.value}
###############################################
# Getting list ESXi Hosts
###############################################
$ESXiHostsURL = $vCenterBaseURL+"host"
Try 
{
$ESXiHostsJSON = Invoke-RestMethod -Method GET -Uri $ESXiHostsURL -TimeoutSec 100 -Headers $vCenterSessionHeader -ContentType $Type
$ESXiHosts = $ESXiHostsJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
###############################################
# Getting VMs on each ESXi Host
###############################################
# Creating array to store results
$VMList = @()
# For each ESXi host getting VMs to overcome max 1,000 VM limit on VM API
ForEach ($ESXiHost in $ESXiHosts)
{
# Setting host ID
$ESXiHostID = $ESXiHost.host
$ESXiHostName = $ESXiHost.name
# Getting VMs on the ESXi host
$ESXiHostVMsURL = $vCenterBaseURL+"vm?filter.hosts="+$ESXiHostID
Try 
{
$ESXiHostVMsJSON = Invoke-RestMethod -Method GET -Uri $ESXiHostVMsURL -TimeoutSec 100 -Headers $vCenterSessionHeader -ContentType $Type
$ESXiHostVMs = $ESXiHostVMsJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
###############################################
# Adding each VM to table
###############################################
ForEach ($ESXiHostVM in $ESXiHostVMs)
{
# Setting variables
$VMID = $ESXiHostVM.vm
$VMName = $ESXiHostVM.name
$VMPowerState = $ESXiHostVM.power_state
$VMCPUCount = $ESXiHostVM.cpu_count
$VMRAMMB = $ESXiHostVM.memory_size_MiB
# Adding to table/array
$VM = New-Object PSObject
$VM | Add-Member -MemberType NoteProperty -Name "VM" -Value "$VMName"
$VM | Add-Member -MemberType NoteProperty -Name "VMID" -Value "$VMID"
$VM | Add-Member -MemberType NoteProperty -Name "State" -Value "$VMPowerState"
$VM | Add-Member -MemberType NoteProperty -Name "CPUs" -Value "$VMCPUCount"
$VM | Add-Member -MemberType NoteProperty -Name "RAMMB" -Value "$VMRAMMB"
$VM | Add-Member -MemberType NoteProperty -Name "Host" -Value "$ESXiHostName"
$VM | Add-Member -MemberType NoteProperty -Name "HostID" -Value "$ESXiHostID"
$VMList += $VM
# End of for each VM on each ESXi host below
}
# End of for each VM on each ESXi host above
#
# End of for each ESXi host below
}
# End of for each ESXi host above
###############################################
# Getting list of Port Groups
###############################################
$PortGroupListURL = $vCenterBaseURL+"network"
Try 
{
$PortGroupListJSON = Invoke-RestMethod -Method Get -Uri $PortGroupListURL -TimeoutSec 100 -Headers $vCenterSessionHeader -ContentType $Type
$PortGroupList = $PortGroupListJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
###################################################################
# Start Per VM Configure Action here
###################################################################
# Counters
$VMsToConfigureCount = $VMsToConfigure | Measure | Select -ExpandProperty Count
$VMsToConfigureCounter = 0
# Output to host
"--------------------------------------------
VMsToConfigure: $VMsToConfigureCount
Starting Re-Configuration:
--------------------------------------------"
# For Each VM
ForEach ($VMToConfigure in $VMsToConfigure)
{
# Incrementing counter
$VMsToConfigureCounter ++
# Setting the variables for the current VM
$VMName = $VMToConfigure.VMName
$VMID = $VMToConfigure.VMID
# Getting NICs to configre
$VMNICsToConfigure = $VMNICS | Where-Object {$_.VMID -eq $VMID} | Sort-Object VMNIC
# Output to host
"$VMsToConfigureCount/$VMsToConfigureCounter VM: $VMName"
###############################################
# Performing For Each VM NIC Action
###############################################
ForEach ($VMNICToConfigure in $VMNICsToConfigure)
{
# Setting variables
$VMNIC = $VMNICToConfigure.VMNIC
$VMNICID = $VMNICToConfigure.VMNICID
$VMNICState = $VMNICToConfigure.State
$VMNICPortGroup = $VMNICToConfigure.PortGroup
# Selecting Port Group ID and Type using name
$VMNICPortGroupID = $PortGroupList | Where-Object {$_.name -eq $VMNICPortGroup} | Select -ExpandProperty network
$VMNICPortGroupType = $PortGroupList | Where-Object {$_.name -eq $VMNICPortGroup} | Select -ExpandProperty type
# Output to host
"
VMNIC: $VMNIC
NewPortGroup: $VMNICPortGroup"
##########################
# Configuring Port Group
##########################
# Building NIC URL
$VMNICURL = $vCenterBaseURL+"vm/"+$VMID+"/hardware/ethernet/"+$VMNICID 
# Building JSON to reconfigure NIC
$VMNICJSON =
"{
  ""spec"":{
  ""backing"":{
    ""type"": ""$VMNICPortGroupType"",
    ""network"": ""$VMNICPortGroupID""
              }
           }
 }"
# Patching NIC with new settings
Try 
{
$VMNICReconfigure = Invoke-RestMethod -Method PATCH -Uri $VMNICURL -Headers $vCenterSessionHeader -Body $VMNICJSON -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
##########################
# Getting NIC State
##########################
# Building NIC URL
$VMNICURL = $vCenterBaseURL+"vm/"+$VMID+"/hardware/ethernet/"+$VMNICID 
# Getting NIC settings
Try 
{
$VMNICSettingsJSON = Invoke-RestMethod -Method GET -Uri $VMNICURL -Headers $vCenterSessionHeader -ContentType $Type
$VMNICSettings = $VMNICSettingsJSON.value
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
# Setting VM nic variables
$VMNICCurrentState = $VMNICSettings.state
# Output to host
"State: $VMNICCurrentState"
##########################
# Connecting NIC If Configured
##########################
IF (($VMNICState -eq "CONNECTED") -and ($VMNICCurrentState -ne "CONNECTED"))
{
# VM NIC is configured to be connected in the CSV, but currently isn't connected, connecting the NIC
"ConnectingNIC: $VMNIC"
# Building Connect NIC URL
$VMConnectNICURL = $vCenterBaseURL+"vm/"+$VMID+"/hardware/ethernet/"+$VMNICID+"/connect"
Try 
{
$VMConnectNICRequest = Invoke-RestMethod -Method POST -Uri $VMConnectNICURL -Headers $vCenterSessionHeader -ContentType $Type
}
Catch 
{
$ErrorMessage = $_.Exception; $ErrorMessage
}
}
##########################
# End of Per NIC actions below
##########################
}
# End of Per NIC actions above
#
# Output to host to separate VMs
"--------------------------------------------"
# End of per VM actions below
}
# End of per VM actions above
#
# Inserting space in log for readability
"End of RecoveryPlan Script"
"--------------------------------------------"
################################################
# Stopping logging
################################################
Stop-Transcript
###############################################
# End of script
###############################################