﻿########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse joshuastenhouse@gmail.com
################################################
# Description:
# This script shows you how to securely connect to Polaris to retrive Radar events and create a CSV
################################################ 
# Requirements:
# - Run PowerShell as administrator with command "Set-ExecutionPolcity unrestricted" on the host running the script
# - A Rubrik Polaris URL with local credentials (not sso)
# - Radar licensed and running
# - Windows or Linux, PowerShell 5.1+
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever 
# (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) 
# arising out of the use of or inability to use the sample scripts or documentation, even if the author has been advised of the possibility of such damages.
################################################
# Configure the variables below for the Rubrik Polaris Instance
################################################
# Your unique Polaris URL
$PolarisURL = "https://rubrik-se.my.rubrik.com"
# Script location (store credentials securely for headless runs) - Ensure you leave a slash on the end (\ for windows, / for linux)
$ScriptDirectory = "C:\RubrikRadarEventsv2\" 
# Create Csv, if not it opens the results using Out-Gridview
$CSVEnabled = $TRUE
# CSV location (store output) - Ensure you leave a slash on the end (\ for windows, / for linux)
$CSVDirectory = "C:\RubrikRadarEventsv2\Events\"
# Optional, disable or enable file collection and CSV export
$EnableFileDownload = $TRUE
# Number of hours to query back
$HoursToCapture = 24
# Only export Encrypted confidence high & suspicious files greater than 0 = highly likely an infection vs anomaly detection
$OnlyExportEncryptedConfidenceHigh = $TRUE
# Keep 1 CSV of suspicious files per object
$KeepCSVPerObject = $FALSE
# Add location and SLA domains field (off by default as it increases collection time, but useful for large NAS environments as you need the share name and object location isn't stored on the radar event API)
$AddLocationandSLADomainFields = $TRUE
################################################
# Bonus Recovery Plan Feature
################################################
# Enable recovery plan CSV creation per cluster (only includes VMware VMs with high confidence if the above TRUE)
$CreateRecoveryPlanCSVPerCluster = $FALSE
# Default recovery plan settings (not used if the above is false)
$HostSelection = "RANDOM"
$DisableNetwork = "FALSE"
$RemoveNetworkDevices = "FALSE"
$KeepMACAddress = "TRUE"
$RecoverTags = "TRUE"
$RubrikPowerOn = "TRUE"
$NextVMFailoverDelay = 0
# Optionally, add a datastore column for exports (rather than live mounts)
$AddDataStoreOptionToRecoveryPlan = $TRUE
# Configure detault datastore location
$DatastoreSelection = "RANDOM"
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
# Getting PowerShell version to handle time formats (my Windows 10 machine says it has a none existent version of PS for some reason)
$PSVersion = $PSVersionTable.values | Sort-Object Major -Desc | Where-Object {$_.Major -ne 10} | Select -ExpandProperty Major -First 1
################################################
# Creating CSV dir if doesn't exist
################################################
# Checking path exists, if not trying to create it
$CSVDirectoryTest = Test-Path $CSVDirectory
# Creating if false
IF ($CSVDirectoryTest -eq $False)
{
New-Item -Path $CSVDirectory -ItemType "directory"
}
################################################
# Creating the Convert-UNIXTime function
################################################
Function Convert-UNIXTime {
Param ($UNIXTime)

# Example: $PSTime = Convert-UNIXTime -UNIXTime $UNIXTime

# Step 1 - Removing characters and trimming snapshot string for conversion
$PSTimeStep1 = $UNIXTime.Replace("T"," ").Replace("Z"," ").TrimEnd()
# Step 2 - Removing last 4 characters
$PSTimeStep2 = $PSTimeStep1.Substring(0,$PSTimeStep1.Length-4)
# Step 3 - Converting string to PowerShell datetime object
$PSTimeStep3 = ([datetime]::ParseExact($PSTimeStep2,”yyyy-MM-dd HH:mm:ss”,$null))

# Returning Result
Return $PSTimeStep3
}
###############################################
# Importing Rubrik credentials - Windows
###############################################
IF (($IsLinux -eq $FALSE) -or ($IsLinux -eq $null))
{
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + $env:COMPUTERNAME + "-" +$env:UserName + "-RubrikCredentials.xml"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik Polaris login credentials"
$RubrikCredentials | EXPORT-CLIXML $RubrikCredentialsFile -Force
}
ELSE
{
# Importing credentials
$RubrikCredentials = IMPORT-CLIXML $RubrikCredentialsFile
}
# Setting credentials
$RubrikUser = $RubrikCredentials.UserName
$RubrikPassword = $RubrikCredentials.GetNetworkCredential().Password
}
ELSE
{
###############################################
# Importing Rubrik credentials - Linux
###############################################
# Setting credential file
$RubrikCredentialsFile = $ScriptDirectory + "RubrikCredentials.bin"
# Testing if file exists
$RubrikCredentialsFileTest =  Test-Path $RubrikCredentialsFile
# Creating encryption key (as you can't use CLIXML in Linux etc)
$Key = (1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24)
# IF doesn't exist, prompting and saving credentials
IF ($RubrikCredentialsFileTest -eq $False)
{
$RubrikCredentials = Get-Credential -Message "Enter Rubrik Polaris login credentials"
$RubrikCredentials.Username | Out-File $RubrikCredentialsFile -Force
$RubrikCredentials.Password | ConvertFrom-SecureString -Key $Key | Out-File $RubrikCredentialsFile -Append
}
# Importing credentials
$RubrikCredentialsImport = Get-Content $RubrikCredentialsFile
$RubrikUser = $RubrikCredentialsImport[0]
$RubrikPasswordStr = $RubrikCredentialsImport[1] | ConvertTo-SecureString -Key $Key 
$RubrikPassword = [System.Net.NetworkCredential]::new("",$RubrikPasswordStr).password 
}
################################################
# Getting times required
################################################
$SystemDateTime = Get-Date
$UTCDateTime = [System.DateTime]::UtcNow
# Getting time range
$TimeRange = $UTCDateTime.AddHours(-$HoursToCapture)
# Converting to UNIX time format
$TimeRangeUNIX = $TimeRange.ToString("yyyy-MM-ddTHH:mm:ss.000Z")
# Logging
Write-Host "----------------------------------
Rubrik Radar Event Collector 
ConnectingTo: $PolarisURL"
Sleep 2
################################################
# Polaris - Building URLs & Authenticating
################################################
$PolarisGraphqlURL = $PolarisURL + "/api/graphql"
$PolarisSessionURL = $PolarisURL + "/api/session"
# Creating Auth Header
$PolarisAuthHeader = @{
        'Content-Type' = 'application/json';
        'Accept' = 'application/json';
    }
# Creating Auth Body
$PolarisAuthBody = 
"{
  ""password"": ""$RubrikPassword"",
  ""username"": ""$RubrikUser""
}"
# Authenticating with API
Try 
{
$PolarisSessionResponse = Invoke-RestMethod -Uri $PolarisSessionURL -Headers $PolarisAuthHeader -Body $PolarisAuthBody -Method POST
}
Catch 
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Extracting the token from the JSON response & creating header
$PolarisSessionHeader = @{
        'Content-Type' = 'application/json';
        'Accept' = 'application/json';
        'Authorization' = $('Bearer '+ $PolarisSessionResponse.access_token);
    }
# Exiting with error if required
IF ($PolarisSessionResponse.access_token -eq $null)
{
Write-Host "Connection Failed - Check PolarisURL, Credentials & try again..."
Sleep 5
Exit
}
################################################
# Getting All Objects (if $AddLocationandSLADomainFields = $TRUE)
################################################
IF ($AddLocationandSLADomainFields -eq $TRUE)
{
Write-Host "----------------------------------
Getting All Polaris Connected Objects"
# Creating array for events
$PolarisObjectsList = @()
# Building GraphQL query
$PolarisObjectsGraphQL = @{"operationName" = "snappableConnection";

"variables" = @{
"first" = 1000
};

"query" = "query snappableConnection(`$after: String) {
  snappableConnection(after: `$after, first: 1000) {
    edges {
      node {
        archivalComplianceStatus
        archivalSnapshotLag
        archiveSnapshots
        archiveStorage
        awaitingFirstFull
        complianceStatus
        dataReduction
        fid
        id
        lastSnapshot
        latestArchivalSnapshot
        latestReplicationSnapshot
        localOnDemandSnapshots
        location
        localSnapshots
        logicalBytes
        logicalDataReduction
        missedSnapshots
        name
        objectType
        physicalBytes
        protectedOn
        protectionStatus
        pullTime
        replicaSnapshots
        replicaStorage
        replicationComplianceStatus
        slaDomain {
          id
          name
          version
        }
        replicationSnapshotLag
        totalSnapshots
        transferredBytes
        cluster {
          id
          name
        }
      }
    }
        pageInfo {
      endCursor
      hasNextPage
      hasPreviousPage
      __typename
    }
  }
}
"
}
# Pagination count
$PaginationCounterStart = 0
$PaginationCounterEnd = 1000
# Logging
# Write-Host "Objects:$PaginationCounterStart-$PaginationCounterEnd"
# Querying API
$PolarisObjectsResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Body $($PolarisObjectsGraphQL | ConvertTo-JSON -Depth 100) -Headers $PolarisSessionHeader
# Setting variable
$PolarisObjectsList += $PolarisObjectsResponse.data.snappableConnection.edges
# Getting all results from paginations
While ($PolarisObjectsResponse.data.snappableConnection.pageInfo.hasNextPage) 
{
$PaginationCounterStart = $PaginationCounterStart + 1000
$PaginationCounterEnd = $PaginationCounterEnd + 1000
# Logging
# Write-Host "Objects:$PaginationCounterStart-$PaginationCounterEnd"
# Getting next 1000 events
$PolarisObjectsGraphQL.variables.after = $PolarisObjectsResponse.data.snappableConnection.pageInfo.endCursor
$PolarisObjectsResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Body $($PolarisObjectsGraphQL | ConvertTo-JSON -Depth 100) -Headers $PolarisSessionHeader
$PolarisObjectsList += $PolarisObjectsResponse.data.snappableConnection.edges
}
# Selecting data
$PolarisObjectsList = $PolarisObjectsList.node
################################################
# Processing All Objects 
################################################
# Counting
$PolarisObjectsCount = $PolarisObjectsList | Measure | Select -ExpandProperty Count
# Logging
"TotalObjects: $PolarisObjectsCount
Processing, this may take a minute.."
# Creating array
$PolarisObjects = [System.Collections.ArrayList]@()
# Processing Objects
$PolarisObjectsCounter = 0
# For Each Object Getting Data
ForEach ($PolarisObject in $PolarisObjectsList)
{
$PolarisObjectsCounter ++
# Write-Host "ProcessingObject: $PolarisObjectsCounter/$PolarisObjectsCount"
# Setting variables
$ObjectID = $PolarisObject.id
$ObjectFID = $PolarisObject.fid
$ObjectName = $PolarisObject.name
$ObjectLocation = $PolarisObject.location
$ObjectType = $PolarisObject.objectType
$ObjectSLADomainInfo = $PolarisObject.slaDomain
$ObjectSLADomain = $ObjectSLADomainInfo.name
$ObjectSLADomainID = $ObjectSLADomainInfo.id
$ObjectTotalSnapshots = $PolarisObject.totalSnapshots
$ObjectLastSnapshot = $PolarisObject.lastSnapshot
$ObjectPendingFirstFull = $PolarisObject.awaitingFirstFull
$ObjectProtectionStatus = $PolarisObject.protectionStatus
$ObjectProtectedOn = $PolarisObject.protectedOn
$ObjectLastUpdated = $PolarisObject.pulltime
$ObjectClusterInfo = $PolarisObject.cluster
$ObjectClusterID = $ObjectClusterInfo.id
$ObjectClusterName = $ObjectClusterInfo.name
# Adding To Array
$Object = New-Object PSObject
$Object | Add-Member -MemberType NoteProperty -Name "Cluster" -Value $ObjectClusterName
$Object | Add-Member -MemberType NoteProperty -Name "Object" -Value $ObjectName
$Object | Add-Member -MemberType NoteProperty -Name "Type" -Value $ObjectType
$Object | Add-Member -MemberType NoteProperty -Name "Location" -Value $ObjectLocation
$Object | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value $ObjectSLADomain
$Object | Add-Member -MemberType NoteProperty -Name "ProtectionStatus" -Value $ObjectProtectionStatus
$Object | Add-Member -MemberType NoteProperty -Name "ProtectedOn" -Value $ObjectProtectedOn
# Snapshot info
$Object | Add-Member -MemberType NoteProperty -Name "TotalSnapshots" -Value $ObjectTotalSnapshots
$Object | Add-Member -MemberType NoteProperty -Name "LastSnapshot" -Value $ObjectLastSnapshot
$Object | Add-Member -MemberType NoteProperty -Name "PendingFirstFull" -Value $ObjectPendingFirstFull
$Object | Add-Member -MemberType NoteProperty -Name "LastUpdated" -Value $ObjectLastUpdated
# IDs
$Object | Add-Member -MemberType NoteProperty -Name "ClusterID" -Value $ObjectClusterID
$Object | Add-Member -MemberType NoteProperty -Name "ObjectID" -Value $ObjectID
$Object | Add-Member -MemberType NoteProperty -Name "ObjectFID" -Value $ObjectFID
$Object | Add-Member -MemberType NoteProperty -Name "SLADomainID" -Value $ObjectSLADomainID
# Adding
$PolarisObjects.Add($Object) | Out-Null
# End of for each object below
}
# End of for each object above
#
# End of bypass for IF ($AddLocationField -eq $TRUE) below
}
# End of bypass for IF IF ($AddLocationField -eq $TRUE) above
################################################
# Getting Radar Events 
################################################
# Logging
Write-Host "----------------------------------
Collecting $HoursToCapture Hours of Radar Events
From(UTC): $TimeRange
To(UTC): $UTCDateTime"
Sleep 5
# Creating array for events
$PolarisEventsList = @()
# Building GraphQL query
$PolarisEventsGraphQL = @{"operationName" = "EventSeriesListQuery";

"variables" = @{
"filters" = @{
    "lastActivityType" = "Anomaly","RadarAnalysis"
    "lastUpdatedGt" = "$TimeRangeUNIX"
  }
"first" = 1000
};

"query" = "query EventSeriesListQuery(`$after: String, `$filters: ActivitySeriesFilterInput, `$first: Int, `$sortBy: ActivitySeriesSortByEnum, `$sortOrder: SortOrderEnum) {
  activitySeriesConnection(after: `$after, first: `$first, filters: `$filters, sortBy: `$sortBy, sortOrder: `$sortOrder) {
    edges {
      node {
        ...EventSeriesFragment
        cluster {
          id
          name
        }
        activityConnection(first: 1) {
          nodes {
            id
            message
            __typename
            activityInfo
          }
          __typename
        }
        __typename
      }
      __typename
    }
    pageInfo {
      endCursor
      hasNextPage
      hasPreviousPage
      __typename
    }
    __typename
  }
}
fragment EventSeriesFragment on ActivitySeries {
  id
  fid
  activitySeriesId
  lastUpdated
  lastActivityType
  lastActivityStatus
  objectId
  objectName
  objectType
  severity
  progress
  isCancelable
  isPolarisEventSeries
  __typename
}"
}
# Pagination count
$PaginationCounterStart = 0
$PaginationCounterEnd = 1000
# Logging
# Write-Host "Events:$PaginationCounterStart-$PaginationCounterEnd"
# Converting to JSON
$PolarisEventsResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Body $($PolarisEventsGraphQL | ConvertTo-JSON -Depth 100) -Headers $PolarisSessionHeader
$PolarisEventsList += $PolarisEventsResponse.data.activitySeriesConnection.edges
# Getting all results from paginations
While ($PolarisEventsResponse.data.activitySeriesConnection.pageInfo.hasNextPage) 
{
$PaginationCounterStart = $PaginationCounterStart + 1000
$PaginationCounterEnd = $PaginationCounterEnd + 1000
# Logging
# Write-Host "Events:$PaginationCounterStart-$PaginationCounterEnd"
# Getting next 1000 events
$PolarisEventsGraphQL.variables.after = $PolarisEventsResponse.data.activitySeriesConnection.pageInfo.endCursor
$PolarisEventsResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Body $($PolarisEventsGraphQL | ConvertTo-JSON -Depth 100) -Headers $PolarisSessionHeader
$PolarisEventsList += $PolarisEventsResponse.data.activitySeriesConnection.edges
}
# Selecting data
$AllRadarEvents = $PolarisEventsList.node
# Filtering data
$RadarEvents = $AllRadarEvents | Where {$_.lastActivityType -eq "Anomaly"}
$RadarScans = $AllRadarEvents | Where {$_.lastActivityType -eq "RadarAnalysis"}
# Counting
$RadarEventsCount = $RadarEvents | Measure | Select -ExpandProperty Count
$RadarEventsCounter = 0
# Scanned
$RadarScansCount = $RadarScans | Measure | Select -ExpandProperty Count
$RadarScannedObjects = $RadarScans | Select ObjectId -Unique
$RadarScannedObjectCount = $RadarScannedObjects | Measure | Select -ExpandProperty Count
# Impacted
$RadarImpactedObjects = $RadarEvents | Select ObjectId -Unique
$RadarImpactedObjectsCount = $RadarImpactedObjects | Measure | Select -ExpandProperty Count
# Logging
Write-Host "----------------------------------
RadarScans: $RadarScansCount
ScannedObjects: $RadarScannedObjectCount
----------------------------------
RadarEventsFound: $RadarEventsCount
ImpactedObjects: $RadarImpactedObjectsCount
----------------------------------"
Sleep 5
################################################
# Processing Radar Events
################################################
# Creating array to store results
$RadarObjects = [System.Collections.ArrayList]@()
$RadarFiles = [System.Collections.ArrayList]@()
# For Each Getting info
ForEach ($Event in $RadarEvents)
{
# Logging
$RadarEventsCounter ++
Write-Host "ProcessingRadarEvent: $RadarEventsCounter/$RadarEventsCount"
# Setting variables
$EventID = $Event.fid
$EventObjectID = $Event.objectId
$EventObject = $Event.objectName
$EventObjectType = $Event.objectType
$EventSeverity = $Event.severity
$EventCreatedUNIX = $Event.lastUpdated
# Converting time
IF ($PSVersion -lt 6){$EventCreated = Convert-UNIXTime -UNIXTime $EventCreatedUNIX; $EventCreatedUNIX=$EventCreatedUNIX.Substring(0,$EventCreatedUNIX.Length-5)}ELSE{$EventCreated = $EventCreatedUNIX;$EventCreatedUNIX = $EventCreated.ToString("yyyy-MM-ddTHH:MM:ss.000Z")}
# Getting cluster settings
$ClusterInfo = $Event.cluster
$Cluster = $ClusterInfo.name
$ClusterID = $ClusterInfo.id
# Getting message
$EventInfo = $Event | Select -ExpandProperty activityConnection -First 1 | Select -ExpandProperty nodes 
$EventMessage = $EventInfo.message
# Getting detail
$EventDetail = $Event | Select -ExpandProperty activityConnection -First 1 | Select -ExpandProperty nodes | Select -ExpandProperty activityInfo | ConvertFrom-JSON
# Infected snapshot date and id
$InfectedSnapshotID = $EventDetail.SnapshotID # CDM snapshot ID
$InfectedSnapshotFID = $EventDetail.SnapshotFID # Polaris snapshot ID
$InfectedSnapshotDateEPOCH = $EventDetail.SnapshotDate
$InfectedSnapshotDate = (Get-Date -Date "01/01/1970").AddMilliseconds($InfectedSnapshotDateEPOCH)
$InfectedSnapshotTimeStamp = $InfectedSnapshotDate.ToString("yyyy-MM-ddTHH:mm:ss")
# Snapshot results
$AnomalyConfidence = $EventDetail.AnomalyConfidence
$EncryptionConfidence = $EventDetail.EncryptionConfidence
$FilesAdded = $EventDetail.FilesAdded
$FilesModified = $EventDetail.FilesModified
$FileDeleted = $EventDetail.FilesDeleted
$FilesAddedSuspicious = $EventDetail.SuspiciousFilesAdded
# Getting Snappable Polaris IDs
$SnappableFID = $EventDetail.SnappableFID
$SnappableID = $EventDetail.SnappableID
# Getting object location and SLA domain if enabled
IF ($AddLocationandSLADomainFields -eq $TRUE)
{
$EventObjectInfo = $PolarisObjects | Where {$_.ObjectFID -eq $SnappableFID}
$EventObjectLocation = $EventObjectInfo.Location 
$EventObjectSLADomain = $EventObjectInfo.SLADomain
$EventObjectSLADomainID = $EventObjectInfo.SLADomainID
}
############################
# Getting Snapshot Detail
############################
# Building GraphQL query
$ObjectSnapshotsGraphQL = @{"query" = "query RubrikPolarisSDKRequest{
    snapshotOfASnappableConnection
    ( 
        snappableId: ""$SnappableFID"",
        sortOrder: Desc
    ){
        edges {
            node {
                id
                expirationDate
                date
                isOnDemandSnapshot
                snappableId
                isIndexed
                indexingAttempts
                ... on PolarisSnapshot{
                    slaDomain {
                        name
                        id
                    }
                }
                 ... on CdmSnapshot {
                    id
                    isAnomaly
                    date
                    slaDomain {
                        id
                        name
                    }
                isIndexed
                }
                
            }
        }
    }
}"
}
# Converting to JSON
$ObjectSnapshotsJSON = $ObjectSnapshotsGraphQL | ConvertTo-JSON -Depth 100
# Querying API
Try
{
$ObjectSnapshotsResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Headers $PolarisSessionHeader -Body $ObjectSnapshotsJSON
$ObjectSnapshots = $ObjectSnapshotsResponse.data.snapshotOfASnappableConnection.edges.node
}
Catch
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Counting
$SnapshotCount = $ObjectSnapshots | Measure | Select -ExpandProperty Count
# Counting how many snapshots until infected one found 
$SnapshotsToSkip = 0
ForEach($Snap in $ObjectSnapshots){$SnapID = $Snap.id; $SnapshotsToSkip++;IF($SnapID -match $InfectedSnapshotFID){Break}}
# Reducing snapshots to those before most recent anomaly
$RecoverySnapshots = $ObjectSnapshots | Select -Skip $SnapshotsToSkip
# Selecting most recent snapshot without anomaly
$RecoverySnapshot = $RecoverySnapshots | Where {$_.isAnomaly -eq $False} | Select -First 1
$RecoverySnapshotDateUNIX = $RecoverySnapshot.date
$RecoverySnapshotFID = $RecoverySnapshot.id # The Polaris ID, not in CDM
# Converting time
IF ($PSVersion -lt 6){$RecoverySnapshotDate = Convert-UNIXTime -UNIXTime $RecoverySnapshotDateUNIX; $RecoverySnapshotTimeStamp=$RecoverySnapshotDateUNIX.Substring(0,$RecoverySnapshotDateUNIX.Length-5)}ELSE{$RecoverySnapshotDate = $RecoverySnapshotDateUNIX;$RecoverySnapshotTimeStamp = $RecoverySnapshotDate.ToString("yyyy-MM-ddTHH:mm:ss")}
# Converting FID to ID - so it can be found in CDM
$RecoverySnapshotGraphQL = @{"query" = "query MyQuery {
  snapshot(snapshotFid: ""$RecoverySnapshotFID"") {
    id
    cdmId
  }
}"
}
# Converting to JSON
$RecoverySnapshotJSON = $RecoverySnapshotGraphQL | ConvertTo-JSON -Depth 100
# Querying API
Try
{
$RecoverySnapshotResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Headers $PolarisSessionHeader -Body $RecoverySnapshotJSON
$RecoverySnapshotInfo = $RecoverySnapshotResponse.data.snapshot
}
Catch
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Getting ID
$RecoverySnapshotID = $RecoverySnapshotInfo.cdmId
############################
# Getting CSV Link
############################
IF ($EnableFileDownload -eq $TRUE)
{
$RadarCSVLinkGraphQL = @{"query" = "query InvestigationCsvDownloadLinkQuery{
  investigationCsvDownloadLink(
    snappableIdNotFid: ""$SnappableID""
    clusterUuid: ""$ClusterID""
    snapshotId: ""$InfectedSnapshotID""
  ) {
    downloadLink
  }
}"
}
# Converting to JSON
$RadarCSVLinkJSON = $RadarCSVLinkGraphQL | ConvertTo-JSON -Depth 100
# Querying API
Try
{
$RadarCSVLinkResponse = Invoke-RestMethod -Method POST -Uri $PolarisGraphqlURL -Headers $PolarisSessionHeader -Body $RadarCSVLinkJSON
$RadarCSVLink = $RadarCSVLinkResponse.data.investigationCsvDownloadLink.downloadLink
}
Catch
{
# $ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
############################
# Downloading & Importing the CSV
############################
# Building filename
$CSVDownloadFile = $CSVDirectory + "RadarFiles-" + $EventObject + "-" + $InfectedSnapshotDate.ToString("yyyy-MM-dd") + "@" + $InfectedSnapshotDate.ToString("HH-mm-ss") + ".csv"
# Downloading CSV file
Try
{
IF ($RadarCSVLink -ne $null){Invoke-RestMethod -Method GET -Uri $RadarCSVLink -OutFile $CSVDownloadFile}
}
Catch
{
$ErrorMessage = $_.ErrorDetails; "ERROR: $ErrorMessage"
}
# Testing if file exists
$CSVTest = Test-Path $CSVDownloadFile
}
ELSE
{
$CSVTest = $FALSE	
}
IF ($CSVTest -eq $TRUE)
{
# Importing the CSV
$CSVImport = Import-Csv $CSVDownloadFile
############################
# Adding CSV Files To Array
############################
ForEach ($File in $CSVImport)
{
# Setting variables
$FilePath = $File.path
$FileName = $File.file_name
$FileDirectory = $File.directory
$FileChange = $File.change
$FileIsSuspicious = $File.is_suspicious
$FileSizeBytes = $File.size
$FileBytesChanged = $File.bytes_changed
$FileModifiedEPOCH = $File.mtime
# Detecting if the file is on a windows machine
$WindowsFile = $FilePath |  Select-String -Pattern "[A-Z]+:"
IF ($WindowsFile -ne $null)
{
# If a drive letter is found, removing trailing / and inverting
$FilePath = $FilePath.substring(1) 
$FilePath = $FilePath.Replace("/","\")
$FileDirectory = $FileDirectory.substring(1) 
$FileDirectory = $FileDirectory.Replace("/","\")
}
# Converting time
$FileModified = (Get-Date 01.01.1970)+([System.TimeSpan]::fromseconds($FileModifiedEPOCH))
# Adding to array
$RadarFile = New-Object PSObject
$RadarFile | Add-Member -MemberType NoteProperty -Name "RubrikCluster" -Value $Cluster
$RadarFile | Add-Member -MemberType NoteProperty -Name "Object" -Value $EventObject
$RadarFile | Add-Member -MemberType NoteProperty -Name "Type" -Value $EventObjectType
IF($AddLocationandSLADomainFields -eq $TRUE){$RadarFile | Add-Member -MemberType NoteProperty -Name "Location" -Value $EventObjectLocation}
$RadarFile | Add-Member -MemberType NoteProperty -Name "File" -Value $FilePath
$RadarFile | Add-Member -MemberType NoteProperty -Name "FileName" -Value $FileName
$RadarFile | Add-Member -MemberType NoteProperty -Name "Directory" -Value $FileDirectory
$RadarFile | Add-Member -MemberType NoteProperty -Name "Change" -Value $FileChange
$RadarFile | Add-Member -MemberType NoteProperty -Name "IsSuspicious" -Value $FileIsSuspicious
$RadarFile | Add-Member -MemberType NoteProperty -Name "SizeBytes" -Value $FileSizeBytes
$RadarFile | Add-Member -MemberType NoteProperty -Name "BytesChanged" -Value $FileSizeBytes
$RadarFile | Add-Member -MemberType NoteProperty -Name "LastModified" -Value $FileModified
$RadarFile | Add-Member -MemberType NoteProperty -Name "Scanned" -Value $EventCreated
$RadarFile | Add-Member -MemberType NoteProperty -Name "EventID" -Value $EventID
$RadarFile | Add-Member -MemberType NoteProperty -Name "RubrikClusterID" -Value $ClusterID
$RadarFile | Add-Member -MemberType NoteProperty -Name "ObjectID" -Value $EventObjectID
$RadarFile | Add-Member -MemberType NoteProperty -Name "InfectedSnapshotUTC" -Value $InfectedSnapshotDate
$RadarFile | Add-Member -MemberType NoteProperty -Name "InfectedSnapshotID" -Value $InfectedSnapshotID
$RadarFile | Add-Member -MemberType NoteProperty -Name "RecoverySnapshotUTC" -Value $RecoverySnapshotDate
$RadarFile | Add-Member -MemberType NoteProperty -Name "RecoverySnapshotID" -Value $RecoverySnapshotID
$RadarFiles.Add($RadarFile) | Out-Null
}
# End of bypass for no CSV downloaded below
}
# End of bypass for no CSV downloaded above
############################
# Deleting CSV per object if $KeepCSVPerObject = $FALSE
############################
IF ($KeepCSVPerObject -eq $FALSE)
{
$CSVTest = Test-Path $CSVDownloadFile
# Deleting if true
IF ($CSVTest -eq $True)
{
Remove-Item $CSVDownloadFile
}
}
############################
# Adding Event To Array
############################
$RadarObject = New-Object PSObject
$RadarObject | Add-Member -MemberType NoteProperty -Name "EventTime" -Value $EventCreated
$RadarObject | Add-Member -MemberType NoteProperty -Name "RubrikCluster" -Value $Cluster
# Object info
$RadarObject | Add-Member -MemberType NoteProperty -Name "Object" -Value $EventObject
$RadarObject | Add-Member -MemberType NoteProperty -Name "Type" -Value $EventObjectType
# Additional info
IF($AddLocationandSLADomainFields -eq $TRUE)
{
$RadarObject | Add-Member -MemberType NoteProperty -Name "Location" -Value $EventObjectLocation
$RadarObject | Add-Member -MemberType NoteProperty -Name "SLADomain" -Value $EventObjectSLADomain
}
$RadarObject | Add-Member -MemberType NoteProperty -Name "TotalSnapshots" -Value $SnapshotCount
# Summary of alert
$RadarObject | Add-Member -MemberType NoteProperty -Name "Anomaly" -Value $AnomalyConfidence
$RadarObject | Add-Member -MemberType NoteProperty -Name "Encryption" -Value $EncryptionConfidence
# Files
$RadarObject | Add-Member -MemberType NoteProperty -Name "FilesAdded" -Value $FilesAdded
$RadarObject | Add-Member -MemberType NoteProperty -Name "FilesModified" -Value $FilesModified
$RadarObject | Add-Member -MemberType NoteProperty -Name "FileDeleted" -Value $FileDeleted
$RadarObject | Add-Member -MemberType NoteProperty -Name "FilesEncrypted" -Value $FilesAddedSuspicious
# Infected snapshot
$RadarObject | Add-Member -MemberType NoteProperty -Name "InfectedSnapshotUTC" -Value $InfectedSnapshotDate
$RadarObject | Add-Member -MemberType NoteProperty -Name "InfectedSnapshotTimeStamp" -Value $InfectedSnapshotTimeStamp
$RadarObject | Add-Member -MemberType NoteProperty -Name "InfectedSnapshotID" -Value $InfectedSnapshotID
# Recovery snapshot
$RadarObject | Add-Member -MemberType NoteProperty -Name "RecoverySnapshotUTC" -Value $RecoverySnapshotDate
$RadarObject | Add-Member -MemberType NoteProperty -Name "RecoverySnapshotTimeStamp" -Value $RecoverySnapshotTimeStamp
$RadarObject | Add-Member -MemberType NoteProperty -Name "RecoverySnapshotID" -Value $RecoverySnapshotID
# Other useful info
$RadarObject | Add-Member -MemberType NoteProperty -Name "RubrikClusterID" -Value $ClusterID
$RadarObject | Add-Member -MemberType NoteProperty -Name "ObjectID" -Value $EventObjectID
$RadarObject | Add-Member -MemberType NoteProperty -Name "EventID" -Value $EventID
# Adding recovery plan settings (if enabled)
IF ($CreateRecoveryPlanCSVPerCluster -eq $TRUE)
{
$RadarObject | Add-Member -MemberType NoteProperty -Name "HostSelection" -Value $HostSelection
# Adding export option (if enabled)
IF ($AddDataStoreOptionToRecoveryPlan -eq $TRUE)
{
$RadarObject | Add-Member -MemberType NoteProperty -Name "DatastoreSelection" -Value $DatastoreSelection
}
$RadarObject | Add-Member -MemberType NoteProperty -Name "DisableNetwork" -Value $DisableNetwork
$RadarObject | Add-Member -MemberType NoteProperty -Name "RemoveNetworkDevices" -Value $RemoveNetworkDevices
$RadarObject | Add-Member -MemberType NoteProperty -Name "KeepMACAddress" -Value $KeepMACAddress
$RadarObject | Add-Member -MemberType NoteProperty -Name "RecoverTags" -Value $RecoverTags
$RadarObject | Add-Member -MemberType NoteProperty -Name "RubrikPowerOn" -Value $RubrikPowerOn
$RadarObject | Add-Member -MemberType NoteProperty -Name "NextVMFailoverDelay" -Value $NextVMFailoverDelay
}
# Adding
$RadarObjects.Add($RadarObject) | Out-Null
############################
# End of for each radar event below
############################
}
# End of for each radar event above
################################################
# Only selecting High confidence encryption events if filter enabled
################################################
IF ($OnlyExportEncryptedConfidenceHigh -eq $TRUE)
{
$RadarObjects = $RadarObjects | Where {(($_.Encryption -eq "High") -and ($_.FilesEncrypted -gt 0))}
$RadarObjectsCount = $RadarObjects | Measure | Select -ExpandProperty Count
# Logging
Write-Host "----------------------------------
HighEncryptionConfidenceEvents: $RadarObjectsCount"
}
################################################
# Summarizing File Counts
################################################
IF ($RadarFiles -ne $null)
{
$SuspiciousFiles = $RadarFiles | Where-Object {$_.IsSuspicious -eq $TRUE} 
$SuspiciousFilesCount = $SuspiciousFiles | Measure | Select -ExpandProperty Count
# Getting first attack range
$FirstEncryptionEvent = $SuspiciousFiles | Sort-Object LastModified | Select -First 1
$FirstEncryptionLastModified = $FirstEncryptionEvent | Select -ExpandProperty LastModified
$FirstEncryptionObject = $FirstEncryptionEvent.Object
$FirstEncryptionObjectType = $FirstEncryptionEvent.Type
# Getting last attack range
$LastEncryptionEvent = $SuspiciousFiles | Sort-Object LastModified | Select -Last 1
$LastEncryptionLastModified = $LastEncryptionEvent | Select -ExpandProperty LastModified
$LastEncryptionObject = $LastEncryptionEvent.Object
$LastEncryptionObjectType = $LastEncryptionEvent.Type
# Logging
Write-Host "----------------------------------
TotalSuspiciousFiles: $SuspiciousFilesCount
----------------------------------
Encryption Start
Time: $FirstEncryptionLastModified
Object: $FirstEncryptionObject
ObjectType: $FirstEncryptionObjectType
----------------------------------
Encryption End
Time: $LastEncryptionLastModified
Object: $LastEncryptionObject
ObjectType: $LastEncryptionObjectType"
# Counts per object
$UniqueObjects = $SuspiciousFiles | Select -ExpandProperty ObjectID
}
################################################
# Opening Results if CSVEnabled is False
################################################
IF ($CSVEnabled -eq $FALSE)
{
# Logging
Write-Host "----------------------------------
OpeningGridView..
----------------------------------"
# Opening
$RadarObjects | Out-Gridview
}
ELSE
{
################################################
# Exporting ALL Radar Events 
################################################
# Creating CSV file name
$CSVExportFile1 = $CSVDirectory + "RadarAlerts-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
$CSVExportFile2 = $CSVDirectory + "RadarFiles-" + $SystemDateTime.ToString("yyyy-MM-dd") + "@" + $SystemDateTime.ToString("HH-mm-ss") + ".csv"
# Logging
Write-Host "----------------------------------
CreatingCSV: $CSVExportFile1"
IF ($EnableFileDownload -eq $TRUE){Write-Host "CreatingCSV: $CSVExportFile2"}
Write-Host "----------------------------------"
# Exporting CSV
IF ($RadarObjects -ne $null){$RadarObjects | Export-Csv $CSVExportFile1 -NoTypeInformation}
IF (($RadarFiles -ne $null) -and ($EnableFileDownload -eq $TRUE)){$RadarFiles | Export-Csv $CSVExportFile2 -NoTypeInformation}
################################################
# Exporting Radar Events Per Cluster - If enabled
################################################
IF ($CreateRecoveryPlanCSVPerCluster -eq $TRUE)
{
# Logging
Write-Host "----------------------------------
CreatingRecoveryPlans"
# Getting unique clusters
$RadarEventClusterIDs = $RadarObjects | Select -ExpandProperty ClusterID -Unique
# Creating CSV for each cluster
ForEach ($RadarEventClusterID in $RadarEventClusterIDs)
{
# Getting data
$RadarClusterEvents = $RadarObjects | Where {(($_.ClusterID -eq $RadarEventClusterID) -and ($_.Type -eq "VmwareVm"))} 
$RadarClusterName = $RadarClusterEvents | Select -ExpandProperty Cluster -First 1
# Creating CSV file name
$CSVExportFile = $CSVDirectory + "RecoveryPlan-" + $RadarClusterName + ".csv"
# Logging
Write-Host "CreatingCSV: $CSVExportFile"
# Exporting CSV if not null
IF ($RadarClusterEvents -ne $null){$RadarClusterEvents | Select @{N=’VM’; E={$_.Object}},@{N=’VMID’; E={$_.ObjectID}},InfectedSnapshotUTC,InfectedSnapshotID,RecoverySnapshotUTC,RecoverySnapshotID,HostSelection,DisableNetwork,RemoveNetworkDevices,KeepMACAddress,RecoverTags,RubrikPowerOn,NextVMFailoverDelay | Export-Csv $CSVExportFile -NoTypeInformation -Force}
}
}
# End of bypass for CSVEnabled is false below
}
# End of bypass for CSVEnabled is false above
################################################
# End of script
################################################