########################################################################################################################
# Start of the script - Description, Requirements & Legal Disclaimer
########################################################################################################################
# Written by: Joshua Stenhouse - joshuastenhouse@gmail.com
################################################
# Description:
# This script uses the defined CSV list of Azure VM sizes to build a list of recommended VM instance sizes for each VM in your vCenter while also giving useful pricing info for ZVR
################################################
# Requirements:
# - CSV with the following columns Cloud, Size, CPU, RAM, Disks, NICs, LocalHDD, WindowsPricePH, LinuxPricePH
# - vCenter IP, username and password with the ability to run get-VM
# - Install VMware PowerCLI (any version above your vCenter version)
# - Write permission to CSV output directory 
################################################
# Legal Disclaimer:
# This script is written by Joshua Stenhouse is not supported under any Zerto support program or service. 
# All scripts are provided AS IS without warranty of any kind. 
# The author and Zerto further disclaims all implied warranties including, without limitation, any implied warranties of merchantability or of fitness for a particular purpose. 
# The entire risk arising out of the use or performance of the sample scripts and documentation remains with you. 
# In no event shall Zerto, its authors, or anyone else involved in the creation, production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation, damages for loss of business profits, business interruption, loss of business information, or other pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even if the author or Zerto has been advised of the possibility of such damages.
################################################
# Configure the import and export CSVs 
################################################
# Script directory
$ScriptDirectory = "C:\Users\joshu\OneDrive\Zerto\#Scripts\#2017Scripts\AzureVMSizingTool"
# Import CSV to use for VM instance sizing
$AzureVMSizingImportCSV = $ScriptDirectory + "\AzureVMSizingImport.CSV"
# Export CSVs for data generated by the script
$AzureVMArrayCSV = $ScriptDirectory + "\AzureVMSizingExport.csv"
$AzureVMSummaryCSV = $ScriptDirectory + "\AzureVMSizingExportSummary.csv"
# Recommended VMs per ZCA is max 200
$VMsPerZCA = 200
################################################
# Variables used for standard pricing, taken from US EAST
################################################
# Use https://azure.microsoft.com/en-us/pricing/calculator/?service=storage to calculate your costs based on VM size
$AzurePageBlobCostPerGB = "0.05"
$AzureBlockBlobCostperGB = "0.02"
$ZertoJournalSpacePercentage = "1.07" # Averages between 7-10%
########################################################################################################################
# Nothing to configure below this line - Starting the main function of the script
########################################################################################################################
################################################
# Prompts for your vCenter information
################################################
$vCenterServer = Read-Host -Prompt "Enter vCenter IP/Hostname"
$Credentials = Get-Credential -Credential $null
$Username = $Credentials.UserName
$Credentials.Password | ConvertFrom-SecureString
$Password = $Credentials.GetNetworkCredential().password
################################################
# Importing CSV
################################################
$CloudVMSizes = import-csv $AzureVMSizingImportCSV
# Creating arrays to collate data for csv export
$AzureVMArray = @()
$VMSizeSummary = @()
#######################
# Installing then importing PowerCLI module
#######################
$PowerCLIModuleCheck = Get-Module -ListAvailable VMware.PowerCLI
IF ($PowerCLIModuleCheck -eq $null)
{
Install-Module -Name VMware.PowerCLI -Scope CurrentUser -Confirm:$false -AllowClobber
}
# Importing PowerCLI
Import-Module VMware.PowerCLI
#######################
# Connecting to vCenter
#######################
Try 
{
Connect-VIServer -Server $vCenterServer -User $Username -Password $Password 
$vCenterAuthentication = "PASS"
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
$vCenterAuthentication = "FAIL"
}
#######################
# Getting list of VMs, filtering by power state On and excluding some VMs as example filters
#######################
Try 
{
$VMList = Get-VM | Where {($_.PowerState -eq "PoweredOn") -and ($_.Name -notlike "*Z-VRA*") -and ($_.Name -notlike "*ZVM*") -and ($_.Name -notlike "*NestedESXi*") -and ($_.Name -notlike "*vCenter*")}
}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
################################################
# Performing VM size selection per VM
################################################
ForEach ($VM in $VMList)
{
# Getting Guest OS for pricing
IF (($VM.GuestID -like "*windows*") -or ($VM.Guest -like "*windows*"))
{
$VMGuestOS = "Windows"
}
ELSE
{
$VMGuestOS = "Linux"
}
# Setting VM Info
$VMName = $VM.Name
$VMNumCPU = $VM.NumCPU
$VMMemoryGB = $VM.MemoryGB
$VMSizeGB = $VM.ProvisionedSpaceGB
$VMSizeGB = [Math]::Round($VMSizeGB, 2)
# Outputting to host
"Collecting VM Sizing Data For: $VMName"
# Getting VM disk and NIC counts
Try 
{
$VMDisks = (Get-HardDisk -VM $VMName).Count}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
Try 
{
$VMNICs = (Get-NetworkAdapter -VM $VMName).Count}
Catch 
{
$_.Exception.ToString()
$error[0] | Format-List -Force
}
#######################
# Azure sizing actions
#######################
# Getting Azure VM size table
$AzureVMSizeTable = $CloudVMSizes | Where-Object {($_.CPU -as [int] -ge $VMNumCPU) -and ($_.RAM  -as [int] -ge $VMMemoryGB) -and ($_.Disks  -as [int] -ge $VMDisks) -and ($_.NICs  -as [int] -ge $VMNICs)}
# Getting Azure VM size closest to VM
$AzureVMSizeInfo = $AzureVMSizeTable | Sort-Object {[int]$_.CPU},{[int]$_.RAM} | Select -First 1
# Setting Azure VM size info
$AzureVMSize = $AzureVMSizeInfo.Size
$AzureVMCPU = $AzureVMSizeInfo.CPU
$AzureVMRAM = $AzureVMSizeInfo.RAM
$AzureVMDisks = $AzureVMSizeInfo.Disks
$AzureVMNICs = $AzureVMSizeInfo.NICs
$AzureVMLocalHDD = $AzureVMSizeInfo.LocalHDD
# Setting VM price depending on OS
if ($VMGuestOS -eq "Windows")
{
$AzureVMPricePH = $AzureVMSizeInfo.WindowsPricePH
}
else
{
$AzureVMPricePH = $AzureVMSizeInfo.LinuxPricePH
}
# Calculating disk sizes
$AzureVMJournalSize = $VMSizeGB * $ZertoJournalSpacePercentage - $VMSizeGB
$AzureVMReplicaSizeGB = $VMSizeGB * $ZertoJournalSpacePercentage
$AzureVMReplicaSizeGB = [Math]::Round($AzureVMReplicaSizeGB, 2)
# Calculating replica data cost stored as page blob disks:
$AzureVMReplicaDataPricePM = $AzureVMReplicaSizeGB * $AzurePageBlobCostPerGB
$AzureVMReplicaDataPricePM = [Math]::Round($AzureVMReplicaDataPricePM, 2)
# Calculating replica data cost stored as block blobs:
$AzureVMJournalDataPricePM = $AzureVMJournalSize * $AzureBlockBlobCostperGB
$AzureVMJournalDataPricePM = [Math]::Round($AzureVMJournalDataPricePM, 2)
# Creating total
$AzureVMTotalReplicaDataPricePM = $AzureVMReplicaDataPricePM + $AzureVMJournalDataPricePM
$AzureVMTotalReplicaDataPricePM = [Math]::Round($AzureVMTotalReplicaDataPricePM, 2)
#######################
# Adding chosen VM size info to array
#######################
$AzureVMArrayLine = New-Object PSObject
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMName" -Value "$VMName"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMGuestOS" -Value "$VMGuestOS"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMNumCPU" -Value "$VMNumCPU"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMMemoryGB" -Value "$VMNumCPU"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMSizeGB" -Value "$VMSizeGB"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMDisks" -Value "$VMDisks"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "VMNIcs" -Value "$VMNIcs"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMSize" -Value "$AzureVMSize"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMCPU" -Value "$AzureVMCPU"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMRAM" -Value "$AzureVMRAM"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMDisks" -Value "$AzureVMDisks"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMNICs" -Value "$AzureVMNICs"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMLocalHDD" -Value "$AzureVMLocalHDD"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMJournalSize" -Value "$AzureVMJournalSize"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMPPH" -Value "$AzureVMPricePH"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMReplicaDataPPM" -Value "$AzureVMReplicaDataPricePM"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMJournalDataPPM" -Value "$AzureVMJournalDataPricePM"
$AzureVMArrayLine | Add-Member -MemberType NoteProperty -Name "AzureVMTotalReplicaDataPPM" -Value "$AzureVMTotalReplicaDataPricePM"
$AzureVMArray += $AzureVMArrayLine
# End of per VM actions below
}
################################################
# End of per VM actions - Finishing script
################################################
#######################
# Creating Summary array for Azure
#######################
# Counting VMs processed & most common size
$VMCount = $VMList.Count
$MostCommonVMSize = $AzureVMArray.AzureVMSize | Sort | Group | Sort Count | Select -ExpandProperty Name -last 1
# Getting ZCA price (Zerto management VM in Azure, 1 per 200 VMs)
$AzureZCAPriceList = $CloudVMSizes | Where-Object {($_.Size -eq "Standard_DS3_V2")} | Sort-Object {[int]$_.WindowsPricePH} | select WindowsPricePH
$AzureZCAPricePH = $AzureZCAPriceList.WindowsPricePH | Measure-Object -Sum | select -ExpandProperty Sum
$AzureZCAPricePY = $AzureZCAPricePH * 8760 # Total hours per year divided into 12 months to get the average
$AzureZCAPricePM = $AzureZCAPricePY / 12 
$AzureZCACount = $VMCount / $VMsPerZCA # VMs per ZCA
$AzureZCACount = [int][Math]::Ceiling($AzureZCACount)
$AzureZCACostPM = $AzureZCAPricePM * $AzureZCACount
$AzureZCACostPY = $AzureZCACostPM * 12
# Calculating totals for VM and journal data
$TotalVMData = $AzureVMArray.VMSizeGB | Measure-Object -Sum | select -ExpandProperty Sum
$TotalVMData = [Math]::Round($TotalVMData, 2)
$TotalJournalData = $AzureVMArray.AzureVMJournalSize | Measure-Object -Sum | select -ExpandProperty Sum
$TotalJournalData = [Math]::Round($TotalJournalData, 2)
$TotalAzureDataPPM = $AzureVMArray.AzureVMTotalReplicaDataPPM | Measure-Object -Sum | select -ExpandProperty Sum
$TotalAzureDataPPY = $TotalAzureDataPPM * 12
$TotalAzureReplicaDataPPM = $AzureVMArray.AzureVMJournalDataPPM | Measure-Object -Sum | select -ExpandProperty Sum
$TotalAzureReplicaDataPPY = $TotalAzureReplicaDataPPM * 12
$TotalAzureJournalPPM = $AzureVMArray.AzureVMJournalDataPPM | Measure-Object -Sum | select -ExpandProperty Sum
# Calculating total cost of run time per VM when running failover, test or move, not including the ZCA
$TotalAzureVMPPH = $AzureVMArray.AzureVMPPH | Measure-Object -Sum | select -ExpandProperty Sum
$TotalAzureVMPPD = $TotalAzureVMPPH * 24
$TotalAzureVMPPY = $TotalAzureVMPPD * 365
$TotalAzureVMPPM = $TotalAzureVMPPY /12
# Calculating total cost per year assuming 20 days of continuous testing
$FailoverRunningPrice20Days = $FailoverRunningPricePerDay * 20
$TotalAzureCostPY = $AzureZCACostPY + $TotalAzureDataPPY + $FailoverRunningPrice20Days
# Adding line to array, PPD = Price Per Day etc
$VMSizeSummaryLine = New-Object PSObject
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "TotalVMs" -Value $VMCount
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "TotalVMDataGB" -Value $TotalVMData
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "TotalJournalDataGB" -Value $TotalJournalData
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "ReplicaDataPPPM" -Value $TotalAzureDataPPM
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "ReplicaDataPPPY" -Value $TotalAzureDataPPY
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "AzureZCACount" -Value $AzureZCACount
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "AzureZCACostPM" -Value $AzureZCACostPM
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "AzureZCACostPY" -Value $AzureZCACostPY
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "MostCommonVMSize" -Value $MostCommonVMSize
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "FailoverRunningPPH" -Value $TotalAzureVMPPH
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "FailoverRunningPPD" -Value $TotalAzureVMPPD
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "FailoverRunningPPM" -Value $TotalAzureVMPPM
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "FailoverRunningPPY" -Value $TotalAzureVMPPY
$VMSizeSummaryLine | Add-Member -MemberType NoteProperty -Name "TotalAzureCostPY" -Value $TotalAzureCostPY
$VMSizeSummary += $VMSizeSummaryLine
#######################
# Outputting VM sizing CSVs
#######################
$AzureVMArray | Sort-Object VMName | export-csv $AzureVMArrayCSV -Force -NoClobber -NoTypeInformation
$VMSizeSummary | Sort-Object Cloud | export-csv -path $AzureVMSummaryCSV -Force -NoClobber -NoTypeInformation
#######################
# Disconnecting from the vCenter
#######################
Disconnect-VIServer * -Confirm:$false